/*
 * pdsawgn.c
 * 
 * Copyright 2011 Fernando Pujaico Rivera <fernando.pujaico.rivera@gmail.com>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 * 
 */


#include <pds/pdsawgn.h>
#include <pds/pdsgaussian.h>
#include <stdio.h>
#include <stdlib.h>


////////////////////////////////////////////////////////////////////////////////
////  Trabajando con PdsAwgn                                                ////
////////////////////////////////////////////////////////////////////////////////


/** \fn PdsAwgn *pds_awgn_new(PdsCmReal Sigma)
 *  \brief Crea un canal de tipo PdsAwgn.
 *  \param[in] Sigma Es el desvío padrón.
 *  \return Un puntero al canal de tipo PdsAwgn, o NULL si hubo un error.
 *  \ingroup PdsAwgnGroup
 */
PdsAwgn *pds_awgn_new(PdsCmReal Sigma)
{
	PdsAwgn *Channel=NULL;

	Channel=(PdsAwgn *)calloc(1,sizeof(PdsAwgn));
	if(Channel==NULL)	return NULL;

	Channel->Z=pds_gaussian_new(0,(PdsRvReal)Sigma);
	if(Channel->Z==NULL)
	{
		free(Channel);
		return NULL;
	}

	return Channel;
}


/** \fn int pds_awgn_evaluate(PdsAwgn *Channel,const PdsVector *VectorInput,PdsVector *VectorOutput)
 *  \brief Evalua un vector en un canal AWGN.
 *  \param[in,out] Channel Canal AWGN.
 *  \param[in] VectorInput Es el vector a la entrada del canal AWGN.
 *  \param[out] VectorOutput Es el vector a la salida del canal AWGN.
 *  \return TRUE si todo fue bien o FALSE si nó, por ejemplo un vector nulo.
 *  \ingroup PdsAwgnGroup
 */
int pds_awgn_evaluate(PdsAwgn *Channel,const PdsVector *VectorInput,PdsVector *VectorOutput)
{
	PdsCmNatural i;
	PdsRvReal x;

	if(VectorInput==NULL)	return FALSE;
	if(VectorOutput==NULL)	return FALSE;
	if(Channel==NULL)	return FALSE;
	if(VectorInput->Nel!=VectorOutput->Nel)	return FALSE;

	for(i=0;i<VectorInput->Nel;i++)
	{
		pds_gaussian_get_value(Channel->Z,&x);
		VectorOutput->V[i]=VectorInput->V[i]+x;
	}

	return TRUE;
}


/** \fn void pds_awgn_free(PdsAwgn *Channel)
 *  \brief Libera un canal de tipo puntero PdsAwgn.
 *  \param[in,out] Channel El canal a liberar.
 *  \return No retorna valor.
 *  \ingroup PdsAwgnGroup
 */
void pds_awgn_free(PdsAwgn *Channel)
{
	if(Channel!=NULL)
	{
		pds_gaussian_free(Channel->Z);
		free(Channel);
	}
}


/** \fn void pds_awgn_destroy(PdsAwgn **Channel)
 *  \brief Libera un canal de tipo puntero PdsAwgn y lo carga con NULL.
 *  \param[in,out] Channel El canal a liberar.
 *  \return No retorna valor.
 *  \ingroup PdsAwgnGroup
 */
void pds_awgn_destroy(PdsAwgn **Channel)
{
	if((*Channel)!=NULL)
	{
		pds_gaussian_free((*Channel)->Z);
		free(*Channel);
		(*Channel)=NULL;
	}
}


