/*
 * pdsbiawgn.c
 * 
 * Copyright 2011 Fernando Pujaico Rivera <fernando.pujaico.rivera@gmail.com>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 * 
 */


#include <pds/pdsbiawgn.h>
#include <pds/pdsgaussian.h>
#include <stdio.h>
#include <stdlib.h>


////////////////////////////////////////////////////////////////////////////////
////  Trabajando con PdsBiAwgn                                                ////
////////////////////////////////////////////////////////////////////////////////


/** \fn PdsBiAwgn *pds_biawgn_new(PdsCmReal A, PdsCmReal Sigma)
 *  \brief Crea un canal de tipo PdsBiAwgn.
 *  \param[in] A Amplitud de la senhal de entrada.
 *  \param[in] Sigma Es el desvío padrón.
 *  \return Un puntero al canal de tipo PdsBiAwgn, o NULL si hubo un error.
 *  \ingroup PdsBiAwgnGroup
 */
PdsBiAwgn *pds_biawgn_new(PdsCmReal A, PdsCmReal Sigma)
{
	PdsBiAwgn *Channel=NULL;

	Channel=(PdsBiAwgn *)calloc(1,sizeof(PdsBiAwgn));
	if(Channel==NULL)	return NULL;

	Channel->A=A;
	
	Channel->Z=pds_gaussian_new(0,(PdsRvReal)Sigma);
	if(Channel->Z==NULL)
	{
		free(Channel);
		return NULL;
	}

	return Channel;
}


/** \fn int pds_biawgn_evaluate(PdsBiAwgn *Channel,const PdsBVector *BVectorInput,PdsVector *VectorOutput)
 *  \brief Evalua un vector binario en un canal BIAWGN.
 *  \param[in,out] Channel Canal BIAWGN.
 *  \param[in] BVectorInput Es el vector binario a la entrada del canal BIAWGN.
 *  \param[out] VectorOutput Es el vector real a la salida del canal AWGN.
 *  \return TRUE si todo fue bien o FALSE si nó, por ejemplo un vector nulo.
 *  \ingroup PdsBiAwgnGroup
 */
int pds_biawgn_evaluate(PdsBiAwgn *Channel,const PdsBVector *BVectorInput,PdsVector *VectorOutput)
{
	PdsCmNatural i;
	PdsCmReal X;
	PdsBaBit Bit;
	PdsRvReal x;

	if(BVectorInput==NULL)	return FALSE;
	if(VectorOutput==NULL)	return FALSE;
	if(Channel==NULL)	return FALSE;
	if(BVectorInput->Nel!=VectorOutput->Nel)	return FALSE;

	for(i=0;i<VectorOutput->Nel;i++)
	{
		Bit=(BVectorInput->V[i/8]&(1<<(i%8)))>>(i%8);
		X=Channel->A*(2.0*Bit-1.0);
		pds_gaussian_get_value(Channel->Z,&x);
		VectorOutput->V[i]=X+x;
	}
	return TRUE;
}


/** \fn void pds_biawgn_free(PdsBiAwgn *Channel)
 *  \brief Libera un canal de tipo puntero PdsBiAwgn.
 *  \param[in,out] Channel El canal a liberar.
 *  \return No retorna valor.
 *  \ingroup PdsBiAwgnGroup
 */
void pds_biawgn_free(PdsBiAwgn *Channel)
{
	if(Channel!=NULL)
	{
		pds_gaussian_free(Channel->Z);
		free(Channel);
	}
}


/** \fn void pds_biawgn_destroy(PdsBiAwgn **Channel)
 *  \brief Libera un canal de tipo puntero PdsBiAwgn y lo carga con NULL.
 *  \param[in,out] Channel El canal a liberar.
 *  \return No retorna valor.
 *  \ingroup PdsBiAwgnGroup
 */
void pds_biawgn_destroy(PdsBiAwgn **Channel)
{
	if((*Channel)!=NULL)
	{
		pds_gaussian_free((*Channel)->Z);
		free(*Channel);
		(*Channel)=NULL;
	}
}


