/*
 * pdsbsc.c
 * 
 * Copyright 2011 Fernando Pujaico Rivera <fernando.pujaico.rivera@gmail.com>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 * 
 */

#include <pds/pdsbsc.h>
#include <pds/pdsuniform.h>
#include <stdio.h>
#include <stdlib.h>


////////////////////////////////////////////////////////////////////////////////
////  Trabajando con PdsBsc                                                ////
////////////////////////////////////////////////////////////////////////////////


/** \fn PdsBsc *pds_bsc_new(PdsCmReal p)
 *  \brief Crea un canal de tipo PdsBsc.
 *  \param[in] p La probabilidad de error del canal.
 *  \return Un puntero al canal de tipo PdsBsc, o NULL si hubo un error.
 *  \ingroup PdsBscGroup
 */
PdsBsc *pds_bsc_new(PdsCmReal p)
{
	PdsBsc *Channel=NULL;

	if((p<0.0)||(p>1.0))	return NULL;

	Channel=(PdsBsc *)calloc(1,sizeof(PdsBsc));
	if(Channel==NULL)	return NULL;

	Channel->p=p;
	
	Channel->Z=pds_uniform_new(0,1.0);
	if(Channel->Z==NULL)
	{
		free(Channel);
		return NULL;
	}

	return Channel;
}


/** \fn int pds_bsc_evaluate(PdsBsc *Channel,const PdsBVector *BVectorInput,PdsBVector *BVectorOutput)
 *  \brief Evalua un vector binario en un canal BSC.
 *  \param[in,out] Channel Canal BSC.
 *  \param[in] BVectorInput Es el vector binario a la entrada del canal BSC.
 *  \param[out] BVectorOutput Es el vector binario a la salida del canal BSC.
 *  \return TRUE si todo fue bien o FALSE si nó, por ejemplo un vector nulo.
 *  \ingroup PdsBscGroup
 */
int pds_bsc_evaluate(PdsBsc *Channel,const PdsBVector *BVectorInput,PdsBVector *BVectorOutput)
{
	PdsCmNatural i,j,k;
	PdsBaByte Byte;
	PdsRvReal x;

	if(BVectorInput==NULL)	return FALSE;
	if(BVectorOutput==NULL)	return FALSE;
	if(Channel==NULL)	return FALSE;
	if(BVectorInput->Nel!=BVectorOutput->Nel)	return FALSE;


	for(i=0,k=0;i<BVectorInput->Nbytes;i++)
	{
		Byte=BVectorInput->V[i];
		for(j=0;(j<8)&&(k<BVectorInput->Nel);j++,k++)
		{
			pds_uniform_get_value(Channel->Z,&x);
			if(((PdsBaReal)x)<=Channel->p)
			{
				Byte=Byte^(1<<j);
			}
			
		}
		BVectorOutput->V[i]=Byte;
	}

	return TRUE;
}


/** \fn int pds_bsc_evaluate_val(PdsBsc *Channel,PdsBaBit u0,PdsBaBit *u)
 *  \brief Evalua un bit en un canal BSC.
 *  \param[in,out] Channel Canal BSC.
 *  \param[in] u0 Es el bit en la entrada del canal BSC.
 *  \param[out] u Es el bit en la salida del canal BSC.
 *  \return TRUE si todo fue bien o FALSE si nó, por ejemplo un vector nulo.
 *  \ingroup PdsBscGroup
 */
int pds_bsc_evaluate_val(PdsBsc *Channel,PdsBaBit u0,PdsBaBit *u)
{
	PdsRvReal x;

	if(u==NULL)	        return FALSE;
	if(Channel==NULL)	return FALSE;

    *u=u0;

	pds_uniform_get_value(Channel->Z,&x);
	if( ((PdsBaReal)x)<=Channel->p )
	{
	    (*u)=u0^1;
	}


	return TRUE;
}

/** \fn void pds_bsc_free(PdsBsc *Channel)
 *  \brief Libera un canal de tipo puntero PdsBsc.
 *  \param[in,out] Channel El canal a liberar.
 *  \return No retorna valor.
 *  \ingroup PdsBscGroup
 */
void pds_bsc_free(PdsBsc *Channel)
{
	if(Channel!=NULL)
	{
		pds_uniform_free(Channel->Z);
		free(Channel);
	}
}


/** \fn void pds_bsc_destroy(PdsBsc **Channel)
 *  \brief Libera un canal de tipo puntero PdsBsc y lo carga con NULL.
 *  \param[in,out] Channel El canal a liberar.
 *  \return No retorna valor.
 *  \ingroup PdsBscGroup
 */
void pds_bsc_destroy(PdsBsc **Channel)
{
	if((*Channel)!=NULL)
	{
		pds_uniform_free((*Channel)->Z);
		free(*Channel);
		(*Channel)=NULL;
	}
}


