/*
 * pdsbscblock.c
 * 
 * Copyright 2011 Fernando Pujaico Rivera <fernando.pujaico.rivera@gmail.com>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 * 
 */

#include <pds/pdsbscblock.h>
#include <pds/pdsbsc.h>
#include <pds/pdsuniform.h>
#include <stdio.h>
#include <stdlib.h>


////////////////////////////////////////////////////////////////////////////////
////  Trabajando con PdsBscBlock                                            ////
////////////////////////////////////////////////////////////////////////////////


/** \fn PdsBscBlock *pds_bscblock_new(PdsVector *P)
 *  \brief Crea un bloque de canales de tipo PdsBscBlock.
 *  \param[in] P Las probabilidades de error de los canales. Cualquier valor 
 *  mayor que 1.0 y menor que 0.0 retornan un vector NULL.
 *  \return Un puntero al canal de tipo PdsBscBlock, o NULL si hubo un error.
 *  \ingroup PdsBscBlockGroup
 */
PdsBscBlock *pds_bscblock_new(PdsVector *P)
{
	PdsBscBlock *BLOCK=NULL;
    PdsCmNatural i,j;
    if(P==NULL) return NULL;

    for(i=0;i<P->Nel;i++)
    {
        if( (P->V[i]>1.0)||(P->V[i]<0.0) ) return NULL;
    }

    BLOCK=(PdsBscBlock *)calloc(1,sizeof(PdsBscBlock));
    if(BLOCK==NULL) return NULL;

    BLOCK->Nel=P->Nel;

    BLOCK->BSC=(PdsBsc **)calloc(BLOCK->Nel,sizeof(PdsBsc *));

    for(i=0;i<BLOCK->Nel;i++)
    {
        BLOCK->BSC[i]=pds_bsc_new(P->V[i]);
        if(BLOCK->BSC[i]==NULL)
        {
            for(j=0;j<i;j++) pds_bsc_free(BLOCK->BSC[j]);
            free(BLOCK->BSC);
            return NULL;
        }
    }
    
    
	return BLOCK;
}

/** \fn PdsBscBlock *pds_bscblock_new_symmetric(PdsCmNatural M,PdsCmReal rho)
 *  \brief Crea un bloque de canales de tipo PdsBscBlock.
 *  \param[in] M es el número de canales BSC en el bloque.
 *  \param[in] rho Las probabilidades de error de todos los canales. Cualquier valor 
 *  mayor que 1.0 y menor que 0.0 retornan un vector NULL.
 *  \return Un puntero al canal de tipo PdsBscBlock, o NULL si hubo un error.
 *  \ingroup PdsBscBlockGroup
 */
PdsBscBlock *pds_bscblock_new_symmetric(PdsCmNatural M,PdsCmReal rho)
{
	PdsBscBlock *BLOCK=NULL;
    PdsCmNatural i,j;

    if(M<1) return NULL;
    if( (rho>1.0)||(rho<0.0) ) return NULL;

    BLOCK=(PdsBscBlock *)calloc(1,sizeof(PdsBscBlock));
    if(BLOCK==NULL) return NULL;

    BLOCK->Nel=M;

    BLOCK->BSC=(PdsBsc **)calloc(BLOCK->Nel,sizeof(PdsBsc *));

    for(i=0;i<BLOCK->Nel;i++)
    {
        BLOCK->BSC[i]=pds_bsc_new(rho);
        if(BLOCK->BSC[i]==NULL)
        {
            for(j=0;j<i;j++) pds_bsc_free(BLOCK->BSC[j]);
            free(BLOCK->BSC);
            return NULL;
        }
    }
    
    
	return BLOCK;
}


/** \fn int pds_bscblock_evaluate(PdsBscBlock *BLOCK,const PdsBVector *BVectorInput,PdsBVector *BVectorOutput)
 *  \brief Evalua un bit u0 en el bloque de canales BSC.
 *  \param[in,out] BLOCK Bloque de canales BSC.
 *  \param[in] u0 Es el bit a evaluar
 *  \param[out] BVectorOutput Es el vector binario a la salida de los canales BSC.
 *  \return TRUE si todo fue bien o FALSE si nó, por ejemplo un vector nulo.
 *  \ingroup PdsBscBlockGroup
 */
int pds_bscblock_evaluate_val(PdsBscBlock *BLOCK,PdsBaBit u0,PdsBVector *BVectorOutput)
{
	PdsCmNatural i;
    int dat;
    PdsBaBit u;

	if(BVectorOutput==NULL)	return FALSE;
	if(BLOCK==NULL)	return FALSE;
	if(BLOCK->Nel!=BVectorOutput->Nel)	return FALSE;

    for(i=0;i<BLOCK->Nel;i++)
    {
        dat=pds_bsc_evaluate_val(BLOCK->BSC[i],u0,&u);
        if(dat==FALSE)  return FALSE;
        dat=pds_bvector_set_bit(BVectorOutput,i,u);
        if(dat==FALSE)  return FALSE;
    }

	return TRUE;
}


/** \fn void pds_bscblock_free(PdsBscBlock *BLOCK)
 *  \brief Libera un canal de tipo puntero PdsBscBlock.
 *  \param[in,out] BLOCK El bloque de canales a liberar.
 *  \return No retorna valor.
 *  \ingroup PdsBscBlockGroup
 */
void pds_bscblock_free(PdsBscBlock *BLOCK)
{
    PdsCmNatural i;
	if(BLOCK!=NULL)
	{
        for(i=0;i<BLOCK->Nel;i++)   pds_bsc_free(BLOCK->BSC[i]);
		free(BLOCK);
	}
}


/** \fn void pds_bscblock_destroy(PdsBscBlock **BLOCK)
 *  \brief Libera un bloque de canales de tipo puntero PdsBscBlock y lo carga con NULL.
 *  \param[in,out] BLOCK El bloque de canales a liberar.
 *  \return No retorna valor.
 *  \ingroup PdsBscBlockGroup
 */
void pds_bscblock_destroy(PdsBscBlock **BLOCK)
{
    PdsCmNatural i;
	if((*BLOCK)!=NULL)
	{
        for(i=0;i<(*BLOCK)->Nel;i++)   pds_bsc_free((*BLOCK)->BSC[i]);
		free(*BLOCK);
        (*BLOCK)=NULL;
	}
}


