/*
 * pdsawgn.h
 * 
 * Copyright 2011 Fernando Pujaico Rivera <fernando.pujaico.rivera@gmail.com>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 * 
 */

/** \file pdsawgn.h
 *  \author Fernando Pujaico Rivera
 *  \date 01-05-2011
 *  \brief Funciones que trabajan con un modelos de canal AWGN.
 *  
 *  Estas funciones trabajan con un canal AWGN así:.<br>
 *  \image html image.png "Canal AWGN"
 *  \b X es la entrada real del canal, \b Z es una variable aleatoria Gaussiana.
 *  de media cero y varianza Sigma^2 e \b Y es la salida real del canal.
 */

#ifndef __PDSAWGN_H__
#define __PDSAWGN_H__

#include <stdio.h>

#ifdef __cplusplus
extern "C" {
#endif 

#include <pds/pdscmglobal.h>
#include <pds/pdsgaussian.h>
#include <pds/pdsvector.h>

#ifndef TRUE
	#define TRUE 1
#endif

#ifndef FALSE
	#define FALSE 0
#endif

/** \defgroup PdsAwgnGroup Funciones del módulo PdsAwgn.
 * @{
 */


/*! \struct PdsAwgn
 *  \brief La estructura tipo  PdsAwgn .
 *  Esta estructura genera una canal AWGN.
 *  Para usar incluir pds/pdsawgn.h.
 *  \ingroup PdsAwgnGroup
 *  \author Fernando Pujaico Rivera
 */
typedef struct 
{
	/*! Una variable aleatoria Gaussiana. */
	PdsGaussian *Z;
}PdsAwgn;


/** \fn PdsAwgn *pds_awgn_new(PdsCmReal Sigma)
 *  \brief Crea un canal de tipo PdsAwgn.
 *  \param[in] Sigma Es el desvío padrón.
 *  \return Un puntero al canal de tipo PdsAwgn.
 *  \ingroup PdsAwgnGroup
 */
PdsAwgn *pds_awgn_new(PdsCmReal Sigma);


/** \fn int pds_awgn_evaluate(PdsAwgn *Channel,const PdsVector *VectorInput,PdsVector *VectorOutput)
 *  \brief Evalua un vector en un canal AWGN.
 *  \param[in,out] Channel Canal AWGN.
 *  \param[in] VectorInput Es el vector a la entrada del canal AWGN.
 *  \param[out] VectorOutput Es el vector a la salida del canal AWGN.
 *  \return TRUE si todo fue bien o FALSE si nó, por ejemplo un vector nulo.
 *  \ingroup PdsAwgnGroup
 */
int pds_awgn_evaluate(PdsAwgn *Channel,const PdsVector *VectorInput,PdsVector *VectorOutput);


/** \fn void pds_awgn_free(PdsAwgn *Channel)
 *  \brief Libera un canal de tipo puntero PdsAwgn.
 *  \param[in,out] Channel El canal a liberar.
 *  \return No retorna valor.
 *  \ingroup PdsAwgnGroup
 */
void pds_awgn_free(PdsAwgn *Channel);


/** \fn void pds_awgn_destroy(PdsAwgn **Channel)
 *  \brief Libera un canal de tipo puntero PdsAwgn y lo carga con NULL.
 *  \param[in,out] Channel El canal a liberar.
 *  \return No retorna valor.
 *  \ingroup PdsAwgnGroup
 */
void pds_awgn_destroy(PdsAwgn **Channel);


/**
 * @}
 */

#ifdef __cplusplus
}
#endif 

#endif
/* FPRNEURON_H */ 

