/*
 * pdsbiawgn.h
 * 
 * Copyright 2011 Fernando Pujaico Rivera <fernando.pujaico.rivera@gmail.com>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 * 
 */

/** \file pdsbiawgn.h
 *  \author Fernando Pujaico Rivera
 *  \date 01-05-2011
 *  \brief Funciones que trabajan con un modelos de canal AWGN.
 *  
 *  Estas funciones trabajan con un canal AWGN así:.<br>
 *  \image html image.png "Canal AWGN"
 *  \b X es la entrada real del canal, \b Z es una variable aleatoria Gaussiana.
 *  de media cero y varianza Sigma^2 e \b Y es la salida real del canal.
 */

#ifndef __PDSBIAWGN_H__
#define __PDSBIAWGN_H__

#include <stdio.h>

#ifdef __cplusplus
extern "C" {
#endif 

#include <pds/pdscmglobal.h>
#include <pds/pdsgaussian.h>
#include <pds/pdsvector.h>
#include <pds/pdsbvector.h>

#ifndef TRUE
	#define TRUE 1
#endif

#ifndef FALSE
	#define FALSE 0
#endif

/** \defgroup PdsBiAwgnGroup Funciones del módulo PdsBiAwgn.
 * @{
 */


/*! \struct PdsBiAwgn
 *  \brief La estructura tipo  PdsBiAwgn .
 *  Esta estructura genera una canal AWGN.
 *  Para usar incluir pds/pdsbiawgn.h.
 *  \ingroup PdsBiAwgnGroup
 *  \author Fernando Pujaico Rivera
 */
typedef struct 
{
	/*! Una variable aleatoria Gaussiana. */
	PdsGaussian *Z;
	PdsCmReal A;
}PdsBiAwgn;


/** \fn PdsBiAwgn *pds_biawgn_new(PdsCmReal A, PdsCmReal Sigma)
 *  \brief Crea un canal de tipo PdsBiAwgn.
 *  \param[in] A Amplitud de la senhal de entrada.
 *  \param[in] Sigma Es el desvío padrón.
 *  \return Un puntero al canal de tipo PdsBiAwgn, o NULL si hubo un error.
 *  \ingroup PdsBiAwgnGroup
 */
PdsBiAwgn *pds_biawgn_new(PdsCmReal A, PdsCmReal Sigma);


/** \fn int pds_biawgn_evaluate(PdsBiAwgn *Channel,const PdsBVector *BVectorInput,PdsVector *VectorOutput)
 *  \brief Evalua un vector binario en un canal BIAWGN.
 *  \param[in,out] Channel Canal BIAWGN.
 *  \param[in] BVectorInput Es el vector binario a la entrada del canal BIAWGN.
 *  \param[out] VectorOutput Es el vector real a la salida del canal AWGN.
 *  \return TRUE si todo fue bien o FALSE si nó, por ejemplo un vector nulo.
 *  \ingroup PdsBiAwgnGroup
 */
int pds_biawgn_evaluate(PdsBiAwgn *Channel,const PdsBVector *BVectorInput,PdsVector *VectorOutput);


/** \fn void pds_biawgn_free(PdsBiAwgn *Channel)
 *  \brief Libera un canal de tipo puntero PdsBiAwgn.
 *  \param[in,out] Channel El canal a liberar.
 *  \return No retorna valor.
 *  \ingroup PdsBiAwgnGroup
 */
void pds_biawgn_free(PdsBiAwgn *Channel);


/** \fn void pds_biawgn_destroy(PdsBiAwgn **Channel)
 *  \brief Libera un canal de tipo puntero PdsBiAwgn y lo carga con NULL.
 *  \param[in,out] Channel El canal a liberar.
 *  \return No retorna valor.
 *  \ingroup PdsBiAwgnGroup
 */
void pds_biawgn_destroy(PdsBiAwgn **Channel);


/**
 * @}
 */

#ifdef __cplusplus
}
#endif 

#endif
/* FPRNEURON_H */ 

