/*
 * pdscomplex.c
 * 
 * Copyright 2011 Fernando Pujaico Rivera <fernando.pujaico.rivera@gmail.com>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 * 
 */

#include <math.h>
#include <pds/pdscomplex.h>


////////////////////////////////////////////////////////////////////////////////
////  Trabajando con PdsComplex                                             ////
////////////////////////////////////////////////////////////////////////////////


/** \fn PdsComplex pds_complex_rect(PdsCnReal a,PdsCnReal b)
 *  \brief Crea la variable complejas. <b>a+bi</b>.
 *  \param[in] a Parte real.
 *  \param[in] b Parte imaginaria.
 *  \return La variable compleja.
 *  \ingroup PdsComplexGroup
 */
PdsComplex pds_complex_rect(PdsCnReal a,PdsCnReal b)
{
	PdsComplex c;
	c.Real=a;
	c.Imag=b;
	return c;
}


/** \fn PdsComplex pds_complex_polar(PdsCnReal a,PdsCnReal b)
 *  \brief Evalua de forma polar. <b>a*e^(bi)</b>.
 *  \param[in] a Módulo de la variable compleja.
 *  \param[in] b Ángulo de la variable compleja.
 *  \return El valor complejo equivalente.
 *  \ingroup PdsComplexGroup
 */
PdsComplex pds_complex_polar(PdsCnReal a,PdsCnReal b)
{
	PdsComplex c;
	c.Real=a*cos(b);
	c.Imag=a*sin(b);
	return c;
}


/** \fn PdsComplex pds_complex_add_complex(PdsComplex a,PdsComplex b)
 *  \brief Suma dos variables complejas. <b>a+b</b>.
 *  \param[in] a Variable compleja.
 *  \param[in] b Variable compleja.
 *  \return La suma de ambas variables.
 *  \ingroup PdsComplexGroup
 */
PdsComplex pds_complex_add_complex(PdsComplex a,PdsComplex b)
{
	PdsComplex c;
	c.Real=a.Real+b.Real;
	c.Imag=a.Imag+b.Imag;
	return c;
}


/** \fn PdsComplex pds_complex_add_real(PdsComplex a,PdsCnReal b)
 *  \brief Suma una variable complejas y una real. <b>a+b</b>.
 *  \param[in] a Variable compleja.
 *  \param[in] b Variable real.
 *  \return La suma de ambas variables.
 *  \ingroup PdsComplexGroup
 */
PdsComplex pds_complex_add_real(PdsComplex a,PdsCnReal b)
{
	PdsComplex c;
	c.Real=a.Real+b;
	c.Imag=a.Imag;
	return c;
}


/** \fn PdsComplex pds_complex_add_imag(PdsComplex a,PdsCnReal b)
 *  \brief Suma una variable complejas y una imaginaria. <b>a+bi</b>.
 *  \param[in] a Variable compleja.
 *  \param[in] b Variable imaginaria.
 *  \return La suma de ambas variables.
 *  \ingroup PdsComplexGroup
 */
PdsComplex pds_complex_add_imag(PdsComplex a,PdsCnReal b)
{
	PdsComplex c;
	c.Real=a.Real;
	c.Imag=a.Imag+b;
	return c;
}


/** \fn PdsComplex pds_complex_sub_complex(PdsComplex a,PdsComplex b)
 *  \brief Resta dos variables complejas. <b>a-b</b>.
 *  \param[in] a Variable compleja.
 *  \param[in] b Variable compleja.
 *  \return La resta de ambas variables.
 *  \ingroup PdsComplexGroup
 */
PdsComplex pds_complex_sub_complex(PdsComplex a,PdsComplex b)
{
	PdsComplex c;
	c.Real=a.Real-b.Real;
	c.Imag=a.Imag-b.Imag;
	return c;
}


/** \fn PdsComplex pds_complex_sub_real(PdsComplex a,PdsCnReal b)
 *  \brief Resta una variable complejas y una real. <b>a-b</b>
 *  \param[in] a Variable compleja.
 *  \param[in] b Variable real.
 *  \return La resta de ambas variables.
 *  \ingroup PdsComplexGroup
 */
PdsComplex pds_complex_sub_real(PdsComplex a,PdsCnReal b)
{
	PdsComplex c;
	c.Real=a.Real-b;
	c.Imag=a.Imag;
	return c;
}


/** \fn PdsComplex pds_complex_sub_imag(PdsComplex a,PdsCnReal b)
 *  \brief Resta una variable complejas y una imaginaria. <b>a-bi</b>
 *  \param[in] a Variable compleja.
 *  \param[in] b Variable imaginaria.
 *  \return La resta de ambas variables.
 *  \ingroup PdsComplexGroup
 */
PdsComplex pds_complex_sub_imag(PdsComplex a,PdsCnReal b)
{
	PdsComplex c;
	c.Real=a.Real;
	c.Imag=a.Imag-b;
	return c;
}


/** \fn PdsComplex pds_complex_mul_complex(PdsComplex a,PdsComplex b)
 *  \brief Multiplica dos variables complejas. <b>a*b</b>.
 *  \param[in] a Variable compleja.
 *  \param[in] b Variable compleja.
 *  \return La multiplicación de ambas variables.
 *  \ingroup PdsComplexGroup
 */
PdsComplex pds_complex_mul_complex(PdsComplex a,PdsComplex b)
{
	PdsComplex c;
	c.Real=a.Real*b.Real-a.Imag*b.Imag;
	c.Imag=a.Imag*b.Real+a.Real*b.Imag;
	return c;
}


/** \fn PdsComplex pds_complex_mul_real(PdsComplex a,PdsCnReal b)
 *  \brief Multiplica una variable compleja y una real. <b>a*b</b>.
 *  \param[in] a Variable compleja.
 *  \param[in] b Variable real.
 *  \return La multiplicación de ambas variables.
 *  \ingroup PdsComplexGroup
 */
PdsComplex pds_complex_mul_real(PdsComplex a,PdsCnReal b)
{
	PdsComplex c;
	c.Real=a.Real*b;
	c.Imag=a.Imag*b;
	return c;
}


/** \fn PdsComplex pds_complex_mul_imag(PdsComplex a,PdsCnReal b)
 *  \brief Multiplica una variable compleja y una imaginaria. <b>a*bi</b>.
 *  \param[in] a Variable compleja.
 *  \param[in] b Variable imaginaria.
 *  \return La multiplicación de ambas variables.
 *  \ingroup PdsComplexGroup
 */
PdsComplex pds_complex_mul_imag(PdsComplex a,PdsCnReal b)
{
	PdsComplex c;
	c.Real=-a.Imag*b;
	c.Imag=a.Real*b;
	return c;
}


/** \fn PdsComplex pds_complex_div_complex(PdsComplex a,PdsComplex b)
 *  \brief Divide dos variables complejas. <b>a/b</b>
 *  \param[in] a Variable compleja.
 *  \param[in] b Variable compleja.
 *  \return La división de ambas variables.
 *  \ingroup PdsComplexGroup
 */
PdsComplex pds_complex_div_complex(PdsComplex a,PdsComplex b)
{
	PdsComplex c;
	PdsCnReal M;
	M=sqrt(b.Imag*b.Imag+b.Real*b.Real);
	c.Real=(a.Real*b.Real+a.Imag*b.Imag)/M;
	c.Imag=(a.Imag*b.Real-a.Real*b.Imag)/M;
	return c;
}


/** \fn PdsComplex pds_complex_div_real(PdsComplex a,PdsCnReal b)
 *  \brief Divide una variable compleja y una real. <b>a/b</b>.
 *  \param[in] a Variable compleja.
 *  \param[in] b Variable real.
 *  \return La división de ambas variables.
 *  \ingroup PdsComplexGroup
 */
PdsComplex pds_complex_div_real(PdsComplex a,PdsCnReal b)
{
	PdsComplex c;
	c.Real=a.Real/b;
	c.Imag=a.Imag/b;
	return c;
}


/** \fn PdsComplex pds_complex_div_imag(PdsComplex a,PdsCnReal b)
 *  \brief Divide una variable compleja y una imaginaria. <b>a/bi</b>.
 *  \param[in] a Variable compleja.
 *  \param[in] b Variable imaginaria.
 *  \return La división de ambas variables.
 *  \ingroup PdsComplexGroup
 */
PdsComplex pds_complex_div_imag(PdsComplex a,PdsCnReal b)
{
	PdsComplex c;
	c.Real=a.Imag/b;
	c.Imag=-a.Real/b;
	return c;
}


/** \fn PdsComplex pds_complex_cis(PdsCnReal b)
 *  \brief Evalua cis(b). <b>cos(b)+i sin(b)=cis(b)</b>.
 *  \param[in] b Variable real.
 *  \return El cis(b).
 *  \ingroup PdsComplexGroup
 */
PdsComplex pds_complex_cis(PdsCnReal b)
{
	PdsComplex c;
	c.Real=cos(b);
	c.Imag=sin(b);
	return c;
}


/** \fn PdsCnReal pds_complex_norm(PdsComplex a)
 *  \brief Evalua |a|. <b>|a|=sqrt(a.Real^2+a.Imag^2)</b>.
 *  \param[in] a Variable compleja.
 *  \return El |a|.
 *  \ingroup PdsComplexGroup
 */
PdsCnReal pds_complex_norm(PdsComplex a)
{
	PdsCnReal r;
	r=sqrt(a.Real*a.Real+a.Imag*a.Imag);
	return r;
}


/** \fn PdsCnReal pds_complex_norm2(PdsComplex a)
 *  \brief Evalua |a|^2. <b>|a|^2=(a.Real^2+a.Imag^2)</b>.
 *  \param[in] a Variable compleja.
 *  \return El |a|^2.
 *  \ingroup PdsComplexGroup
 */
PdsCnReal pds_complex_norm2(PdsComplex a)
{
	PdsCnReal r2;
	r2=(a.Real*a.Real+a.Imag*a.Imag);
	return r2;
}


