/*
 * pdsquaternion.h
 * 
 * Copyright 2011 Fernando Pujaico Rivera <fernando.pujaico.rivera@gmail.com>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 * 
 */

/** \file pdsquaternion.h
 *  \author Fernando Pujaico Rivera
 *  \date 01-05-2011
 * \brief Archivo de definición de variable compleja.
 *
 */
#ifndef __PDSQUATERNION_H__
#define __PDSQUATERNION_H__


#include <pds/pdscnglobal.h>

#ifdef __cplusplus
extern "C" {
#endif 

/** \defgroup PdsQuaternionGroup Funciones del módulo PdsQuaternion.
 * @{
 */


/*! \struct PdsQuaternion
 *  \brief La estructura tipo  PdsQuaternion .
 *
 *  Esta estructura genera un quaternio do tipo <b>q=W+Xi+Yj+Zk</b>.
 *  \f[q=W+Xi+Yj+Zk\equiv(W,X,Y,Z)\f]
 *  
 *  Para usar incluir pdsquaternion/pdsquaternion.h.
 *  \ingroup PdsQuaternionGroup
 *  \author Fernando Pujaico Rivera
 */
typedef struct 
{
	/*! Parte real. */
	PdsCnReal W;
	/*! Parte imaginaria X. */
	PdsCnReal X;
	/*! Parte imaginaria Y. */
	PdsCnReal Y;
	/*! Parte imaginaria Z. */
	PdsCnReal Z;
}PdsQuaternion;



/*! \def PDS_QRECT(A,B,C,D)
 *  \brief Es equivalente a <b>A+Bi+Ci+Di</b>.
 *  \ingroup PdsQuaternionGroup
*/
#define PDS_QRECT(A,B,C,D) pds_quaternion_rect(A,B,C,D)

/** \fn PdsQuaternion pds_quaternion_rect(PdsCnReal w,PdsCnReal x,PdsCnReal y,PdsCnReal z)
 *  \brief Crea la variable complejas. <b>q=w+xi+yi+zi</b>.
 *
 *  \f[ q=w+xi+yj+zk\f]
 *
 *  \param[in] w Parte real.
 *  \param[in] x Parte imaginaria vinculada a i.
 *  \param[in] y Parte imaginaria vinculada a j.
 *  \param[in] z Parte imaginaria vinculada a k.
 *  \return Un quaternio.
 *  \ingroup PdsQuaternionGroup
 */
PdsQuaternion pds_quaternion_rect(PdsCnReal w,PdsCnReal x,PdsCnReal y,PdsCnReal z);



/*! \def PDS_QPOLAR(A,B,C,D)
 *  \brief Es equivalente a <b>(A*e^{Bi+Cj+Dk})</b>.
 *  \ingroup PdsQuaternionGroup
*/
#define PDS_QPOLAR(A,B,C,D) pds_quaternion_polar(A,B,C,D)

/** \fn PdsQuaternion pds_quaternion_polar(PdsCnReal a,PdsCnReal b,PdsCnReal c,PdsCnReal d)
 *  \brief Evalua de forma polar. <b>ae^{bi+cj+dk}</b>.
 *  
 *  \f[\sqrt{b^2+c^2+d^2}=M\f]
 *  \f[a~e^{bi+cj+dk}=a \left(cos(M)+sin(M)\frac{bi+cj+dk}{M}\right)\f]
 *
 *  \param[in] a parametro 1 del quaternio.
 *  \param[in] b parametro 2 del quaternio.
 *  \param[in] c parametro 3 del quaternio.
 *  \param[in] d parametro 4 del quaternio.
 *  \return El valor complejo equivalente.
 *  \ingroup PdsQuaternionGroup
 */
PdsQuaternion pds_quaternion_polar(PdsCnReal a,PdsCnReal b,PdsCnReal c,PdsCnReal d);



/*! \def PDS_QADDQ(A,B)
 *  \brief Es equivalente a <b>(A+B)</b>.
 *  \ingroup PdsQuaternionGroup
*/
#define PDS_QADDQ(A,B) pds_quaternion_add_quaternion(A,B)

/** \fn PdsQuaternion pds_quaternion_add_quaternion(PdsQuaternion a,PdsQuaternion b)
 *  \brief Suma dos quaternios. <b>a+b</b>.
 *  \param[in] a Quaternio.
 *  \param[in] b Quaternio.
 *  \return La suma de ambas variables.
 *  \ingroup PdsQuaternionGroup
 */
PdsQuaternion pds_quaternion_add_quaternion(PdsQuaternion a,PdsQuaternion b);



/*! \def PDS_QADDR(A,B)
 *  \brief Es equivalente a <b>(A+B)</b>.
 *  \ingroup PdsQuaternionGroup
*/
#define PDS_QADDR(A,B) pds_quaternion_add_real(A,B)

/** \fn PdsQuaternion pds_quaternion_add_real(PdsQuaternion a,PdsCnReal b)
 *  \brief Suma un quaternio y una real. <b>a+b</b>.
 *  \param[in] a Quaternio.
 *  \param[in] b Variable real.
 *  \return La suma de ambas variables.
 *  \ingroup PdsQuaternionGroup
 */
PdsQuaternion pds_quaternion_add_real(PdsQuaternion a,PdsCnReal b);



/*! \def PDS_QSUBQ(A,B)
 *  \brief Es equivalente a <b>(A-B)</b>.
 *  \ingroup PdsQuaternionGroup
*/
#define PDS_QSUBQ(A,B) pds_quaternion_sub_quaternion(A,B)

/** \fn PdsQuaternion pds_quaternion_sub_quaternion(PdsQuaternion a,PdsQuaternion b)
 *  \brief Resta dos quaternios. <b>a-b</b>.
 *  \param[in] a Quaternio.
 *  \param[in] b Quaternio.
 *  \return La resta de ambas variables.
 *  \ingroup PdsQuaternionGroup
 */
PdsQuaternion pds_quaternion_sub_quaternion(PdsQuaternion a,PdsQuaternion b);



/*! \def PDS_QSUBR(A,B)
 *  \brief Es equivalente a <b>(A-B)</b>.
 *  \ingroup PdsQuaternionGroup
*/
#define PDS_QSUBR(A,B) pds_quaternion_sub_real(A,B)

/** \fn PdsQuaternion pds_quaternion_sub_real(PdsQuaternion a,PdsCnReal b)
 *  \brief Resta un quaternio y una real. <b>a-b</b>
 *  \param[in] a Quaternio.
 *  \param[in] b Variable real.
 *  \return La resta de ambas variables.
 *  \ingroup PdsQuaternionGroup
 */
PdsQuaternion pds_quaternion_sub_real(PdsQuaternion a,PdsCnReal b);


/*! \def PDS_QMULQ(A,B)
 *  \brief Es equivalente a <b>(A*B)</b>.
 *  \ingroup PdsQuaternionGroup
*/
#define PDS_QMULQ(A,B) pds_quaternion_mul_quaternion(A,B)

/** \fn PdsQuaternion pds_quaternion_mul_quaternion(PdsQuaternion a,PdsQuaternion b)
 *  \brief Multiplica dos quaternios. <b>a*b</b>.
 *  \param[in] a Quaternio.
 *  \param[in] b Quaternio.
 *  \return La multiplicación de ambas variables.
 *  \ingroup PdsQuaternionGroup
 */
PdsQuaternion pds_quaternion_mul_quaternion(PdsQuaternion a,PdsQuaternion b);



/*! \def PDS_QMULR(A,B)
 *  \brief Es equivalente a <b>(A*B)</b>.
 *  \ingroup PdsQuaternionGroup
*/
#define PDS_QMULR(A,B) pds_quaternion_mul_real(A,B)

/** \fn PdsQuaternion pds_quaternion_mul_real(PdsQuaternion a,PdsCnReal b)
 *  \brief Multiplica un quaternio y una real. <b>a*b</b>.
 *  \param[in] a Quaternio.
 *  \param[in] b Variable real.
 *  \return La multiplicación de ambas variables.
 *  \ingroup PdsQuaternionGroup
 */
PdsQuaternion pds_quaternion_mul_real(PdsQuaternion a,PdsCnReal b);


/*! \def PDS_QDIVQ(A,B)
 *  \brief Es equivalente a <b>(A/B)</b>.
 *  \ingroup PdsQuaternionGroup
*/
#define PDS_QDIVQ(A,B) pds_quaternion_div_quaternion(A,B)

/** \fn PdsQuaternion pds_quaternion_div_quaternion(PdsQuaternion a,PdsQuaternion b)
 *  \brief Divide dos quaternios. <b>a/b</b>
 *  \param[in] a Quaternio.
 *  \param[in] b Quaternio.
 *  \return La división de ambas variables.
 *  \ingroup PdsQuaternionGroup
 */
PdsQuaternion pds_quaternion_div_quaternion(PdsQuaternion a,PdsQuaternion b);



/*! \def PDS_QDIVR(A,B)
 *  \brief Es equivalente a <b>(A/B)</b>.
 *  \ingroup PdsQuaternionGroup
*/
#define PDS_QDIVR(A,B) pds_quaternion_div_real(A,B)

/** \fn PdsQuaternion pds_quaternion_div_real(PdsQuaternion a,PdsCnReal b)
 *  \brief Divide un quaternio y una real. <b>a/b</b>.
 *  \param[in] a Quaternio.
 *  \param[in] b Variable real.
 *  \return La división de ambas variables.
 *  \ingroup PdsQuaternionGroup
 */
PdsQuaternion pds_quaternion_div_real(PdsQuaternion a,PdsCnReal b);


/*! \def PDS_QINV(A)
 *  \brief Es equivalente a <b>b=a^{-1}</b>.
 *  \ingroup PdsQuaternionGroup
*/
#define PDS_QINV(A) pds_quaternion_inv(A)

/** \fn PdsQuaternion pds_quaternion_inv(PdsQuaternion a)
 *  \brief Retona el inverso de un quaternio a. <b>b=a^{-1}</b>.
 * 
 *  \f[\sqrt{W^2+X^2+Y^2+Z^2}=|a|\f]
 *  \f[b=\frac{conj(a)}{|a|^2}\f]
 *
 *  \param[in] a Quaternio.
 *  \return el inverso de a.
 *  \ingroup PdsQuaternionGroup
 */
PdsQuaternion pds_quaternion_inv(PdsQuaternion a);


/*! \def PDS_QUNIT(A)
 *  \brief Es equivalente a <b>b=a/|a|</b>.
 *  \ingroup PdsQuaternionGroup
*/
#define PDS_QUNIT(A) pds_quaternion_unit(A)

/** \fn PdsQuaternion pds_quaternion_unit(PdsQuaternion a)
 *  \brief Retona un quaternio unitario de un quaternio a. <b>b=a/|a|</b>.
 * 
 *  \f[\sqrt{W^2+X^2+Y^2+Z^2}=|a|\f]
 *  \f[b=\frac{a}{|a|}\f]
 *
 *  \param[in] a Quaternio.
 *  \return el quaternio unitario de a.
 *  \ingroup PdsQuaternionGroup
 */
PdsQuaternion pds_quaternion_unit(PdsQuaternion a);


/*! \def PDS_QCONJ(A)
 *  \brief Es equivalente a <b>conjugate(A)</b>.
 *  \ingroup PdsQuaternionGroup
*/
#define PDS_QCONJ(A) pds_quaternion_conj(A)

/** \fn PdsQuaternion pds_quaternion_conj(PdsQuaternion a)
 *  \brief Retona el conjugado de un quaternio. <b>b=a.W-a.X i-a.Y j-a.Z k</b>.
 *
 *  \f[a=W+Xi+Yj+Zk\f]
 *  \f[b=conj(a)=W-Xi-Yj-Zk\f]
 *
 *  \param[in] a Quaternio.
 *  \return el conjgado de a.
 *  \ingroup PdsQuaternionGroup
 */
PdsQuaternion pds_quaternion_conj(PdsQuaternion a);


/*! \def PDS_QEXP(B)
 *  \brief Es equivalente a <b>exp(B),B=W+Xi+Yj+Zk</b>..
 *  \ingroup PdsQuaternionGroup
*/
#define PDS_QEXP(B) pds_quaternion_exp(B)

/** \fn PdsQuaternion pds_quaternion_exp(PdsQuaternion q)
 *  \brief Evalua <b>exp(q),q=W+Xi+Yj+Zk</b>.
 *
 *  \f[\sqrt{X^2+Y^2+Z^2}=M\f]
 *  \f[e^{q}=e^W \left(cos(M)+sin(M)\frac{Xi+Yj+Zk}{M}\right)\f]
 *
 *  \param[in] q Quaternio.
 *  \return El exp(q).
 *  \ingroup PdsQuaternionGroup
 */
PdsQuaternion pds_quaternion_exp(PdsQuaternion q);

/*! \def PDS_QLOG(B)
 *  \brief Es equivalente a <b>ln(B),B=W+Xi+Yj+Zk</b>..
 *  \ingroup PdsQuaternionGroup
*/
#define PDS_QLOG(B) pds_quaternion_log(B)

/** \fn PdsQuaternion pds_quaternion_log(PdsQuaternion q)
 *  \brief Evalua <b>ln(q),q=W+Xi+Yj+Zk</b>.
 *
 *  \f[\sqrt{W^2+X^2+Y^2+Z^2}=|q|\f]
 *  \f[\sqrt{X^2+Y^2+Z^2}=V\f]
 *  \f[e^{q}=ln(|q|)+arccos\left(\frac{W}{|q|}\right)\frac{Xi+Yj+Zk}{V} \f]
 *
 *  \param[in] q Quaternio.
 *  \return El ln(q).
 *  \ingroup PdsQuaternionGroup
 */
PdsQuaternion pds_quaternion_log(PdsQuaternion q);

/*! \def PDS_QPOW(B,a)
 *  \brief Es equivalente a <b>pow(B,a),B=W+Xi+Yj+Zk</b>, siendo 'a' real.
 *  \ingroup PdsQuaternionGroup
*/
#define PDS_QPOW(B,a) pds_quaternion_pow(B,a)

/** \fn PdsQuaternion pds_quaternion_pow(PdsQuaternion q,PdsCnReal a)
 *  \brief Evalua <b>pow(q,a),q=W+Xi+Yj+Zk</b>, siendo 'a' real.
 *
 *  \f[\sqrt{W^2+X^2+Y^2+Z^2}=|q|\f]
 *  \f[\sqrt{X^2+Y^2+Z^2}=V\f]
 *  \f[ \frac{W}{|q|}=cos(\theta )\f]
 *  \f[q^a= |q|^a \left( cos(a \theta )+sin(a \theta ) \frac{Xi+Yj+Zk}{V} \right) \f]
 *
 *  \param[in] q Quaternio.
 *  \param[in] a Variable real.
 *  \return El q^a.
 *  \ingroup PdsQuaternionGroup
 */
PdsQuaternion pds_quaternion_pow(PdsQuaternion q,PdsCnReal a);


/*! \def PDS_QNORM(A)
 *  \brief Es equivalente a <b>|A|=sqrt(A.W^2+A.X^2+A.Y^2+A.Z^2)</b>.
 *  \ingroup PdsQuaternionGroup
*/
#define PDS_QNORM(A) pds_quaternion_norm(A)

/** \fn PdsCnReal pds_quaternion_norm(PdsQuaternion a)
 *  \brief Evalua |a|. <b>|a|=sqrt(a.W^2+a.X^2+a.Y^2+a.Z^2)</b>.
 *
 *  \f[\sqrt{W^2+X^2+Y^2+Z^2}=|a|\f]
 *
 *  \param[in] a Quaternio.
 *  \return El |a|.
 *  \ingroup PdsQuaternionGroup
 */
PdsCnReal pds_quaternion_norm(PdsQuaternion a);



/*! \def PDS_QNORM2(A)
 *  \brief Es equivalente a <b>|A|=(A.W^2+A.X^2+A.Y^2+A.Z^2)</b>.
 *  \ingroup PdsQuaternionGroup
*/
#define PDS_QNORM2(A) pds_quaternion_norm2(A)

/** \fn PdsCnReal pds_quaternion_norm2(PdsQuaternion a)
 *  \brief Evalua |a|^2. <b>|a|^2=(a.W^2+a.X^2+a.Y^2+a.Z^2)</b>.
 *
 *  \f[W^2+X^2+Y^2+Z^2=|a|^2\f]
 *
 *  \param[in] a Quaternio.
 *  \return El |a|^2.
 *  \ingroup PdsQuaternionGroup
 */
PdsCnReal pds_quaternion_norm2(PdsQuaternion a);

/*! \def PDS_QGYRO(Wx,Wy,Wz,dt)
 *  \brief Es el quaternio de giro 'q' proveniente del vector
 *  de velocidad angular (Wx,Wy,Wz), muestreado un tiempo dt.
 *  \ingroup PdsQuaternionGroup
*/
#define PDS_QGYRO(Wx,Wy,Wz,dt) pds_quaternion_gyro(Wx,Wy,Wz,dt)

/** \fn PdsQuaternion pds_quaternion_gyro(PdsCnReal Wx,PdsCnReal Wy,PdsCnReal Wz,PdsCnReal dt)
 *  \brief Retorna un quaternio de giro q, provocado por el vetor de velocidad 
 *  angular <b>W = (Wx,Wy,Wz) = Wx i + Wy j + Wz k</b> con un tiempo hasta la muestra 
 *  anterior de 'dt'.
 *
 *  Para calcular el quaternio, es usado :
 *  \f[|W|=\sqrt{{W_x}^2+{W_y}^2+{W_z}^2}\f]
 *  \f[e_w=\frac{W}{|W|}\f]
 *  de modo que
 *  \f[\theta=|W|~dt\f]
 *  e
 *
 *  \f[q=e^{\frac{\theta}{2}~e_w}=cos(\frac{\theta}{2})+sin(\frac{\theta}{2})e_w\f]
 *
 *  Para mas información visitar fuente \cite quaternio .
 *  \param[in] Wx Velocidad angular en el eje X.
 *  \param[in] Wy Velocidad angular en el eje Y.
 *  \param[in] Wz Velocidad angular en el eje Z.
 *  \param[in] dt Tiempo de muestreo desde la muestra anterior.
 *  \return El quaternio de rotación q.
 *  \ingroup PdsQuaternionGroup
 */
PdsQuaternion pds_quaternion_gyro(PdsCnReal Wx,PdsCnReal Wy,PdsCnReal Wz,PdsCnReal dt);


/**
 * @}
 */

#ifdef __cplusplus
}
#endif 


#endif /* __PDSQUATERNION_H__ */
