/*
 * pdsrecord.c
 * 
 * Copyright 2011 Fernando Pujaico Rivera <fernando.pujaico.rivera@gmail.com>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 * 
 */

#include <pds/pdsra.h>
#include <pds/pdsrecord.h>


/** \fn PdsRecord *pds_record_new(PdsDaNatural *freq,PdsDaNatural *bits)
 *  \brief Crea una estructura para la grabación de audio digital.
 *  \param[in,out] freq Frecuencia de muestreo,si no es posible es cargada en freq
 *  la frecuencia mas próxima.
 *  \param[in,out] bits La cantidad de bits en el muestreo, si no es posible es cargada
 *  la cantidad de bits posible próxima o inmediata superior. Los valores posibles 
 *  son:8, 16,24 o32 bits.
 *  \return Un puntero a la estructura para la grabación de audio digital. o NULL
 *  en caso de error.
 *  \ingroup PdsRecordGroup
 */
PdsRecord *pds_record_new(PdsDaNatural *freq,PdsDaNatural *bits)
{
	int err;
	PdsRecord *A=NULL;
	snd_pcm_hw_params_t	*hw_params;
	int LE;

	*bits=8*((*bits)/8);

	if((*bits!=8)&&(*bits!=16)&&(*bits!=24)&&(*bits!=32))	
	{
		printf("Please tries with 8,16,24 o 32 bits.\n");
		return NULL;
	}

	A=(PdsRecord *)calloc(1,sizeof(PdsRecord));
	
	// Pido el manejador capture_handle en modo captura por la tarjeta de sonido por defecto
	err = snd_pcm_open (&(A->capture_handle), "default", SND_PCM_STREAM_CAPTURE, 0);
	if (err < 0) 
	{
		fprintf (stderr, "cannot open audio device %s (%s)\n", "default", snd_strerror (err));
		free(A);
		return NULL;
	}

	// Reservo memoria para obtener parametros del hardware de la tarjeta.
	err = snd_pcm_hw_params_malloc (&hw_params);
	if (err < 0) 
	{
		fprintf (stderr, "cannot allocate hardware parameter structure (%s)\n", snd_strerror (err));
		free(A->capture_handle);
		free(A);
		return NULL;
	}

	// Inicializa los parametros con un completo espacio de configuration para un PCM.
	err = snd_pcm_hw_params_any (A->capture_handle, hw_params);
	if (err < 0) 
	{
		fprintf (stderr, "cannot initialize hardware parameter structure (%s)\n", snd_strerror (err));
		free(A->capture_handle);
		free(hw_params);
		free(A);
		return NULL;
	}

	// Restringir el espacio de configuración con un solo tipo de acceso.
	// SND_PCM_ACCESS_MMAP_INTERLEAVED 	mmap access with simple interleaved channels
	// SND_PCM_ACCESS_MMAP_NONINTERLEAVED 	mmap access with simple non interleaved channels
	// SND_PCM_ACCESS_MMAP_COMPLEX 	mmap access with complex placement
	// SND_PCM_ACCESS_RW_INTERLEAVED 	snd_pcm_readi/snd_pcm_writei access
	// SND_PCM_ACCESS_RW_NONINTERLEAVED 	snd_pcm_readn/snd_pcm_writen access 
	err = snd_pcm_hw_params_set_access (A->capture_handle, hw_params, SND_PCM_ACCESS_RW_INTERLEAVED);
	if (err < 0) 
	{
		fprintf (stderr, "cannot set access type (%s)\n", snd_strerror (err));
		free(A->capture_handle);
		free(hw_params);
		free(A);
		return NULL;
	}

	LE=if_little_endian();

	// Restringir el espacio de configuración con un solo formato.
	// SND_PCM_FORMAT_S8 		Signed 8 bit
	// SND_PCM_FORMAT_U8 		Unsigned 8 bit
	// SND_PCM_FORMAT_S16_LE 	Signed 16 bit Little Endian
	// SND_PCM_FORMAT_U16_LE 	Unsigned 16 bit Little Endian
	// SND_PCM_FORMAT_S24_LE 	Signed 24 bit Little Endian using low three bytes in 32-bit word
	// SND_PCM_FORMAT_U24_LE 	Unsigned 24 bit Little Endian using low three bytes in 32-bit word
	// SND_PCM_FORMAT_S32_LE 	Signed 32 bit Little Endian
	// SND_PCM_FORMAT_U32_LE 	Unsigned 32 bit Little Endian
	switch(*bits)
	{
		case 8:
		{	err = snd_pcm_hw_params_set_format (A->capture_handle, hw_params, SND_PCM_FORMAT_S8);
			A->bytes=1;
			A->bits=8;
			if(err<0)	printf("Error with signed 8 bit format\nTrying signed 16 bit format\n");
			else		break;
		}

		case 16:
		{	if(LE==TRUE)	err = snd_pcm_hw_params_set_format (A->capture_handle, hw_params, SND_PCM_FORMAT_S16_LE);
			else		err = snd_pcm_hw_params_set_format (A->capture_handle, hw_params, SND_PCM_FORMAT_S16_BE);
			A->bytes=2;
			A->bits=16;
			if(err<0)	printf("Error with signed 16 bit format\nTrying signed 24 bit format\n");
			else		break;
		}

		case 24:
		{	if(LE==TRUE)	err = snd_pcm_hw_params_set_format (A->capture_handle, hw_params, SND_PCM_FORMAT_S24_LE);
			else		err = snd_pcm_hw_params_set_format (A->capture_handle, hw_params, SND_PCM_FORMAT_S24_BE);
			A->bytes=4;
			A->bits=24;
			if(err<0)	printf("Error with signed 24 bit format\nTrying signed 32 bit format\n");
			else		break;
		}

		case 32:
		{	if(LE==TRUE)	err = snd_pcm_hw_params_set_format (A->capture_handle, hw_params, SND_PCM_FORMAT_S32_LE);
			else		err = snd_pcm_hw_params_set_format (A->capture_handle, hw_params, SND_PCM_FORMAT_S32_BE);
			A->bytes=4;
			A->bits=32;
			if(err<0)	printf("Error with signed 32 bit format\nPlease tries with 8,16 or 24 bits\n");
		}
	}
	*bits=A->bits;
	if (err < 0) 
	{
		fprintf (stderr, "cannot set sample format (%s)\n", snd_strerror (err));
		free(A->capture_handle);
		free(hw_params);
		free(A);
		return NULL;
	}


	// Restringe el espacio de configuración para la tasa más cercana a un objetivo. 
	err = snd_pcm_hw_params_set_rate_near (A->capture_handle, hw_params,freq, 0);
	if (err < 0) 
	{
		fprintf (stderr, "cannot set sample rate (%s)\n", snd_strerror (err));
		free(A->capture_handle);
		free(hw_params);
		free(A);
		return NULL;
	}
	A->freq=*freq;

	//Restringe el espacio de configuración para solo un canal.
	err = snd_pcm_hw_params_set_channels (A->capture_handle, hw_params, 1);
	if (err < 0) 
	{
		fprintf (stderr, "cannot set channel count (%s)\n", snd_strerror (err));
		free(A->capture_handle);
		free(hw_params);
		free(A);
		return NULL;
	}

	// Instala la configuración de hardware PCM para espacio de configuración restringido.
	err = snd_pcm_hw_params (A->capture_handle, hw_params);
	if (err < 0) 
	{
		fprintf (stderr, "cannot set parameters (%s)\n", snd_strerror (err));
		free(A->capture_handle);
		free(hw_params);
		free(A);
		return NULL;
	}

	// Libera hw_params.
	snd_pcm_hw_params_free (hw_params);
	

	// Prepara todo para lectura de audio.
	err = snd_pcm_prepare (A->capture_handle);
	if (err < 0) 
	{
		fprintf (stderr, "cannot prepare audio interface for use (%s)\n", snd_strerror (err));
		free(A->capture_handle);
		free(A);
		return NULL;
	}

	A->samples=0;
	A->buffer1B=NULL;
	A->buffer2B=NULL;
	A->buffer4B=NULL;
	return A;
}

/** \fn int pds_record_set_samples_buffer(PdsRecord *A,PdsDaNatural samples)
 *  \brief Establece la cantidad de muestras en el buffer de lectura.
 *  \param[in,out] A estructura para la grabación de audio digital.
 *  \param[in] samples La cantidad de muestras en el buffer.
 *  \return TRUE si todo fue bien o FALSE si no. (ejem A==NULL o samples==0).
 *  \ingroup PdsRecordGroup
 */
int pds_record_set_samples_buffer(PdsRecord *A,PdsDaNatural samples)
{
	if(A==NULL)	return FALSE;
	if(samples==0)	return FALSE;

	A->samples=samples;

	switch(A->bytes)
	{
		case 1:
		{
			free(A->buffer2B);	A->buffer2B=NULL;
			free(A->buffer4B);	A->buffer4B=NULL;
			A->buffer1B=(char *)realloc(A->buffer1B,A->samples*sizeof(char));
			if(A->buffer1B==NULL)	return FALSE;
			break;
		}
		case 2:
		{
			free(A->buffer1B);	A->buffer1B=NULL;
			free(A->buffer4B);	A->buffer4B=NULL;
			A->buffer2B=(short *)realloc(A->buffer2B,A->samples*sizeof(short));
			if(A->buffer2B==NULL)	return FALSE;
			break;
		}
		case 4:
		{
			free(A->buffer1B);	A->buffer1B=NULL;
			free(A->buffer2B);	A->buffer2B=NULL;
			A->buffer4B=(int *)realloc(A->buffer4B,A->samples*sizeof(int));
			if(A->buffer4B==NULL)	return FALSE;
			break;
		}

	}


	return TRUE;
}


/** \fn int pds_record_read(PdsRecord *A,PdsVector *X)
 *  \brief Lee los datos en el buffer de lectura. Los datos estarán comprendidos 
 *  entre [-1.0,1.0>.
 *  \param[in,out] A Estructura para la grabación de audio digital.
 *  \param[in] X Vector real donde se guardarán los datos del buffer.
 *  \return TRUE si todo fue bien o FALSE si no. (ejem A==NULL, X==NULL o el
 *  número de elementos de X es distinto a la cantidad de samples en el buffer.).
 *  \ingroup PdsRecordGroup
 */
int pds_record_read(PdsRecord *A,PdsVector *X)
{
	int err;
	PdsDaNatural i;

	if(A->samples!=X->Nel)	return FALSE;

	switch(A->bytes)
	{
		case 1:	
		{	err = snd_pcm_readi (A->capture_handle, A->buffer1B, A->samples);
			break;
		}
		case 2:
		{	err = snd_pcm_readi (A->capture_handle, A->buffer2B, A->samples);
			break;
		}
		case 4:
		{	err = snd_pcm_readi (A->capture_handle, A->buffer4B, A->samples);
		}

	}
	if (err != A->samples ) 
	{
		fprintf (stderr, "read from audio interface failed (%s)\n", snd_strerror (err));
		return FALSE;
	}


	switch(A->bits)
	{
		case 8:
		{	
			for(i=0;i<A->samples;i++)	X->V[i]=A->buffer1B[i]/128.0;
			break;
		}
		case 16:
		{	
			for(i=0;i<A->samples;i++)	X->V[i]=A->buffer2B[i]/32768.0;
			break;
		}
		case 24:
		{	
			for(i=0;i<A->samples;i++)	X->V[i]=A->buffer4B[i]/8388608.0;
			break;
		}
		case 32:
		{	
			for(i=0;i<A->samples;i++)	X->V[i]=A->buffer4B[i]/2147483648.0;
		}
	}

	return TRUE;
}


/** \fn void pds_record_free(PdsRecord *A)
 *  \brief Libera la estructura para la grabación de audio digital.
 *  \param[in,out] A Estructura para la grabación de audio digital.
 *  \ingroup PdsRecordGroup
 */
void pds_record_free(PdsRecord *A)
{	
	snd_pcm_close (A->capture_handle);
	free(A->buffer1B);
	free(A->buffer2B);
	free(A->buffer4B);
	free(A);
}


/** \fn void pds_record_destroy(PdsRecord **A)
 *  \brief Libera la estructura para la grabación de audio digital, y carga con 
 *  NULL la estructura.
 *  \param[in,out] A Estructura para la grabación de audio digital.
 *  \ingroup PdsRecordGroup
 */
void pds_record_destroy(PdsRecord **A)
{	
	if((*A)!=NULL)
	{
		snd_pcm_close ((*A)->capture_handle);
		free((*A)->buffer1B);
		free((*A)->buffer2B);
		free((*A)->buffer4B);
		free((*A));
		(*A)=NULL;
	}
}



