/*
 * pdsfilesfunc.c
 * 
 * Copyright 2011 Fernando Pujaico Rivera <fernando.pujaico.rivera@gmail.com>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 * 
 */

#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <pds/pdsfilesfunc.h>
#include <pds/pdscstring.h>

#include <fcntl.h>           /* Definition of AT_* constants */
#include <unistd.h>
#include <dirent.h>
#include <fnmatch.h>

#include <sys/types.h>
#include <sys/stat.h>

/** \fn PdsCellString *pds_get_arquives_new(const char *dirpath,ArqType Type,const char* pattern_include,const char* pattern_exclude)
 *  \brief Retorna una estructura PdsCellString con un arreglo de células de nombres de arquivo.
 *  A partir de aqui pueden ser leidos los siguientes con la función pds_cell_string_read().
 *  \param[in] dirpath Camino del directorio en consulta.
 *  \param[in] Type El tipo de arquivo a leer, pueden ser los valores TYPE_DIR
 *  para directorios, TYPE_DIRTOP para directorios incluyendo la raiz o TYPE_FILE 
 *  para fiheros. Cualquier otro valor es equivalente a usar TYPE_FILE.
 *  \param[in] pattern_include Patrón de archivos a incluir solamente.
 *  \param[in] pattern_exclude Patrón de archivos a excluir.
 *  \return Retorna una estructura PdsCellString con un arreglo de células con 
 *  nombres de arquivo.
 *  \ingroup PdsFilesFuncGroup
 */
PdsCellString *pds_get_arquives_new(const char *dirpath,ArqType Type,const char* pattern_include,const char* pattern_exclude)
{
    char *dir_path=NULL;
    unsigned char  search_type;
    char *PatternInc=NULL;
    char *PatternExc=NULL;

    DIR *dir=NULL;
    struct dirent *lsdir=NULL;
    char *TEMPORAL_STRING=NULL;
    PdsCellString *fullnames=NULL;

    if (dirpath==NULL)      return NULL;
    if (strlen(dirpath)==0) return NULL;


    dir_path=realpath(dirpath,NULL);
    if (dir_path==NULL)      return NULL;    

    dir = opendir(dir_path);
    if (dir==NULL)
    {
        TEMPORAL_STRING=(char *)calloc(strlen(dir_path)+32,sizeof(char));
        sprintf(TEMPORAL_STRING,"ERROR opening %s",dir_path);
        perror(TEMPORAL_STRING);
        free(TEMPORAL_STRING);
        return NULL;
    }


    fullnames=pds_cell_string_new(1);
    if(fullnames==NULL) return NULL;

    if(Type==TYPE_DIR)          search_type=DT_DIR;
    else if(Type==TYPE_DIRTOP)  
    {
        search_type=DT_DIR;
        pds_cell_string_add(fullnames,dir_path);
    }
    else                        search_type=DT_REG;


    if (pattern_include==NULL)
    {
        PatternInc=(char*)calloc(2,sizeof(char));
        PatternInc[0]='*';
    }
    else if (strlen(pattern_include)==0)
    {
        PatternInc=(char*)calloc(2,sizeof(char));
        PatternInc[0]='*';
    }
    else
    {
        PatternInc=(char*)calloc(strlen(pattern_include)+1,sizeof(char));
        sprintf(PatternInc,"%s",pattern_include);
    }

    if (pattern_exclude==NULL)
    {
        PatternExc=(char*)calloc(2,sizeof(char));
        PatternExc[0]=0;
    }
    else if (strlen(pattern_exclude)==0)
    {
        PatternExc=(char*)calloc(2,sizeof(char));
        PatternExc[0]=0;
    }
    else
    {
        PatternExc=(char*)calloc(strlen(pattern_exclude)+1,sizeof(char));
        sprintf(PatternExc,"%s",pattern_exclude);
    }

    while ( ( lsdir = readdir(dir) ) != NULL )
    {
        if(lsdir->d_type==search_type)
        if( (strcmp(lsdir->d_name,".")!=0)&&(strcmp(lsdir->d_name,"..")!=0) )
        if(fnmatch(PatternInc, lsdir->d_name, FNM_NOESCAPE)==0)
        if(fnmatch(PatternExc, lsdir->d_name, FNM_NOESCAPE)!=0)
        {

            TEMPORAL_STRING=(char *)calloc(strlen(dir_path)+4+strlen(lsdir->d_name),sizeof(char));
            sprintf(TEMPORAL_STRING,"%s%c%s",dir_path,pds_filesep(),lsdir->d_name);

            pds_cell_string_add(fullnames,TEMPORAL_STRING);

            free(TEMPORAL_STRING);
        }
    }

    closedir(dir);
    free(dir_path);
    free(PatternInc);
    free(PatternExc);

    if(pds_cell_string_get_last_empty(fullnames)==0)
    {
        pds_cell_string_destroy(&fullnames);
	    return NULL;
    }
    else
    {
	    return fullnames;
    }
}

/** \fn char pds_filesep(void)
 *  \brief Retorna um caracter con el separador de archivo para el sistema.
 *  \return Retorna um caracter con el separador de archivo para el sistema.
 *  En caso de error de dedución, retorna '/' por defecto.
 *  \ingroup PdsFilesFuncGroup
 */
char pds_filesep(void)
{
    char c='/';

    char *string_data=NULL;

    string_data=(char*)getcwd(0,0);

    if(string_data==NULL)
    {
        perror("ERROR deducing the file separator assuming: /");
        return c;
    }

    if (strlen(string_data)>=2)
    if ( ((string_data[0]>=65)&&(string_data[0]<=90)) || ((string_data[0]>=97)&&(string_data[0]<=122)) )
    if (string_data[1]==':')
    {   
        c='\\';
    }
    
    free(string_data);
    return c;
}



/** \fn PdsCellString *pds_get_all_subdirs_new(const char *dirpath,const char* pattern_include,const char* pattern_exclude)
 *  \brief Retorna una estructura PdsCellString con un arreglo de células con 
 *  nombres de subdirectorios incluyendo la raiz y excluyendo los simbolicos.
 *  A partir de aquí pueden ser leidos los datos con la función pds_cell_string_read().
 *  \param[in] dirpath Camino del directorio en consulta.
 *  \param[in] pattern_include Patrón de directorios a incluir solamente.
 *  \param[in] pattern_exclude Patrón de directorios a excluir.
 *  \return Retorna una estructura PdsCellString con un arreglo de células com 
 *  nombres de subdirectorios incluyendo la raiz y excluyendo los simbólicos.
 *  \ingroup PdsFilesFuncGroup
 */
PdsCellString *pds_get_all_subdirs_new(const char *dirpath,const char* pattern_include,const char* pattern_exclude)
{
    int dat;
    char *dir_path=NULL;

    PdsCellString *C0=NULL;
    PdsCellString *Ca=NULL;
    PdsCellString *Cb=NULL;

    if(dirpath==NULL)    return NULL;

    C0=pds_cell_string_new(0);
    if(C0==NULL)    return NULL;

    Ca=pds_cell_string_new(1);
    if(Ca==NULL)    {   pds_cell_string_free(C0);
                        return NULL;}


    dir_path=realpath(dirpath,NULL);
    if(dir_path==NULL)    return NULL;

    dat=pds_cell_string_add(Ca,dir_path);
    if(dat==FALSE)  {   free(dir_path);
                        pds_cell_string_free(Ca);
                        pds_cell_string_free(C0);
                        return NULL;}


    dat=pds_cell_string_add_cell_string(C0,Ca);
    if(dat==FALSE)  {   free(dir_path);
                        pds_cell_string_free(Ca);
                        pds_cell_string_free(C0);
                        return NULL;}
    
    do{

        Cb=pds_get_subdirs_new(Ca,pattern_include,pattern_exclude);


        dat=pds_cell_string_add_cell_string(C0,Cb);
        if(dat==FALSE)  {   free(dir_path);
                            pds_cell_string_free(Ca);
                            pds_cell_string_free(Cb);
                            pds_cell_string_free(C0);
                            return NULL;}
        
        pds_cell_string_free(Ca);
        Ca=Cb;

    }while(Cb!=NULL);

    free(dir_path);    
    pds_cell_string_free(Ca);
    pds_cell_string_free(Cb);

    return C0;
}

/** \fn PdsCellString *pds_get_subdirs_new(const PdsCellString *C0,const char* pattern_include,const char* pattern_exclude)
 *  \brief Retorna una estructura PdsCellString con un arreglo de células, con  
 *  los nombres de los subdirectorios del siguiente nivel de los directorios del 
 *  arreglo de células C0 y excluyendo los simbólicos. 
 *  El arreglo de células de salida no contiene los directorios de entrada dentro de C0.
 *  A partir de aquí pueden ser leidos los datos con la función pds_cell_string_read().
 *  \param[in] C0 Un arreglo de células con los directorios a buscar.
 *  \param[in] pattern_include Patrón de directorios a incluir solamente.
 *  \param[in] pattern_exclude Patrón de directorios a excluir.
 *  \return Retorna una estructura PdsCellString con un arreglo de células con 
 *  los nombres de los subdirectorios del siguiente nivel (solamente un nivel)
 *  y excluyendo los simbólicos O NULL en caso de error.
 *  \ingroup PdsFilesFuncGroup
 */
PdsCellString *pds_get_subdirs_new(const PdsCellString *C0,const char* pattern_include,const char* pattern_exclude)
{
    int i,dat;
    PdsCellString *Cout=NULL;
    PdsCellString *Ctmp=NULL;

    if(C0==NULL)    return NULL;

    Cout=pds_cell_string_new(0);
    if(Cout==NULL)    return NULL;


    for(i=0;i<C0->Ncell;i++)
    {
        Ctmp=pds_get_arquives_new(C0->data[i],TYPE_DIR,pattern_include,pattern_exclude);
        dat=pds_cell_string_add_cell_string(Cout,Ctmp);
        if(dat==FALSE)
        {
            pds_cell_string_free(Ctmp);
            pds_cell_string_free(Cout);
            return NULL;
        }
    }

    pds_cell_string_free(Ctmp);

    if(pds_cell_string_get_last_empty(Cout)==0)
    {
        pds_cell_string_free(Cout);
        return NULL;
    }

    return Cout;
}




/** \fn PdsCellString *pds_get_all_files_new(const char *dirpath,   const char* pattern_include,const char* pattern_exclude,const char* file_pattern_include,const char* file_pattern_exclude)
 *  \brief Retorna una estructura PdsCellString con un arreglo de células con los 
 *  nombres de archivos. No son buscados archivos en los directorios simbólicos.
 *  A partir de aquí pueden ser leidos los datos con la función pds_cell_string_read().
 *  \param[in] dirpath Camino del directorio en consulta.
 *  \param[in] pattern_include Patrón de directorios a incluir solamente.
 *  \param[in] pattern_exclude Patrón de directorios a excluir.
 *  \param[in] file_pattern_include Patrón de archivos a incluir solamente.
 *  \param[in] file_pattern_exclude Patrón de archivos a excluir.
 *  \return Retorna una estructura PdsCellString con un arreglo de células con 
 *  los nombres de todos los archivos.
 *  \ingroup PdsFilesFuncGroup
 */
PdsCellString *pds_get_all_files_new(const char *dirpath,   const char* pattern_include,
                                                            const char* pattern_exclude,
                                                            const char* file_pattern_include,
                                                            const char* file_pattern_exclude)
{
    PdsCellString *DIRECTORY=NULL;
    PdsCellString *TmpFileCell=NULL;
    PdsCellString *FileCell=NULL;
    int i,dat;

    if(dirpath==NULL)   return NULL;
    
    FileCell=pds_cell_string_new(0);
    if(FileCell==NULL)   return NULL;    

    DIRECTORY=pds_get_all_subdirs_new(dirpath,pattern_include,pattern_exclude);
    if(DIRECTORY==NULL)  { pds_cell_string_free(FileCell); return NULL;}

    for(i=0;i<DIRECTORY->Ncell;i++)
    {
        TmpFileCell=pds_get_arquives_new(   DIRECTORY->data[i],TYPE_FILE,
                                            file_pattern_include,
                                            file_pattern_exclude);
        dat=pds_cell_string_add_cell_string(FileCell,TmpFileCell);
        if(dat==FALSE)
        {
            pds_cell_string_free(FileCell);
            pds_cell_string_free(TmpFileCell);
            pds_cell_string_free(DIRECTORY);
            return NULL;
        }
    }


    pds_cell_string_free(TmpFileCell);
    pds_cell_string_free(DIRECTORY);

    if(pds_cell_string_get_last_empty(FileCell)==0)
    {
        pds_cell_string_free(FileCell);
        return NULL;
    }

    return FileCell;
}


/** \fn char *pds_get_basename(const char * file_path)
 *  \brief Obtén el nombre base desde una ruta de fichero.
 *
 *  Si la ruta de fichero es: /path_to_file/filename.c
 *  La función retorna:       filename.c
 *  \param[in] file_path Ruta del fichero a analizar.
 *  \return Retorna una nueva cadena de texto con el nombre base, o NULL en caso 
 *  de error de reserva de memoria.
 *  \ingroup PdsFilesFuncGroup
 */
char *pds_get_basename(const char * file_path)
{
    char *ptr = NULL;

    ptr=strrchr(file_path,pds_filesep());

    if(ptr==NULL)   return strdup(file_path);
    else            return strdup(ptr + 1);
}


/** \fn char *pds_get_filename_extension(const char * file_path)
 *  \brief Obtén la extención de fichero desde una ruta de fichero.
 *  Esta función tendrá problemas con archivos ocultos ".abcd", pues
 *  entenderá "abcd" como una extención de fihero.
 *
 *  Si la ruta de fichero es: /path_to_file/filename.c
 *  La función retorna:       c
 *  \param[in] file_path Ruta del fichero a analizar.
 *  \return Retorna una nueva cadena de texto con la extención de fichero, o 
 *  NULL en caso de error de reserva de memoria o fihero sin extención.
 *  \ingroup PdsFilesFuncGroup
 */
char *pds_get_filename_extension(const char * file_path)
{
    char *ptr = NULL;
    char *ptrout = NULL;
    char *base_name=NULL;

    base_name=pds_get_basename(file_path);
    if(base_name==NULL)  return NULL;

    ptr=strrchr(base_name,'.');

    if(ptr==NULL)
    {
        free(base_name); 
        return NULL;
    }
    else
    {
        ptrout=strdup(ptr + 1);
        free(base_name);
        return ptrout;
    }
}


/** \fn char *pds_get_filename(const char * file_path)
 *  \brief Obtén el nombre de fichero desde una ruta de fichero.
 *  Esta función tendrá problemas con archivos ocultos ".abcd", pues
 *  retornará una cadena vacía.
 *
 *  Si la ruta de fichero es: /path_to_file/filename.c
 *  La función retorna:       filename
 *  \param[in] file_path Ruta del fichero a analizar.
 *  \return Retorna una nueva cadena de texto con el nombre de fichero, o NULL 
 *  en caso de error de reserva de memoria.
 *  \ingroup PdsFilesFuncGroup
 */
char *pds_get_filename(const char * file_path)
{
    char *base_name=NULL;
    char *file_name=NULL;
    char *ptr=NULL;
    int file_name_size;
    int i;

    base_name=pds_get_basename(file_path);
    if(base_name==NULL)  return NULL;

    ptr=strrchr(base_name,'.');

    if(ptr==NULL)
    {
        file_name=strdup(base_name);
    }
    else
    {
        file_name_size=(int)(ptr-base_name);
        file_name=(char*) calloc(file_name_size+1,sizeof(char));
        if(file_name!=NULL)
        {
            for(i=0;i<file_name_size;i++) file_name[i]=base_name[i];
        }
    }

    free(base_name);
    return file_name;
}


/** \fn char *pds_get_dirname(const char * file_path)
 *  \brief Obtén el directorio madre desde una ruta de fichero.
 *
 *  Si la ruta de fichero es: /path_to_file/filename.c
 *  La función retorna:       /path_to_file
 *  \param[in] file_path Ruta del fichero a analizar.
 *  \return Retorna una nueva cadena de texto con el directorio madre, o 
 *  NULL en caso de error de reserva de memoria o fihero sin directorio madre.
 *  \ingroup PdsFilesFuncGroup
 */
char *pds_get_dirname(const char * file_path)
{
    char *dir_path=NULL;
    char *ptr=NULL;
    int dir_path_size;
    int i;

    ptr=strrchr(file_path,pds_filesep());

    if(ptr==NULL)
    {
        dir_path=NULL;
    }
    else
    {
        dir_path_size=(int)(ptr-file_path);
        dir_path=(char*) calloc(dir_path_size+1,sizeof(char));
        if(dir_path!=NULL)
        {
            for(i=0;i<dir_path_size;i++) dir_path[i]=file_path[i];
        }
    }

    return dir_path; 
}





/** \fn PdsCellString *pds_get_all_filename_extensions(const char *dirpath,   const char* pattern_include,const char* pattern_exclude,const char* file_pattern_include,const char* file_pattern_exclude)
 *  \brief Retorna una estructura PdsCellString con un arreglo de células con las 
 *  extenciones de archivos en el directorio. No son buscados archivos en los directorios simbólicos.
 *  A partir de aquí pueden ser leidos los datos con la función pds_cell_string_read().
 *  \param[in] dirpath Camino del directorio en consulta.
 *  \param[in] pattern_include Patrón de directorios a incluir solamente.
 *  \param[in] pattern_exclude Patrón de directorios a excluir.
 *  \param[in] file_pattern_include Patrón de archivos a incluir solamente.
 *  \param[in] file_pattern_exclude Patrón de archivos a excluir.
 *  \return Retorna una estructura PdsCellString con un arreglo de células con 
 *  los nombres de todas las extenciones de archivos.
 *  \ingroup PdsFilesFuncGroup
 */
PdsCellString *pds_get_all_filename_extensions(const char *dirpath,   const char* pattern_include,
                                                            const char* pattern_exclude,
                                                            const char* file_pattern_include,
                                                            const char* file_pattern_exclude)
{
    PdsCellString *ExtCell=NULL;
    PdsCellString *FileCell=NULL;
    char *strPtr=NULL;
    int i,dat;

    if(dirpath==NULL)   return NULL;

    FileCell=pds_get_all_files_new(dirpath,pattern_include,pattern_exclude,file_pattern_include,file_pattern_exclude);
    if(FileCell==NULL)  return NULL;

    
    ExtCell=pds_cell_string_new(0);
    if(ExtCell==NULL)   {pds_cell_string_free(FileCell);return NULL;}

    for(i=0;i<FileCell->Ncell;i++)
    {
        strPtr=pds_get_filename_extension(FileCell->data[i]);
        
        if(strPtr!=NULL)
        if(strlen(strPtr)>0)
        if(pds_cell_string_find(ExtCell,strPtr)<0)
        {
            dat=pds_cell_string_add(ExtCell,strPtr);
            if(dat==FALSE)
            {
                pds_cell_string_free(FileCell);
                pds_cell_string_free(ExtCell);
                return NULL;
            }
        }
    }


    pds_cell_string_free(FileCell);

    if(pds_cell_string_get_last_empty(ExtCell)==0)
    {
        pds_cell_string_free(ExtCell);
        return NULL;
    }

    return ExtCell;
}


/** \fn char *pds_fgets(FILE *fd)
 *  \brief Retorna una linea leida desde un desriptor de fichero de texto.
 *
 *  \param[in] fd Descriptor de fichero de texto a analizar.
 *  \return Retorna una nueva cadena de texto con una copia de la linea leida 
 *  incluyendo los caracteres de control finales exepto EOF.
 *  \ingroup PdsFilesFuncGroup
 */
char *pds_fgets(FILE *fd)
{
    char *line=NULL;
    int i;
    fpos_t init;
    int count,N;
    char c;

    if(fd==NULL)    return NULL;

    fgetpos(fd,&init);

    count=0;
    c=0;
    while( (c!=10)&&(c!=EOF) )  /* \n */
    {
        c=fgetc(fd);
        if(c!=EOF) count++;
    }

    if(c!=EOF)
    {
        c=fgetc(fd);
        /* \r */
        if(c==13) count++;
    }
    fsetpos(fd,&init);

    

    N=count;


    line=(char*)calloc(N+1,sizeof(char));
    if (line==NULL) return NULL;

    for(i=0;i<N;i++)
    {    
        line[i]=fgetc(fd);
    }

    if(c==EOF)  
    {
        fseek( fd, 0, SEEK_END );
        fgetc(fd);
    }

    return line;
}


/** \fn char *pds_read_line(FILE *fd)
 *  \brief Retorna una linea leida desde un desriptor de fichero de texto.
 *
 *  \param[in] fd Descriptor de fichero de texto a analizar.
 *  \return Retorna una nueva cadena de texto con una copia de la linea leida 
 *  descartando los caracteres despues de '\n'.
 *  \ingroup PdsFilesFuncGroup
 */
char *pds_read_line(FILE *fd)
{
    char *data=NULL;
    char *tmp=NULL;
    int i,N;

    if(fd==NULL)    return NULL;

    tmp=pds_fgets(fd);
    if(tmp==NULL)   return NULL;

    N=strlen(tmp);

    for(i=0;i<strlen(tmp);i++)
    {
        if(tmp[i]==10) /* \n */
        {
            N=i;
            break;
        }
    }

    data=(char*)calloc(N+1,sizeof(char));
    if(data==NULL)
    {
        free(tmp);
        return NULL;
    }
    
    for(i=0;i<N;i++)    data[i]=tmp[i];

    free(tmp);
    return data;
}


/** \fn int pds_is_file(const char *path)
 *  \brief TRUE si es un fichero (simbólico o real) o FALSE si no.
 *
 *  \param[in] path Dirección de fichero a testar.
 *  \return Retorna TRUE is es un fichero o FALSE si no.
 *  \ingroup PdsFilesFuncGroup
 */
int pds_is_file(const char *path)
{
    struct stat stat_result;
    stat(path, &stat_result);

    if(S_ISREG(stat_result.st_mode))    return TRUE;
    else                                return FALSE;
}


/** \fn int pds_is_dir(const char *path)
 *  \brief TRUE is es un directorio (simbólico o real) o FALSE si no.
 *
 *  \param[in] path Dirección de directorio a testar.
 *  \return Retorna TRUE is es un directorio o FALSE si no.
 *  \ingroup PdsFilesFuncGroup
 */
int pds_is_dir(const char *path)
{
    struct stat stat_result;
    stat(path, &stat_result);

    if(S_ISDIR(stat_result.st_mode))    return TRUE;
    else                                return FALSE;
}


/** \fn long pds_get_file_size(const char *path)
 *  \brief Retorna el número de bytes del archivo.
 *
 *  \param[in] path Dirección de archivo a testar.
 *  \return Retorna el número de bytes del archivo.
 *  \ingroup PdsFilesFuncGroup
 */
long pds_get_file_size(const char *path)
{
    FILE *fd=NULL;
    long sz;

    if(!pds_is_file(path))  return -1;

    fd=fopen(path,"rb");
    if(fd==NULL)    return -1;

    fseek(fd, 0L, SEEK_END);
    sz = ftell(fd);

    fclose(fd);
    return sz;
}


/** \fn int pds_exist_file(const char *path)
 *  \brief Retorna TRUE si path es un fichero real o simbólico (no un directorio) 
 *  y es accesible.
 *
 *  \param[in] path Dirección de fichero a testar.
 *  \return Retorna TRUE si el archivo existe o FALSE si no.
 *  \ingroup PdsFilesFuncGroup
 */
int pds_exist_file(const char *path)
{
    if(pds_archive_is_accessible(path))
    {
        if(pds_is_file(path))   return TRUE;
        else                    return FALSE;
    }
    else    return FALSE;
    
}


/** \fn int pds_archive_is_accessible(const char *path)
 *  \brief Retorna TRUE si el archivo simbólico o real (Directorio o fichero) 
 *  es accesible.
 *
 *  \param[in] path Dirección de archivo a testar.
 *  \return Retorna TRUE si el archivo existe o FALSE si no.
 *  \ingroup PdsFilesFuncGroup
 */
int pds_archive_is_accessible(const char *path)
{
    if( access( path, F_OK ) != -1 ) 
    {
        return TRUE;
    } 
    else
    {
        return FALSE;
    }
}

