/*
 * pdscstring.h
 * 
 * Copyright 2011 Fernando Pujaico Rivera <fernando.pujaico.rivera@gmail.com>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 * 
 */

/** \file pdscstring.h
 *  \author Fernando Pujaico Rivera
 *  \date 30-07-2016
 *  \brief Funciones que trabajan con arreglos de células con cadenas de texto.
 *   
 */

#ifndef __PDSCSTRING_H__
#define __PDSCSTRING_H__


#ifdef __cplusplus
extern "C" {
#endif 

#include <stdio.h>

#ifndef TRUE
	#define TRUE 1
#endif

#ifndef FALSE
	#define FALSE 0
#endif

/** \defgroup PdsCellStringGroup Funciones del módulo PdsCellString.
 *
 *  Estas funciones manipulan un arreglo de células con elemento de tipo cadena.
 *  <br>
 *  Funciones.
 * @{
 */



/*! \struct PdsCellString
 *  \brief Una estructura tipo  PdsCellString .
 *  Esta estructura genera un arreglo de células con de cadenas te texto.
 *  <br>
 *
 *  \ingroup PdsCellStringGroup
 *  \author Fernando Pujaico Rivera
 */
typedef struct 
{
	char **data;
    int Ncell;
}PdsCellString;



/** \fn PdsCellString *pds_cell_string_new(int N)
 *  \brief Crea un arreglo de células de tipo PdsCellString con N elementos 
 *  vacíos (NULL).
 *  \param[in] N Número de elementos de elementos del arreglo de células.
 *  \return Un puntero a una estructura de tipo PdsCellString.
 *  \ingroup PdsCellStringGroup
 */
PdsCellString *pds_cell_string_new(int N);


/** \fn PdsCellString *pds_cell_string_copy_new(const PdsCellString *Cin)
 *  \brief Crea un arreglo de células de tipo PdsCellString copiando los datos de otra.
 *  \param[in] Cin La célula a ser copiada.
 *  \return Un puntero a una estructura de tipo PdsCellString o NULL en caso de error.
 *  \ingroup PdsCellStringGroup
 */
PdsCellString *pds_cell_string_copy_new(const PdsCellString *Cin);


/** \fn int pds_cell_string_write(PdsCellString *C,int id,const char* string_data)
 *  \brief Escribe un dato de tipo char* en el elemento id del arreglo de células.
 *  \param[in,out] C El arreglo de células a escribir.
 *  \param[in] id Posición en donde escribir el dato.
 *  \param[in] string_data La cadena a escribir.
 *  \return TRUE si todo fue bien o FALSE si no. Retorna FALSE si el id está fuera 
 *  de rango.
 *  \ingroup PdsCellStringGroup
 */
int pds_cell_string_write(PdsCellString *C,int id,const char* string_data);



/** \fn int pds_cell_string_all_printf(const PdsCellString *C)
 *  \brief Imprime en pantalla los datos del arreglo de células de tipo puntero 
 *  PdsCellString. Escribe un dato por cada linea.
 *  \param[in] C El arreglo de células a imprimir en pantalla.
 *  \return TRUE si todo fue bien o FALSE si no.
 *  \ingroup PdsCellStringGroup
 */
int pds_cell_string_all_printf(const PdsCellString *C);


/** \fn int pds_cell_string_all_fprintf(FILE* fd,const PdsCellString *C)
 *  \brief Imprime en el descriptor de fichero los datos del arreglo de células 
 *  de tipo puntero PdsCellString. Escribe un dato por cada linea.
 *  \param[in,out] fd El descriptor de fichero sobre el cual escribir.
 *  \param[in] C El arreglo de células a escribir.
 *  \return TRUE si todo fue bien o FALSE si no.
 *  \ingroup PdsCellStringGroup
 */
int pds_cell_string_all_fprintf(FILE* fd,const PdsCellString *C);


/** \fn int pds_cell_string_save(FILE* fd,const PdsCellString *C)
 *  \brief Imprime en el descriptor de fichero los datos del arreglo de células 
 *  de tipo puntero PdsCellString. Escribe un dato por cada linea.
 *  \param[in] pathname La dirección de fichero sobre el cual escribir.
 *  \param[in] C El arreglo de células a escribir.
 *  \return TRUE si todo fue bien o FALSE si no.
 *  \ingroup PdsCellStringGroup
 */
int pds_cell_string_save(const char* pathname,const PdsCellString *C);


/** \fn void pds_cell_string_free(PdsCellString *C)
 *  \brief Libera un arreglo de células de tipo puntero PdsCellString. 
 *  \param[in,out] C El arreglo de células a liberar.
 *  \return No retorna valor.
 *  \ingroup PdsCellStringGroup
 */
void pds_cell_string_free(PdsCellString *C);


/** \fn void pds_cell_string_destroy(PdsCellString **C)
 *  \brief Libera un arreglo de células de tipo puntero PdsCellString y carga con NULL.
 *  \param[in,out] C El arreglo de células a liberar.
 *  \return No retorna valor.
 *  \ingroup PdsCellStringGroup
 */
void pds_cell_string_destroy(PdsCellString **C);


/** \fn int pds_cell_string_add(PdsCellString *C,const char* string_data)
 *  \brief Agrega un elemento en la ultima posición vacía(NULL) del arreglo de células. 
 *  O agrega un nuevo registro de memoria.
 *  \param[in,out] C El arreglo de células para agregar datos.
 *  \param[in] string_data La cadena a escribir.
 *  \return TRUE si todo fue bien o FALSE si no. Si string_data==NULL 
 *  no es considerado error y se finaliza la función sin hacer ningún cambio.
 *  \ingroup PdsCellStringGroup
 */
int pds_cell_string_add(PdsCellString *C,const char* string_data);


/** \fn int pds_cell_string_add_cell_string(PdsCellString *C,const PdsCellString *Cin)
 *  \brief Agrega un conjunto de elementos a partir de la última posición 
 *  vacía (NULL) del arreglo de células (Y/O agrega nuevos registros de memoria).
 *  Si el sumando Cin es null no agrega nada y sale sin decir error y sin alterar 
 *  el acumulador C.
 *
 *  ESTA FUNCION DEBE SER HECHA DE NUEVO SIN USAR pds_cell_string_add PARA 
 *  NO LIBERAR Y RESERVAR MEMORIA TAN ALEGREMENTE. ASSI SE GANARIA VELOCIDAD.
 *  \param[in,out] C El arreglo de células a agregar datos.
 *  \param[in] Cin El arreglo de células de donde se leen los datos.
 *  \return TRUE si todo fue bien o FALSE si no.
 *  \ingroup PdsCellStringGroup
 */
int pds_cell_string_add_cell_string(PdsCellString *C,const PdsCellString *Cin);


/** \fn int pds_cell_string_read(const PdsCellString *C,int id,char** string_data)
 *  \brief Lee una cadena de texto, en la posición id del arreglo de células.
 *  \param[in,out] C El arreglo de células a analizar.
 *  \param[id] id La posición a leer.
 *  \param[out] string_data La dirección de la cadena a leer (NO debe liberarse esta 
 *  dirección de memoria).
 *  \return TRUE si todo fue bien o FALSE si no.
 *  \ingroup PdsCellStringGroup
 */
int pds_cell_string_read(const PdsCellString *C,int id,char** string_data);


/** \fn int pds_cell_string_get_ncell(const PdsCellString *C)
 *  \brief Cuenta la cantidad de células incluyendo las vacías.
 *  \param[in] C El arreglo de células a analizar.
 *  \return Retorna la cantidad de células o un número menor que cero 
 *  en caso de error.
 *  \ingroup PdsCellStringGroup
 */
int pds_cell_string_get_ncell(const PdsCellString *C);


/** \fn int pds_cell_string_get_last_empty(const PdsCellString *C)
 *  \brief Retorna el id de la última célula libre (osea con NULL).
 *  \param[in] C El arreglo de células a analizar.
 *  \return Retorna el id de la última célula libre, o un número menor que cero 
 *  en caso de error. En caso de que el arreglo esté lleno la función retornará
 *  C->Ncell.
 *  \ingroup PdsCellStringGroup
 */
int pds_cell_string_get_last_empty(const PdsCellString *C);


/** \fn int pds_cell_string_find(const PdsCellString *C,const char *str)
 *  \brief Retorna el id de la primera ocurrencia de la cadena str en el arreglo de 
 *  células C.
 *  \param[in] C El arreglo de células a analizar.
 *  \param[in] str La cadena a buscar.
 *  \return Retorna el id de la primera ocurrencia de la cadena str, o un número 
 *  menor que cero en caso de error.
 *  \ingroup PdsCellStringGroup
 */
int pds_cell_string_find(const PdsCellString *C,const char *str);


/** \fn int pds_cell_string_findci(const PdsCellString *C,const char *str)
 *  \brief Retorna el id de la primera ocurrencia de la cadena str en el arreglo de 
 *  células C. La función es case-insensitive.
 *  \param[in] C El arreglo de células a analizar.
 *  \param[in] str La cadena a buscar.
 *  \return Retorna el id de la primera ocurrencia de la cadena str, o un número 
 *  menor que cero en caso de error.
 *  \ingroup PdsCellStringGroup
 */
int pds_cell_string_findci(const PdsCellString *C,const char *str);


/** \fn PdsCellString *pds_cell_string_new_load_data(const char*filepath)
 *  \brief Crea un arreglo de células de tipo PdsCellString con los datos de cada 
 *  linea del archivo.
 *  \param[in] filepath La dirección del archivo de donde se cargarán los datos.
 *  \return Un puntero a una estructura de tipo PdsCellString.
 *  \ingroup PdsCellStringGroup
 */
PdsCellString *pds_cell_string_new_load_data(const char*filepath);


/** \fn int pds_cell_string_remove(PdsCellString *C,int id)
 *  \brief Remueve la célula de posición id.
 *
 *  \param[in,out] C El arreglo de células a remover datos.
 *  \param[in] id El id de la célula a remover.
 *  \return TRUE si todo fue bien o FALSE si no.
 *  En caso de error no modifica el contenido de C.
 *  \ingroup PdsCellStringGroup
 */
int pds_cell_string_remove(PdsCellString *C,int id);


/** \fn int pds_cell_string_filter_orinc(PdsCellString *C,const PdsCellString *pattern)
 *  \brief Remueve las células que no cumplen con por lo menos un patrón 
 *  especificado por pattern.
 *
 *  \param[in,out] C El arreglo de células a remover datos.
 *  \param[in] pattern El patrón de búsqueda.
 *  \return TRUE si todo fue bien o FALSE si no.
 *  En caso de error el contenido de C no es modificado. Si pattern es NULL no 
 *  es considerado error y se retorn TRUE sin hacer nada.
 *  \ingroup PdsCellStringGroup
 */
int pds_cell_string_filter_orinc(PdsCellString *C,const PdsCellString *pattern);


/** \fn int pds_cell_string_findci_and_add(PdsCellString *C,const char* string_data)
 *  \brief Agrega un nuevo registro de memoria si la cadena string_data no existe 
 *  en la estructura *C. Si esta ya existe la sobre escribe pues esta función no
 *  diferencia mayúsculas y minúsculas.
 *  \param[in,out] C El arreglo de células para agregar datos.
 *  \param[in] string_data La cadena a escribir.
 *  \return TRUE si todo fue bien o FALSE si no. Si string_data==NULL 
 *  no es considerado error y se finaliza la función sin hacer ningún cambio.
 *  \ingroup PdsCellStringGroup
 */
int pds_cell_string_findci_and_add(PdsCellString *C,const char* string_data);


/** \fn int pds_cell_string_find_and_remove(const PdsCellString *C,const char *str)
 *  \brief Procura todas las células con cadenas igual a str y las remueve.
 *  \param[in,out] C El arreglo de células a analizar.
 *  \param[in] str La cadena a buscar.
 *  \return Retorna TRUE si todo fue bien o FALSE si no. Si C o str son nulos
 *  entonces retorna FALSE sin hacer nada sobre C.
 *  \ingroup PdsCellStringGroup
 */
int pds_cell_string_find_and_remove(PdsCellString *C,const char *str);


/** \fn int pds_cell_string_findci_and_remove(const PdsCellString *C,const char *str)
 *  \brief Procura todas las células con cadenas igual a str y las remueve.
 *  \param[in,out] C El arreglo de células a analizar.
 *  La función es case-insensitive.
 *  \param[in] str La cadena a buscar.
 *  \return Retorna TRUE si todo fue bien o FALSE si no. Si C o str son nulos
 *  entonces retorna FALSE sin hacer nada sobre C.
 *  \ingroup PdsCellStringGroup
 */
int pds_cell_string_findci_and_remove(PdsCellString *C,const char *str);



/**
 * @}
 */

#ifdef __cplusplus
}
#endif 

#endif	/* __PDSCSTRING_H__ */ 

