/*
 * pdsfilesfunc.h
 * 
 * Copyright 2011 Fernando Pujaico Rivera <fernando.pujaico.rivera@gmail.com>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 * 
 */

/** \file pdsfilesfunc.h
 *  \author Fernando Pujaico Rivera
 *  \date 30-07-2016
 *  \brief Funciones que manipulan archivos y caminos.
 *   
 */

#ifndef __PDSFILESFUNC_H__
#define __PDSFILESFUNC_H__


#ifdef __cplusplus
extern "C" {
#endif 

#include <stdio.h>
#include <stdio.h>
#include <pds/pdscstring.h>

#ifndef TRUE
	#define TRUE 1
#endif

#ifndef FALSE
	#define FALSE 0
#endif

/** \defgroup PdsFilesFuncGroup Funciones del módulo PdsFilesFunc.
 *
 *  <br>
 *  Funciones.
 * @{
 */


/*! \enum ArqType
 *  \brief Una union de tipo ArqType.
 *  Esta union tiene dos tipos de datos tipo directorio y tipo documento.
 *  <br>
 *
 *  \ingroup PdsFilesFuncGroup
 *  \author Fernando Pujaico Rivera
 */
typedef enum{
    TYPE_FILE, /**< Buscando ficheros excluyendo los simbólicos*/
    TYPE_DIR,  /**< Buscando directorios excluyendo los simbólicos*/
    TYPE_DIRTOP, /**< Buscando directorios excluyendo los simbólicos incluyendo la raiz*/
}ArqType;


/** \fn PdsCellString *pds_get_arquives_new(const char *dirpath,ArqType Type,const char* pattern_include,const char* pattern_exclude)
 *  \brief Retorna una estructura PdsCellString con un arreglo de células de nombres de arquivo.
 *  A partir de aqui pueden ser leidos los siguientes con la función pds_cell_string_read().
 *  \param[in] dirpath Camino del directorio en consulta.
 *  \param[in] Type El tipo de arquivo a leer, pueden ser los valores TYPE_DIR
 *  para directorios, TYPE_DIRTOP para directorios incluyendo la raiz o TYPE_FILE 
 *  para fiheros. Cualquier otro valor es equivalente a usar TYPE_FILE.
 *  \param[in] pattern_include Patrón de archivos a incluir solamente.
 *  \param[in] pattern_exclude Patrón de archivos a excluir.
 *  \return Retorna una estructura PdsCellString con un arreglo de células con 
 *  nombres de arquivo.
 *  \ingroup PdsFilesFuncGroup
 */
PdsCellString *pds_get_arquives_new(const char *dirpath,ArqType Type,const char* pattern_include,const char* pattern_exclude);


/** \fn char pds_filesep(void)
 *  \brief Retorna um caracter con el separador de archivo para el sistema.
 *  \return Retorna um caracter con el separador de archivo para el sistema.
 *  En caso de error de dedución, retorna '/' por defecto.
 *  \ingroup PdsFilesFuncGroup
 */
char pds_filesep(void);


/** \fn PdsCellString *pds_get_all_subdirs_new(const char *dirpath,const char* pattern_include,const char* pattern_exclude)
 *  \brief Retorna una estructura PdsCellString con un arreglo de células con 
 *  nombres de subdirectorios incluyendo la raiz y excluyendo los simbólicos.
 *  A partir de aquí pueden ser leidos los datos con la función pds_cell_string_read().
 *  \param[in] dirpath Camino del directorio en consulta.
 *  \param[in] pattern_include Patrón de directorios a incluir solamente.
 *  \param[in] pattern_exclude Patrón de directorios a excluir.
 *  \return Retorna una estructura PdsCellString con un arreglo de células com 
 *  nombres de subdirectorios incluyendo la raiz y excluyendo los simbólicos.
 *  \ingroup PdsFilesFuncGroup
 */
PdsCellString *pds_get_all_subdirs_new(const char *dirpath,const char* pattern_include,const char* pattern_exclude);


/** \fn PdsCellString *pds_get_subdirs_new(const PdsCellString *C0,const char* pattern_include,const char* pattern_exclude)
 *  \brief Retorna una estructura PdsCellString con un arreglo de células, con  
 *  los nombres de los subdirectorios del siguiente nivel de los directorios del 
 *  arreglo de células C0 y excluyendo los simbólicos. 
 *  El arreglo de células de salida no contiene los directorios de entrada dentro de C0.
 *  A partir de aquí pueden ser leidos los datos con la función pds_cell_string_read().
 *  \param[in] C0 Un arreglo de células con los directorios a buscar.
 *  \param[in] pattern_include Patrón de directorios a incluir solamente.
 *  \param[in] pattern_exclude Patrón de directorios a excluir.
 *  \return Retorna una estructura PdsCellString con un arreglo de células con 
 *  los nombres de los subdirectorios del siguiente nivel (solamente un nivel)
 *  y excluyendo los simbólicos O NULL en caso de error.
 *  \ingroup PdsFilesFuncGroup
 */
PdsCellString *pds_get_subdirs_new(const PdsCellString *C0,const char* pattern_include,const char* pattern_exclude);


/** \fn PdsCellString *pds_get_all_files_new(const char *dirpath,   const char* pattern_include,const char* pattern_exclude,const char* file_pattern_include,const char* file_pattern_exclude)
 *  \brief Retorna una estructura PdsCellString con un arreglo de células con los 
 *  nombres de archivos. No son buscados archivos en los directorios simbólicos.
 *  A partir de aquí pueden ser leidos los datos con la función pds_cell_string_read().
 *  \param[in] dirpath Camino del directorio en consulta.
 *  \param[in] pattern_include Patrón de directorios a incluir solamente.
 *  \param[in] pattern_exclude Patrón de directorios a excluir.
 *  \param[in] file_pattern_include Patrón de archivos a incluir solamente.
 *  \param[in] file_pattern_exclude Patrón de archivos a excluir.
 *  \return Retorna una estructura PdsCellString con un arreglo de células con 
 *  los nombres de todos los archivos.
 *  \ingroup PdsFilesFuncGroup
 */
PdsCellString *pds_get_all_files_new(const char *dirpath,   const char* pattern_include,
                                                            const char* pattern_exclude,
                                                            const char* file_pattern_include,
                                                            const char* file_pattern_exclude);


/** \fn char *pds_get_basename(const char * file_path)
 *  \brief Obtén el nombre base desde una ruta de fichero.
 *
 *  Si la ruta de fichero es: /path_to_file/filename.c
 *  La función retorna:       filename.c
 *  \param[in] file_path Ruta del fichero a analizar.
 *  \return Retorna una nueva cadena de texto con el nombre base, o NULL en caso 
 *  de error de reserva de memoria.
 *  \ingroup PdsFilesFuncGroup
 */
char *pds_get_basename(const char * file_path);


/** \fn char *pds_get_filename_extension(const char * file_path)
 *  \brief Obtén la extención de fichero desde una ruta de fichero.
 *  Esta función tendrá problemas con archivos ocultos ".abcd", pues
 *  entenderá "abcd" como una extención de fihero.
 *
 *  Si la ruta de fichero es: /path_to_file/filename.c
 *  La función retorna:       c
 *  \param[in] file_path Ruta del fichero a analizar.
 *  \return Retorna una nueva cadena de texto con la extención de fichero, o 
 *  NULL en caso de error de reserva de memoria o fihero sin extención.
 *  \ingroup PdsFilesFuncGroup
 */
char *pds_get_filename_extension(const char * file_path);


/** \fn char *pds_get_filename(const char * file_path)
 *  \brief Obtén el nombre de fichero desde una ruta de fichero.
 *  Esta función tendrá problemas con archivos ocultos ".abcd", pues
 *  retornará una cadena vacía.
 *
 *  Si la ruta de fichero es: /path_to_file/filename.c
 *  La función retorna:       filename
 *  \param[in] file_path Ruta del fichero a analizar.
 *  \return Retorna una nueva cadena de texto con el nombre de fichero, o NULL 
 *  en caso de error de reserva de memoria.
 *  \ingroup PdsFilesFuncGroup
 */
char *pds_get_filename(const char * file_path);


/** \fn char *pds_get_dirname(const char * file_path)
 *  \brief Obtén el directorio madre desde una ruta de fichero.
 *
 *  Si la ruta de fichero es: /path_to_file/filename.c
 *  La función retorna:       /path_to_file
 *  \param[in] file_path Ruta del fichero a analizar.
 *  \return Retorna una nueva cadena de texto con el directorio madre, o 
 *  NULL en caso de error de reserva de memoria o fihero sin directorio madre.
 *  \ingroup PdsFilesFuncGroup
 */
char *pds_get_dirname(const char * file_path);




/** \fn PdsCellString *pds_get_all_filename_extensions(const char *dirpath,   const char* pattern_include,const char* pattern_exclude,const char* file_pattern_include,const char* file_pattern_exclude)
 *  \brief Retorna una estructura PdsCellString con un arreglo de células con las 
 *  extenciones de archivos en el directorio. No son buscados archivos en los directorios simbólicos.
 *  A partir de aquí pueden ser leidos los datos con la función pds_cell_string_read().
 *  \param[in] dirpath Camino del directorio en consulta.
 *  \param[in] pattern_include Patrón de directorios a incluir solamente.
 *  \param[in] pattern_exclude Patrón de directorios a excluir.
 *  \param[in] file_pattern_include Patrón de archivos a incluir solamente.
 *  \param[in] file_pattern_exclude Patrón de archivos a excluir.
 *  \return Retorna una estructura PdsCellString con un arreglo de células con 
 *  los nombres de todas las extenciones de archivos.
 *  \ingroup PdsFilesFuncGroup
 */
PdsCellString *pds_get_all_filename_extensions(const char *dirpath,   const char* pattern_include,
                                                            const char* pattern_exclude,
                                                            const char* file_pattern_include,
                                                            const char* file_pattern_exclude);


/** \fn char *pds_fgets(FILE *fd)
 *  \brief Retorna una linea leida desde un desriptor de fichero de texto.
 *
 *  \param[in] fd Descriptor de fichero de texto a analizar.
 *  \return Retorna una nueva cadena de texto con una copia de la linea leida 
 *  incluyendo los caracteres de control finales exepto EOF.
 *  \ingroup PdsFilesFuncGroup
 */
char *pds_fgets(FILE *fd);


/** \fn char *pds_read_line(FILE *fd)
 *  \brief Retorna una linea leida desde un desriptor de fichero de texto.
 *
 *  \param[in] fd Descriptor de fichero de texto a analizar.
 *  \return Retorna una nueva cadena de texto con una copia de la linea leida 
 *  descartando los caracteres despues de '\n'.
 *  \ingroup PdsFilesFuncGroup
 */
char *pds_read_line(FILE *fd);


/** \fn int pds_is_file(const char *path)
 *  \brief TRUE si es un fichero (simbólico o real) o FALSE si no.
 *
 *  \param[in] path Dirección de fichero a testar.
 *  \return Retorna TRUE is es un fichero o FALSE si no.
 *  \ingroup PdsFilesFuncGroup
 */
int pds_is_file(const char *path);


/** \fn int pds_is_dir(const char *path)
 *  \brief TRUE is es un directorio (simbólico o real) o FALSE si no.
 *
 *  \param[in] path Dirección de directorio a testar.
 *  \return Retorna TRUE is es un directorio o FALSE si no.
 *  \ingroup PdsFilesFuncGroup
 */
int pds_is_dir(const char *path);


/** \fn long pds_get_file_size(const char *path)
 *  \brief Retorna el número de bytes del archivo.
 *
 *  \param[in] path Dirección de archivo a testar.
 *  \return Retorna el número de bytes del archivo.
 *  \ingroup PdsFilesFuncGroup
 */
long pds_get_file_size(const char *path);


/** \fn int pds_exist_file(const char *path)
 *  \brief Retorna TRUE si path es un fichero real o simbólico (no un directorio) 
 *  y es accesible.
 *
 *  \param[in] path Dirección de fichero a testar.
 *  \return Retorna TRUE si el archivo existe o FALSE si no.
 *  \ingroup PdsFilesFuncGroup
 */
int pds_exist_file(const char *path);


/** \fn int pds_archive_is_accessible(const char *path)
 *  \brief Retorna TRUE si el archivo simbólico o real (Directorio o fichero) 
 *  es accesible.
 *
 *  \param[in] path Dirección de archivo a testar.
 *  \return Retorna TRUE si el archivo existe o FALSE si no.
 *  \ingroup PdsFilesFuncGroup
 */
int pds_archive_is_accessible(const char *path);

/**
 * @}
 */

#ifdef __cplusplus
}
#endif 

#endif	/* __PDSFILESFUNC_H__ */ 

