/*
 * pdsfilesfunc.h
 * 
 * Copyright 2011 Fernando Pujaico Rivera <fernando.pujaico.rivera@gmail.com>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 * 
 */

/** \file pdsfilesfunc.h
 *  \author Fernando Pujaico Rivera
 *  \date 30-07-2016
 *  \brief Funciones que manipulan archivos y caminos.
 *   
 */

#ifndef __PDSFILESFUNC_H__
#define __PDSFILESFUNC_H__


#ifdef __cplusplus
extern "C" {
#endif 

#include <stdio.h>
#include <stdio.h>
#include <pds/pdscstring.h>

#ifndef TRUE
	#define TRUE 1
#endif

#ifndef FALSE
	#define FALSE 0
#endif

/** \defgroup PdsFilesFuncGroup Funciones del módulo PdsFilesFunc.
 *
 *  <br>
 *  Funciones.
 * @{
 */


/*! \enum ArqType
 *  \brief Una union de tipo ArqType.
 *  Esta union tiene dos tipos de datos tipo directorio y tipo documento.
 *  <br>
 *
 *  \ingroup PdsFilesFuncGroup
 *  \author Fernando Pujaico Rivera
 */
typedef enum{
    TYPE_FILE, /**< Buscando ficheros excluyendo los simbólicos*/
    TYPE_DIR,  /**< Buscando directorios excluyendo los simbólicos*/
    TYPE_DIRTOP, /**< Buscando directorios excluyendo los simbólicos incluyendo la raiz*/
}ArqType;




/** @name pds_get_<one_level_search_methods>
 *  Funciones con buquedas archivos en un solo subdirectorio.
 *
 * @{
 */

/** \fn PdsCellString *pds_get_subdirs_new(const PdsCellString *C0,const char* pattern_include,const char* pattern_exclude)
 *  \brief Retorna una estructura PdsCellString  con los directorios (solo un nivel), 
 *  de cada directorio espesificado en C0 (excluyendo los simbólicos). 
 *
 *  El arreglo de células retornado, no contiene los directorios de entrada entregados por C0.
 *  A partir de aquí pueden ser leidos los datos con la función pds_cell_string_read().
 *  \param[in] C0 Un arreglo de células con los directorios a buscar.
 *  \param[in] pattern_include Patrón de directorios a incluir solamente.
 *  \param[in] pattern_exclude Patrón de directorios a excluir.
 *  \return Retorna una estructura PdsCellString con un arreglo de células con 
 *  los nombres de los subdirectorios del siguiente nivel (solamente un nivel)
 *  y excluyendo los simbólicos O NULL en caso de error.
 *  \ingroup PdsFilesFuncGroup
 */
PdsCellString *pds_get_subdirs_new(const PdsCellString *C0,const char* pattern_include,const char* pattern_exclude);



/** \fn PdsCellString *pds_get_arquives_new(const char *dirpath,ArqType Type,const char* pattern_include,const char* pattern_exclude)
 *  \brief Retorna los archivos dentro de un directorio (siguiendo algunos criterios).
 *
 *  Retorna una estructura PdsCellString con un arreglo de células de nombres de arquivo.
 *  A partir de aqui pueden ser leidos los siguientes con la función pds_cell_string_read().
 *  \param[in] dirpath Camino del directorio en consulta.
 *  \param[in] Type El tipo de arquivo a leer, pueden ser usados los valores :
 <ul>
    <li><b>TYPE_DIR</b> para directorios, </li>
    <li><b>TYPE_DIRTOP</b> para directorios incluyendo la raiz o </li>
    <li><b>TYPE_FILE</b>  para fiheros.</li> 
    <li>Cualquier otro valor es equivalente a usar TYPE_FILE.</li>
 </ul>
 *  \param[in] pattern_include Patrón de archivos a incluir solamente.
 *  \param[in] pattern_exclude Patrón de archivos a excluir.
 *  \return Retorna una estructura PdsCellString con un arreglo de células con 
 *  nombres de arquivo.
 *  \ingroup PdsFilesFuncGroup
 */
PdsCellString *pds_get_arquives_new(const char *dirpath,ArqType Type,const char* pattern_include,const char* pattern_exclude);

//@}

/** @name pds_get_filedata_<methods>
 *  Funciones con buquedas de datos en ficheros.
 *
 * @{
 */

/** \fn char *pds_get_filedata_string(const char *pathname,const char* pattern)
 *  \brief La funcion analiza linea por linea el archivo y retorna
 *  el texto despues de el primer patrón pattern encontrado.
 *
 *  \param[in] pathname Fichero de texto a analizar.
 *  \param[in] pattern Patrón a buscar.
 *  \return Retorna una nueva cadena de texto con el dato encontrado,
 *  una cadena vacia si no existe ningun dato o NULL
 *  si se encontró algun error.
 *  \ingroup PdsFilesFuncGroup
 */
char *pds_get_filedata_string(const char *pathname,const char* pattern);


/** \fn PdsCellString *pds_get_filedata_cell_string(const char *pathname,const char* pattern,const char *separator)
 *  \brief La funcion analiza linea por linea el archivo y retorna
 *  el texto despues de el primer patrón pattern encontrado, Pero estos datos son
 *  separados usando algunos de los separadores usados en separator y ordenandos en una PdsCellString.
 *
 *  \param[in] pathname Fichero de texto a analizar.
 *  \param[in] pattern Patrón a buscar.
 *  \param[in] separator Separador de texto. Si separator es igual a NULL o vacia la 
 *  función retorna la cadena sem dividir.
 *  \return Retorna una nueva estructurs PdsCellString con los datos encontrado.
 *  Si no se encontró nada retorna una estructura vacia o NULL
 *  si se encontró algun error.
 *  \ingroup PdsFilesFuncGroup
 */
PdsCellString *pds_get_filedata_cell_string(const char *pathname,const char* pattern,const char *separator);

//@}

/** @name pds_get_all_<methods>
 *  Funciones con buquedas recursivas.
 *
 * @{
 */



/** \fn PdsCellString *pds_get_all_subdirs_new(const char *dirpath,const char* pattern_include,const char* pattern_exclude)
 *  \brief Retorna una estructura PdsCellString con un arreglo de células con 
 *  nombres de subdirectorios incluyendo la raiz y excluyendo los simbólicos.
 *  A partir de aquí pueden ser leidos los datos con la función pds_cell_string_read().
 *  \param[in] dirpath Camino del directorio en consulta.
 *  \param[in] pattern_include Patrón de directorios a incluir solamente.
 *  \param[in] pattern_exclude Patrón de directorios a excluir.
 *  \return Retorna una estructura PdsCellString con un arreglo de células com 
 *  nombres de subdirectorios incluyendo la raiz y excluyendo los simbólicos.
 *  \ingroup PdsFilesFuncGroup
 */
PdsCellString *pds_get_all_subdirs_new(const char *dirpath,const char* pattern_include,const char* pattern_exclude);



/** \fn PdsCellString *pds_get_all_files_new(const char *dirpath,   const char* pattern_include,const char* pattern_exclude,const char* file_pattern_include,const char* file_pattern_exclude)
 *  \brief Retorna una estructura PdsCellString con un arreglo de células con los 
 *  nombres de archivos. No son buscados archivos en los directorios simbólicos.
 *  A partir de aquí pueden ser leidos los datos con la función pds_cell_string_read().
 *  \param[in] dirpath Camino del directorio en consulta.
 *  \param[in] pattern_include Patrón de directorios a incluir solamente.
 *  \param[in] pattern_exclude Patrón de directorios a excluir.
 *  \param[in] file_pattern_include Patrón de archivos a incluir solamente.
 *  \param[in] file_pattern_exclude Patrón de archivos a excluir.
 *  \return Retorna una estructura PdsCellString con un arreglo de células con 
 *  los nombres de todos los archivos.
 *  \ingroup PdsFilesFuncGroup
 */
PdsCellString *pds_get_all_files_new(const char *dirpath,   const char* pattern_include,
                                                            const char* pattern_exclude,
                                                            const char* file_pattern_include,
                                                            const char* file_pattern_exclude);


/** \fn PdsCellString *pds_get_all_filename_extensions(const char *dirpath,   const char* pattern_include,const char* pattern_exclude,const char* file_pattern_include,const char* file_pattern_exclude)
 *  \brief Retorna una estructura PdsCellString con un arreglo de células con las 
 *  extenciones de archivos en el directorio. No son buscados archivos en los directorios simbólicos.
 *  A partir de aquí pueden ser leidos los datos con la función pds_cell_string_read().
 *  \param[in] dirpath Camino del directorio en consulta.
 *  \param[in] pattern_include Patrón de directorios a incluir solamente.
 *  \param[in] pattern_exclude Patrón de directorios a excluir.
 *  \param[in] file_pattern_include Patrón de archivos a incluir solamente.
 *  \param[in] file_pattern_exclude Patrón de archivos a excluir.
 *  \return Retorna una estructura PdsCellString con un arreglo de células con 
 *  los nombres de todas las extenciones de archivos.
 *  \ingroup PdsFilesFuncGroup
 */
PdsCellString *pds_get_all_filename_extensions(const char *dirpath,   const char* pattern_include,
                                                            const char* pattern_exclude,
                                                            const char* file_pattern_include,
                                                            const char* file_pattern_exclude);

//@}


/** @name pds_<filepath_methods>
 *  Funciones para manipular un fillepath
 *
 * @{
 */


/** \fn char *pds_get_realpath(const char* filepath)
 *  \brief Expande todos los enlaces simbolicos y resuleve referencia a /./,
 *  /../ y caracteres extras '/'  en filepath, asi es producido un pathname 
 *  canonizado y absoluto. El resultado es cargado en una cadena nueva, 
 *  retornada por la funcion. El path resultante no tiene enlaces simbolicos, 
 *  /./ o /../ componentes. No expande '~/'.
 *
 *  Si la ruta de fichero es: ./Makefile
 *                            src/../Makefile
 *                            src/../////Makefile
 *                            Makefile 
 *  La función retorna:       /path_of_file/Makefile
 *  \param[in] filepath Ruta del fichero a analizar. Este fichero tiene que existir,
 *  caso contrario la función retornará NULL.
 *  \return Retorna una nueva cadena de texto con el path absoluto y canonizado, 
 *  o NULL en caso de error de reserva de memoria.
 *  \ingroup PdsFilesFuncGroup
 */
char *pds_get_realpath(const char* filepath);

/** \fn char *pds_get_homedir(void)
 *  \brief Retorna una nueva cadena de texto con la dirección del directorio de 
 *  usuario. HOME en gnu-linux y {HOMEDRIVE,HOMEPATH} en Windows.
 *  \return Retorna el directorio de usuario o NULL en caso de problemas en la
 *  reserva de memoria.
 *  \ingroup PdsFilesFuncGroup
 */
char *pds_get_homedir(void);

/** \fn char pds_filesep(void)
 *  \brief Retorna um caracter con el separador de archivo para el sistema.
 *  \return Retorna um caracter con el separador de archivo para el sistema.
 *  En caso de error de dedución, retorna '/' por defecto.
 *  \ingroup PdsFilesFuncGroup
 */
char pds_filesep(void);


/** \fn char *pds_get_basename(const char * file_path)
 *  \brief Obtén el nombre base desde una ruta de fichero.
 *
 *  Si la ruta de fichero es: /path_to_file/filename.c
 *  La función retorna:       filename.c
 *  \param[in] file_path Ruta del fichero a analizar.
 *  \return Retorna una nueva cadena de texto con el nombre base, o NULL en caso 
 *  de error de reserva de memoria.
 *  \ingroup PdsFilesFuncGroup
 */
char *pds_get_basename(const char * file_path);


/** \fn char *pds_get_filename_extension(const char * file_path)
 *  \brief Obtén la extención de fichero desde una ruta de fichero.
 *  Esta función tendrá problemas con archivos ocultos ".abcd", pues
 *  entenderá "abcd" como una extención de fihero.
 *
 *  Si la ruta de fichero es: /path_to_file/filename.c
 *  La función retorna:       c
 *  \param[in] file_path Ruta del fichero a analizar.
 *  \return Retorna una nueva cadena de texto con la extención de fichero, o 
 *  NULL en caso de error de reserva de memoria o fihero sin extención.
 *  \ingroup PdsFilesFuncGroup
 */
char *pds_get_filename_extension(const char * file_path);


/** \fn char *pds_get_filename(const char * file_path)
 *  \brief Obtén el nombre de fichero desde una ruta de fichero.
 *  Esta función tendrá problemas con archivos ocultos ".abcd", pues
 *  retornará una cadena vacía.
 *
 *  Si la ruta de fichero es: /path_to_file/filename.c
 *  La función retorna:       filename
 *  \param[in] file_path Ruta del fichero a analizar.
 *  \return Retorna una nueva cadena de texto con el nombre de fichero, o NULL 
 *  en caso de error de reserva de memoria.
 *  \ingroup PdsFilesFuncGroup
 */
char *pds_get_filename(const char * file_path);


/** \fn char *pds_get_dirname(const char * file_path)
 *  \brief Obtén el directorio madre desde una ruta de fichero.
 *
 *  Si la ruta de fichero es: /path_to_file/filename.c
 *  La función retorna:       /path_to_file
 *  \param[in] file_path Ruta del fichero a analizar.
 *  \return Retorna una nueva cadena de texto con el directorio madre, o 
 *  NULL en caso de error de reserva de memoria o fihero sin directorio madre.
 *  \ingroup PdsFilesFuncGroup
 */
char *pds_get_dirname(const char * file_path);


/** \fn char *pds_get_filepath_with_extension(const char *file_path,const char *extension)
 *  \brief Obtén un filepath con el mismo filename pero con la extención cambiada 
 *  a extension.
 *
 *  Si la ruta de fichero es: /path_to_file/filename.c
 *  La función retorna:       /path_to_file/filename.[extension]
 *  \param[in] file_path Ruta del fichero a analizar.
 *  \param[in] extension Extención a usar.
 *  \return Retorna una nueva cadena de texto con el filepath nuevo, o 
 *  NULL en caso de error de reserva de memoria.
 *  \ingroup PdsFilesFuncGroup
 */
char *pds_get_filepath_with_extension(const char *file_path,const char *extension);


/** \fn char *pds_get_temporal_filepath(const char *filepath,const char *groupname)
 *  \brief Obtén un nombre de archivo con la misma extención y directorio pero con nombre de
 *  archivo diferente y aleatoriamente diseñado.
 *
 *  Si la ruta de fichero es: /path_to_file/filename.c
 *  La función retorna:       /path_to_file/filename_[groupname][clock()][rand()].c
 *  
 *  \param[in] filepath Ruta del fichero a analizar. Esta ruta no precisa existir.
 *  \param[in] groupname nombre que se agregará al nombre de archivo retornado.
 *  \return Retorna una nueva cadena de texto con el filepath nuevo, o 
 *  NULL en caso de error de reserva de memoria.
 *  \ingroup PdsFilesFuncGroup
 */
char *pds_get_temporal_filepath(const char *filepath,const char *groupname);


/** \fn char* pds_get_absolute_dirname(void)
 *  \brief Obtén el directorio base del programa que invoca esta función.
 *
 *  Si la ruta del programa es: /path_to_file/programfilename
 *  La función retorna:       /path_to_file
 *  \return Retorna una nueva cadena de texto con el nombre del directorio base, 
 *  o NULL en caso de error de reserva de memoria.
 *  \ingroup PdsFilesFuncGroup
 */
char* pds_get_absolute_dirname(void);

/** \fn char* pds_get_absolute_programpath(void)
 *  \brief Obtén la ruta del programa que invoca esta función.
 *
 *  Si la ruta del programa es: /path_to_file/programfilename
 *  La función retorna:       /path_to_file/programfilename
 *  \return Retorna una nueva cadena de texto con el nombre de la ruta base, 
 *  o NULL en caso de error de reserva de memoria.
 *  \ingroup PdsFilesFuncGroup
 */
char* pds_get_absolute_programpath(void);

//@}



/** @name pds_<file_descriptor_methods>
 *  Funciones que usan um file descriptor para trabajar cor archivos.
 *
 * @{
 */

/** \fn char *pds_fgets(FILE *fd)
 *  \brief Retorna una linea leida desde un desriptor de fichero de texto.
 *
 *  \param[in] fd Descriptor de fichero de texto a analizar.
 *  \return Retorna una nueva cadena de texto con una copia de la linea leida 
 *  incluyendo los caracteres de control finales exepto EOF.
 *  \ingroup PdsFilesFuncGroup
 */
char *pds_fgets(FILE *fd);


/** \fn char *pds_read_line(FILE *fd)
 *  \brief Retorna una linea leida desde un desriptor de fichero de texto.
 *
 *  \param[in] fd Descriptor de fichero de texto a analizar.
 *  \return Retorna una nueva cadena de texto con una copia de la linea leida 
 *  descartando salto de linea '\\n'.
 *  \ingroup PdsFilesFuncGroup
 */
char *pds_read_line(FILE *fd);

//@}


/** @name pds_<file_details_methods>
 *  Funciones que muestran caracteristicas de los archivos.
 *
 * @{
 */

/** \fn int pds_is_file(const char *path)
 *  \brief TRUE si es un fichero (simbólico o real) o FALSE si no.
 *
 *  \param[in] path Dirección de fichero a testar.
 *  \return Retorna TRUE is es un fichero o FALSE si no.
 *  \ingroup PdsFilesFuncGroup
 */
int pds_is_file(const char *path);


/** \fn int pds_is_dir(const char *path)
 *  \brief TRUE is es un directorio (simbólico o real) o FALSE si no.
 *
 *  \param[in] path Dirección de directorio a testar.
 *  \return Retorna TRUE is es un directorio o FALSE si no.
 *  \ingroup PdsFilesFuncGroup
 */
int pds_is_dir(const char *path);


/** \fn long pds_get_file_size(const char *path)
 *  \brief Retorna el número de bytes del archivo.
 *
 *  \param[in] path Dirección de archivo a testar.
 *  \return Retorna el número de bytes del archivo.
 *  \ingroup PdsFilesFuncGroup
 */
long pds_get_file_size(const char *path);

/** \fn long pds_get_number_of_chars(const char *path)
 *  \brief Retorna el número de caracteres de un archivo.
 *
 *  \param[in] path Dirección de archivo a testar.
 *  \return Retorna el número de caracteres de un archivo
 *  o negativo en caso de error
 *  \ingroup PdsFilesFuncGroup
 */
long pds_get_number_of_chars(const char *path);


/** \fn int pds_exist_file(const char *path)
 *  \brief Retorna TRUE si path es un fichero real o simbólico (no un directorio) 
 *  y es accesible.
 *
 *  \param[in] path Dirección de fichero a testar.
 *  \return Retorna TRUE si el archivo existe o FALSE si no.
 *  \ingroup PdsFilesFuncGroup
 */
int pds_exist_file(const char *path);


/** \fn int pds_archive_is_accessible(const char *path)
 *  \brief Retorna TRUE si el archivo simbólico o real (Directorio o fichero) 
 *  es accesible.
 *
 *  \param[in] path Dirección de archivo a testar.
 *  \return Retorna TRUE si el archivo existe o FALSE si no.
 *  \ingroup PdsFilesFuncGroup
 */
int pds_archive_is_accessible(const char *path);

//@}

/** @name pds_<replace_methods>
 *  Funciones para remplazar texto en archivos.
 *
 * @{
 */



/** \fn int pds_file_pattern_replacement(const char *filename_out,const char *filename_in,const char *pat,const char *rep)
 *  \brief Esta función busca en un archivo de texto filename_in, un patron pat y los 
 *  remplaza con el contenido de la cadena rep, el resultado es retornado en un 
 *  archivo filename_out.
 *
 *  Cuando encuentra una coincidencia en la posición p0 del archivo, continua la
 *  siguiente busqueda en p0+strlen(pat).
 *  \param[in] filename_out Archivo de salida donde se retornará el resultado del remplazo.
 *  \param[in] filename_in Archivo de entrada donde se realizará la búsqueda.
 *  \param[in] pat Patron a buscar.
 *  \param[in] rep Patron a de remplazo.
 *  \return Retorna TRUE si todo fue bien o FALSE si no.
 *  \ingroup PdsFilesFuncGroup
 */
int pds_file_pattern_replacement(const char *filename_out,const char *filename_in,const char *pat,const char *rep);

/** \fn int pds_file_content_replacement(   const char *filename_out,const char *filename_in,const char *pat_open,const char *pat_close,char *(*func)(const char *))
 *  \brief Esta función busca en un archivo de texto filename_in, el contenido
 *  entre un patrón pat_open y pat_close y lo remplaza  por el valor devuelto de 
 *  la función func, el resultado de todo esto es escrito en un archivo filename_out.
 *
 *  La función func debe ser de la forma :
\code{.c}
char *func(const char* content);
\endcode
 *
\code{.c}
pds_file_content_replacement("output.html","input.html","<b>","</b>",func);
\endcode
 *  \param[in] filename_out Archivo de salida donde se retornará el resultado del remplazo.
 *  \param[in] filename_in Archivo de entrada donde se realizará la búsqueda.
 *  \param[in] pat_open Patrón inicial a buscar.
 *  \param[in] pat_close Patrón final a buscar.
 *  \param[in] func Es una función que se usará para remplazar el contenido 
 *  encontrado. Por ejemplo si entre pat_open y pat_close es encontrado content
\code{.c}
replace=func(content);
\endcode
 *  este será remplazado por el contenido de la cadena replace, luego la cadena 
 *  replace será liberada internamente con free(replace).
 *  \return Retorna TRUE si todo fue bien o FALSE si no.
 *  \ingroup PdsFilesFuncGroup
 */
int pds_file_content_replacement(   const char *filename_out,const char *filename_in,
                                    const char *pat_open,const char *pat_close,
                                    char *(*func)(const char *));


//@}

/**
 * @}
 */

#ifdef __cplusplus
}
#endif 

#endif	/* __PDSFILESFUNC_H__ */ 

