/*
 * pdsargsfunc.c
 * 
 * Copyright 2011 Fernando Pujaico Rivera <fernando.pujaico.rivera@gmail.com>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 * 
 */

#include <stdlib.h>
#include <string.h>
#include <pds/pdsargsfunc.h>
#include <pds/pdsstring.h>


/** \fn int pds_get_int_param(int argc,char* const argv[],const char *data,int *param)
 *  \brief Obtiene el parámetro param de tipo int que va después del 
 *  parámetro data en una cadena de parámetros argv de argc elementos. si el 
 *  parámetro data o param no es encontrado la función retorna PDS_WRONG, y simplemente 
 *  no modifica param.
 *  Si existen parámetros data repetidos, solo se atiende al primero.
 *  \param[in] argc Cantidad de elementos de argv.
 *  \param[in] argv Cadenas de caracteres con  parámetros de entrada.
 *  \param[in] data Dato a buscar en las cadenas argv.
 *  \param[out] param Parámetro existente después de data.
 *  \return PDS_OK si todo fue bien o PDS_WRONG si no, ejemplo argv==NULL ó data==NULL.
 *  si el parámetro no es encontrado o no es un entero da PDS_WRONG.
 *  \ingroup PdsArgsFuncGroup
 */
int pds_get_int_param(int argc,char* const argv[],const char *data,int *param)
{
	int i;
	if((argv==NULL)||(data==NULL))	return PDS_WRONG;
	for(i=0;i<(argc-1);i++)
	{
		if(strcmp(argv[i],data)==0)
		{
            if(isintnumber(argv[i+1]))
            {
			    *param=atoi(argv[i+1]);
			    return PDS_OK;
            }
            else
                return PDS_WRONG;
		}
	}
	return PDS_WRONG;
}


/** \fn int pds_get_int_param_with_default(int argc,char* const argv[],const char *data,int *param,int default_val)
 *  \brief Obtiene el parámetro param de tipo int que va después del 
 *  parámetro data en una cadena de parámetros argv de argc elementos. Si el 
 *  parámetro data o param no es encontrado entonces simplemente carga en param 
 *  el valor por defecto y PDS_OK es retornado.
 *  Si existen parámetros data repetidos, solo se atiende al primero.
 *  \param[in] argc Cantidad de elementos de argv.
 *  \param[in] argv Cadenas de caracteres con  parámetros de entrada.
 *  \param[in] data Dato a buscar en las cadenas argv.
 *  \param[out] param Parámetro existente después de data.
 *  \param[in] default_val Valor por defecto.
 *  \return PDS_OK si todo fue bien o PDS_WRONG si no, ejemplo argv==NULL ó data==NULL.
 *  Si el parámetro por defecto es usado entonces es retornado PDS_OK.
 *  \ingroup PdsArgsFuncGroup
 */
int pds_get_int_param_with_default(int argc,char* const argv[],const char *data,int *param,int default_val)
{
	int id;
	if((argv==NULL)||(data==NULL))	return PDS_WRONG;

	id=pds_get_int_param(argc,argv,data,param);
    if(id==PDS_WRONG)   *param=default_val;

	return PDS_OK;
}


/** \fn int pds_get_llint_param(int argc,char* const argv[],const char *data,long int *param)
 *  \brief Obtiene el parámetro param de tipo long int que va después del 
 *  parámetro data en una cadena de parámetros argv de argc elementos. si el 
 *  parámetro data o param no es encontrado la función retorna PDS_WRONG, y simplemente 
 *  no modifica param.
 *  Si existen parámetros data repetidos, solo se atiende al primero.
 *  \param[in] argc Cantidad de elementos de argv.
 *  \param[in] argv Cadenas de caracteres con  parámetros de entrada.
 *  \param[in] data Dato a buscar en las cadenas argv.
 *  \param[out] param Parámetro existente después de data.
 *  \return PDS_OK si todo fue bien o PDS_WRONG si no, ejemplo argv==NULL ó data==NULL.
 *  si el parámetro no es encontrado  o no es un entero da PDS_WRONG.
 *  \ingroup PdsArgsFuncGroup
 */
int pds_get_llint_param(int argc,char* const argv[],const char *data,long long *param)
{
	int i;
	if((argv==NULL)||(data==NULL))	return PDS_WRONG;
	for(i=0;i<(argc-1);i++)
	{
		if(strcmp(argv[i],data)==0)
		{
            if(isintnumber(argv[i+1]))
            {
			    *param=atoll(argv[i+1]);
			    return PDS_OK;
            }
            else
                return PDS_WRONG;
		}
	}
	return PDS_WRONG;
}


/** \fn int pds_get_float_param(int argc,char* const argv[],const char *data,float *param)
 *  \brief Obtiene el parámetro param de tipo float que va después del 
 *  parámetro data en una cadena de parámetros argv de argc elementos. si el 
 *  parámetro data o param no es encontrado la función retorna PDS_WRONG, y simplemente 
 *  no modifica param.
 *  Si existen parámetros data repetidos, solo se atiende al primero.
 *  \param[in] argc Cantidad de elementos de argv.
 *  \param[in] argv Cadenas de caracteres con  parámetros de entrada.
 *  \param[in] data Dato a buscar en las cadenas argv.
 *  \param[out] param Parámetro existente después de data.
 *  \return PDS_OK si todo fue bien o PDS_WRONG si no, ejemplo argv==NULL ó data==NULL.
 *  si el parámetro no es encontrado da PDS_WRONG.
 *  \ingroup PdsArgsFuncGroup
 */
int pds_get_float_param(int argc,char* const argv[],const char *data,float *param)
{
	int i;
	if((argv==NULL)||(data==NULL))	return PDS_WRONG;
	for(i=0;i<(argc-1);i++)
	{
		if(strcmp(argv[i],data)==0)
		{
			*param=atof(argv[i+1]);
			return PDS_OK;
		}
	}
	return PDS_WRONG;
}


/** \fn int pds_get_double_param(int argc,char* const argv[],const char *data,double *param)
 *  \brief Obtiene el parámetro param de tipo double que va después del 
 *  parámetro data en una cadena de parámetros argv de argc elementos. si el 
 *  parámetro data o param no es encontrado la función retorna PDS_WRONG, y simplemente 
 *  no modifica param.
 *  Si existen parámetros data repetidos, solo se atiende al primero.
 *  \param[in] argc Cantidad de elementos de argv.
 *  \param[in] argv Cadenas de caracteres con  parámetros de entrada.
 *  \param[in] data Dato a buscar en las cadenas argv.
 *  \param[out] param Parámetro existente después de data.
 *  \return PDS_OK si todo fue bien o PDS_WRONG si no, ejemplo argv==NULL ó data==NULL.
 *  si el parámetro no es encontrado da PDS_WRONG.
 *  \ingroup PdsArgsFuncGroup
 */
int pds_get_double_param(int argc,char* const argv[],const char *data,double *param)
{
	int i;
	if((argv==NULL)||(data==NULL))	return PDS_WRONG;
	for(i=0;i<(argc-1);i++)
	{
		if(strcmp(argv[i],data)==0)
		{
			*param=atof(argv[i+1]);
			return PDS_OK;
		}
	}
	return PDS_WRONG;
}


/** \fn int pds_get_chars_param(int argc,char* const argv[],const char *data,char **param)
 *  \brief Obtiene la dirección del parámetro param de tipo char * que va después del 
 *  parámetro data en una cadena de parámetros argv de argc elementos. si el 
 *  parámetro data o param no es encontrado la función retorna PDS_WRONG, y simplemente 
 *  no modifica param. 
 *  Si existen parámetros data repetidos, solo se atiende al primero.
 *  \param[in] argc Cantidad de elementos de argv.
 *  \param[in] argv Cadenas de caracteres con  parámetros de entrada.
 *  \param[in] data Dato a buscar en las cadenas argv.
 *  \param[out] param Dirección del parámetro existente después de data. 
 *  la memoria de este puntero nunca debe ser liberada.
 *  \return PDS_OK si todo fue bien o PDS_WRONG si no, ejemplo argv==NULL ó data==NULL.
 *  si el parámetro no es encontrado da PDS_WRONG.
 *  \ingroup PdsArgsFuncGroup
 */
int pds_get_chars_param(int argc,char* const argv[],const char *data,char **param)
{
	int i;
	if((argv==NULL)||(data==NULL))	return PDS_WRONG;
	for(i=0;i<(argc-1);i++)
	{
		if(strcmp(argv[i],data)==0)
		{
			*param=argv[i+1];
			return PDS_OK;
		}
	}
	return PDS_WRONG;
}

 

/** \fn int pds_exist_param(int argc,char* const argv[],const char *data)
 *  \brief Retorna PDS_OK si existe el parámetro param.
 *  \param[in] argc Cantidad de elementos de argv.
 *  \param[in] argv Cadenas de caracteres con  parámetros de entrada.
 *  \param[in] data Dato a buscar en las cadenas argv.
 *  \return PDS_OK si todo fue bien o PDS_WRONG si no, ejemplo argv==NULL ó data==NULL.
 *  si el parámetro no es encontrado da PDS_WRONG.
 *  \ingroup PdsArgsFuncGroup
 */
int pds_exist_param(int argc,char* const argv[],const char *data)
{
	int i;
	if((argv==NULL)||(data==NULL))	return PDS_WRONG;
	for(i=0;i<argc;i++)
	{
		if(strcmp(argv[i],data)==0)	return PDS_OK;
	}
	return PDS_WRONG;
}


