/*
 * pdslstring.c
 * 
 * Copyright 2011 Fernando Pujaico Rivera <fernando.pujaico.rivera@gmail.com>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 * 
 */

#include <stdlib.h>
#include <string.h>
#include <pds/pdslstring.h>


/** \fn PdsListString *pds_list_string_new(void)
 *  \brief Crea una lista de tipo PdsListString vacia.
 *  \return Un puntero a un elemento de tipo PdsListString.
 *  \ingroup PdsListStringGroup
 */
PdsListString *pds_list_string_new(void)
{
	PdsListString *L=NULL;
	L=(PdsListString *)calloc(1,sizeof(PdsListString));
	if(L==NULL)	return NULL;
	
	L->AddPrev=NULL;
	L->S=NULL;
	L->AddNext=NULL;

	return L;
}

/** \fn PdsListString *pds_list_string_new_with_string(const char* data)
 *  \brief Crea una lista de tipo PdsListString con el dato data.
 *  \return Un puntero a un elemento de tipo PdsListString.
 *  \ingroup PdsListStringGroup
 */
PdsListString *pds_list_string_new_with_string(const char* data)
{
	PdsListString *L=NULL;
	L=(PdsListString *)calloc(1,sizeof(PdsListString));
	if(L==NULL)	return NULL;
	
	L->AddPrev=NULL;
	L->AddNext=NULL;

	L->S=(char*)calloc(strlen(data)+1,sizeof(char));
	if(L->S==NULL)
    {   
        free(L);
	    return NULL;
    }

    strcpy(L->S,data);

	return L;
}


/** \fn int pds_list_string_push(PdsListString **Lnode,const char* string_data)
 *  \brief Agrega un elemento a la cima de la lista. O al nodo actual si el 
 *  string es null. Si un nuevo nodo es agregado Lnode apuntara a este nodo.
 *  Si no le das el último nodo de la lista da error no hace nada y retorna PDS_WRONG.
 *  \param[in,out] Lnode El nodo de la cima de la lista.
 *  \param[in] string_data La cadena a escribir.
 *  \return PDS_OK si todo fue bien o PDS_WRONG si no. Si no le das el último nodo da error.
 *  \ingroup PdsListStringGroup
 */
int pds_list_string_push(PdsListString **Lnode,const char* string_data)
{
	PdsListString *L=NULL;

	if((*Lnode)==NULL)		return PDS_WRONG;
	if((*Lnode)->AddNext!=NULL)	return PDS_WRONG;

	if((*Lnode)->S==NULL)
	{
		(*Lnode)->S=(char*)calloc(sizeof(char),strlen(string_data)+1);
        sprintf((*Lnode)->S,"%s",string_data);
		return PDS_OK;
	}
	else
	{
		L=(PdsListString *)calloc(1,sizeof(PdsListString));
		if(L==NULL)	return PDS_WRONG;
	
		L->AddPrev=(struct PdsListString *)(*Lnode);
		L->S=(char*)calloc(sizeof(char),strlen(string_data)+1);
		if(L->S==NULL)
		{
			free(L);
			return PDS_WRONG;
		}
        sprintf(L->S,"%s",string_data);

		L->AddNext=NULL;

		(*Lnode)->AddNext=(struct PdsListString *)L;

		(*Lnode)=L;
	
		return PDS_OK;
	}
}


/** \fn int pds_list_string_pop(PdsListString **Lnode)
 *  \brief Quita un nodo al final de la lista. Si el nodo es removido, Lnode
 *  apuntará al nuevo nodo en la cima.
 *  Si la lista tiene un solo nodo, este es limpiado y Lnode no cambia de dirección.
 *  Si no le das el último nodo de la lista da error, no hace nada y retorna PDS_WRONG.
 *  \param[in,out] Lnode El nodo de la cima de la lista.
 *  \return PDS_OK si todo fue bien o PDS_WRONG si no. Si no le das el último nodo da error.
 *  \ingroup PdsListStringGroup
 */
int pds_list_string_pop(PdsListString **Lnode)
{
	PdsListString *Lprev=NULL;

	if((*Lnode)==NULL)		    return PDS_WRONG;
	if((*Lnode)->AddNext!=NULL)	return PDS_WRONG;
	
	if((*Lnode)->AddPrev==NULL)
	{
		if((*Lnode)->S==NULL)
		{
			return PDS_OK;
		}
		else
		{
			free((*Lnode)->S);
			(*Lnode)->S=NULL;
			return PDS_OK;
		}
	}
	else
	{
		Lprev=(PdsListString *)((*Lnode)->AddPrev);
	
		free((*Lnode)->S);
		(*Lnode)->S=NULL;
		free((*Lnode));

		(*Lnode)=Lprev;
		(*Lnode)->AddNext=NULL;
		return PDS_OK;
	}
}


/** \fn int pds_list_string_shift(PdsListString **Lnode)
 *  \brief Quita un nodo al inicio de la lista. Si el nodo es removido, Lnode
 *  apuntará al nuevo nodo inicial.
 *  Si la lista tiene un solo nodo, este es limpiado y Lnode no cambia de dirección.
 *  Si no le das el último nodo de la lista da error, no hace nada y retorna PDS_WRONG.
 *  \param[in,out] Lnode El nodo al inicio de la lista.
 *  \return PDS_OK si todo fue bien o PDS_WRONG si no. Si no le das el primer nodo da error.
 *  \ingroup PdsListStringGroup
 */
int pds_list_string_shift(PdsListString **Lnode)
{
	PdsListString *Lnext=NULL;

	if((*Lnode)==NULL)		    return PDS_WRONG;
	if((*Lnode)->AddPrev!=NULL)	return PDS_WRONG;
	
	if((*Lnode)->AddNext==NULL)
	{
		if((*Lnode)->S==NULL)
		{
			return PDS_OK;
		}
		else
		{
			free((*Lnode)->S);
			(*Lnode)->S=NULL;
			return PDS_OK;
		}
	}
	else
	{
		Lnext=(PdsListString *)((*Lnode)->AddNext);
	
		free((*Lnode)->S);
		(*Lnode)->S=NULL;
		free((*Lnode));

		(*Lnode)=Lnext;
		(*Lnode)->AddPrev=NULL;
		return PDS_OK;
	}
}


/** \fn int pds_list_string_unshift(PdsListString **Lnode,const char *string_data)
 *  \brief Agrega un elemento al inicio de la lista. O al nodo actual si el 
 *  string es null. Si un nuevo nodo es agregado Lnode apuntara a este nodo.
 *  Si no le das el primer nodo de la lista da error y no hace nada.
 *  \param[in,out] Lnode El nodo inicial de la lista.
 *  \param[in] string_data La cadena a escribir.
 *  \return PDS_OK si todo fue bien o PDS_WRONG si no. Si no le das el primer nodo da error.
 *  \ingroup PdsListStringGroup
 */
int pds_list_string_unshift(PdsListString **Lnode,const char *string_data)
{
	PdsListString *L=NULL;

	if((*Lnode)==NULL)		return PDS_WRONG;
	if((*Lnode)->AddPrev!=NULL)	return PDS_WRONG;

	if((*Lnode)->S==NULL)
	{
		(*Lnode)->S=(char*)calloc(sizeof(char),strlen(string_data)+1);
        sprintf((*Lnode)->S,"%s",string_data);
		return PDS_OK;
	}
	else
	{
		L=(PdsListString *)calloc(1,sizeof(PdsListString));
		if(L==NULL)	return PDS_WRONG;
	
		L->AddNext=(struct PdsListString *)(*Lnode);
		L->S=(char*)calloc(sizeof(char),strlen(string_data)+1);
		if(L->S==NULL)
		{
			free(L);
			return PDS_WRONG;
		}
        sprintf(L->S,"%s",string_data);
		L->AddPrev=NULL;

		(*Lnode)->AddPrev=(struct PdsListString *)L;

		(*Lnode)=L;
	
		return PDS_OK;
	}
}


/** \fn int pds_list_string_go_top(PdsListString **L)
 *  \brief Busca el elemento final; superior; de la lista.
 *  \param[in,out] L Un nodo de la lista, en donde se cargará el último nodo.
 *  \return PDS_OK si todo fue bien o PDS_WRONG si no. 
 *  \ingroup PdsListStringGroup
 */
int pds_list_string_go_top(PdsListString **L)
{
	if((*L)==NULL)	return PDS_WRONG;

	while((*L)->AddNext!=NULL)
	{
		(*L)=(PdsListString*)((*L)->AddNext);
	}
	return PDS_OK;
}


/** \fn int pds_list_string_go_bottom(PdsListString **L)
 *  \brief Busca el elemento inicial; inferior; de la lista.
 *  \param[in,out] L Un nodo de la lista, en donde se cargará el primer nodo.
 *  \return PDS_OK si todo fue bien o PDS_WRONG si no. 
 *  \ingroup PdsListStringGroup
 */
int pds_list_string_go_bottom(PdsListString **L)
{
	if((*L)==NULL)	return PDS_WRONG;

	while((*L)->AddPrev!=NULL)
	{
		(*L)=(PdsListString*)((*L)->AddPrev);
	}
	return PDS_OK;
}


/** \fn int pds_list_string_printf(const PdsListString *L)
 *  \brief Imprime en pantalla los datos de un nodo de tipo puntero PdsListString.
 *  \param[in] L El nodo a imprimir en pantalla.
 *  \return PDS_OK si todo fue bien o PDS_WRONG si no.
 *  \ingroup PdsListStringGroup
 */
int pds_list_string_printf(const PdsListString *L)
{
	if(L==NULL)		return PDS_WRONG;

	if(L->S!=NULL)	
	{
		printf("%s\n",L->S);
	}
	
	return PDS_OK;
}


/** \fn int pds_list_string_detailed_printf(const PdsListString *L)
 *  \brief Imprime en pantalla detalladamente los datos de un nodo de tipo puntero PdsListString.
 *  \param[in] L El nodo a imprimir en pantalla.
 *  \return PDS_OK si todo fue bien o PDS_WRONG si no.
 *  \ingroup PdsListStringGroup
 */
int pds_list_string_detailed_printf(const PdsListString *L)
{
	if(L==NULL)		return PDS_WRONG;

	printf("%p\t",L);
	printf("AddPrev : %p\n",L->AddPrev);
	if(L->S!=NULL)	
	{
		printf("\t\tS       : ");
		printf("%s\n",L->S);
	}
	else		printf("\t\tS       :\n");
	printf("\t\tAddNext : %p\n",L->AddNext);
	
	return PDS_OK;
}


/** \fn int pds_list_string_all_printf(const PdsListString *L)
 *  \brief Imprime en pantalla todos los datos de la lista.
 *  \param[in] L Un nodo de la lista a imprimir en pantalla.
 *  \return PDS_OK si todo fue bien o PDS_WRONG si no.
 *  \ingroup PdsListStringGroup
 */
int pds_list_string_all_printf(const PdsListString *L)
{
	PdsListString *Lbottom=NULL;
	int id;

	if(L==NULL)		return PDS_WRONG;

	Lbottom=(PdsListString *)L;

	id=pds_list_string_go_bottom(&Lbottom);
	if(id==PDS_WRONG)	return PDS_WRONG;

	
	while(1)
	{
		id=pds_list_string_printf(Lbottom);
		if(id==PDS_WRONG)	return PDS_WRONG;

		if(Lbottom->AddNext!=NULL)	Lbottom=(PdsListString*)Lbottom->AddNext;
		else				break;
	}
		
	return PDS_OK;
}



/** \fn int pds_list_string_all_detailed_printf(const PdsListString *L)
 *  \brief Imprime en pantalla todos los datos de la lista detalladamente.
 *  \param[in] L Un nodo de la lista a imprimir en pantalla.
 *  \return PDS_OK si todo fue bien o PDS_WRONG si no.
 *  \ingroup PdsListStringGroup
 */
int pds_list_string_all_detailed_printf(const PdsListString *L)
{
	PdsListString *Lbottom=NULL;
	int id;

	if(L==NULL)		return PDS_WRONG;

	Lbottom=(PdsListString *)L;

	id=pds_list_string_go_bottom(&Lbottom);
	if(id==PDS_WRONG)	return PDS_WRONG;

	
	while(1)
	{
		id=pds_list_string_detailed_printf(Lbottom);
		if(id==PDS_WRONG)	return PDS_WRONG;

		if(Lbottom->AddNext!=NULL)	Lbottom=(PdsListString*)Lbottom->AddNext;
		else				break;
	}
		
	return PDS_OK;
}



/** \fn void pds_list_string_free(PdsListString *L)
 *  \brief Libera una lista entera de tipo puntero PdsListString. 
 *  \param[in,out] L La lista a liberar.
 *  \return No retorna valor.
 *  \ingroup PdsListStringGroup
 */
void pds_list_string_free(PdsListString *L)
{
	PdsListString *Lbottom=NULL;
	int id;

	if(L!=NULL)
	{
		Lbottom=L;

		id=pds_list_string_go_bottom(&Lbottom);
		if(id==PDS_OK)
		{
			while(1)
			{
				pds_list_string_shift(&Lbottom);
				if(Lbottom->AddNext==NULL)	break;
			}

			free(Lbottom);
		}
	}
}


/** \fn void pds_list_string_destroy(PdsListString **L)
 *  \brief Libera una lista de tipo puntero PdsListString, y limpia el puntero con NULL.
 *  \param[in,out] L La lista a liberar y limpiar.
 *  \return No retorna valor.
 *  \ingroup PdsListStringGroup
 */
void pds_list_string_destroy(PdsListString **L)
{
	PdsListString *Lbottom=NULL;
	int id;

	if(L!=NULL)
	if((*L)!=NULL)
	{
		Lbottom=*L;

		id=pds_list_string_go_bottom(&Lbottom);
		if(id==PDS_OK)
		{
			while(1)
			{
				pds_list_string_shift(&Lbottom);
				if(Lbottom->AddNext==NULL)	break;
			}

			free(Lbottom);
			*L=NULL;
		}
	}
}




/** \fn int pds_list_string_add_top_node(PdsListString **Lnode,const char* string_data)
 *  \brief Agrega un elemento a la cima de la lista. Si la lista no está en la cima,
 *  la lista es llevada a la cima y es dejada en la cima luego de agregar esta nueva posición.
 *  \param[in,out] Lnode Un nodo cualquiera de la lista.
 *  \param[in] string_data La cadena a escrever.
 *  \return PDS_OK si todo fue bien o PDS_WRONG si no.
 *  \ingroup PdsListStringGroup
 */
int pds_list_string_add_top_node(PdsListString **Lnode,const char* string_data)
{
    int id;

	if((*Lnode)==NULL)  return PDS_WRONG;

	if((*Lnode)->AddNext!=NULL) 
    {
        id=pds_list_string_go_top(Lnode);
        if (id==PDS_WRONG)  return PDS_WRONG;
    }

    id=pds_list_string_push(Lnode,string_data);
    return id;
}

/** \fn int pds_list_string_delete_top_node(PdsListString **Lnode)
 *  \brief Elimina un elemento a la cima de la lista. Si la lista no está en la cima,
 *  la lista es llevada a la cima y es dejada en la cima luego de eliminar la ultima posición.
 *  Si la lista tiene un solo nodo, este es limpiado y Lnode no cambia de dirección
 *  \param[in,out] Lnode Un nodo cualquiera de la lista.
 *  \return PDS_OK si todo fue bien o PDS_WRONG si no.
 *  \ingroup PdsListStringGroup
 */
int pds_list_string_delete_top_node(PdsListString **Lnode)
{
    int id;

	if((*Lnode)==NULL)  return PDS_WRONG;

	if((*Lnode)->AddNext!=NULL) 
    {
        id=pds_list_string_go_top(Lnode);
        if (id==PDS_WRONG)  return PDS_WRONG;
    }

    id=pds_list_string_pop(Lnode);
    return id;
}


/** \fn int pds_list_string_read_and_advance(PdsListString **L,char** string_data)
 *  \brief Lee una cadena de texto del nodo actual de la lista y se coloca en el siguiente 
 *  nodo de la lista, si está ya en el último nodo no se mueve al siguiente nodo.
 *  \param[in,out] L Un nodo cualquiera de la lista.
 *  \param[out] string_data La dirección de memoria interna en el PdsListString  
 *  de la cadena a leer (NO liberar esta dirección de memoria).
 *  \return PDS_OK si fue posible moverse al siguiente nodo o PDS_WRONG si no pudo 
 *  moverse al siguiente nodo o si (*L)==NULL.
 *  \ingroup PdsListStringGroup
 */
int pds_list_string_read_and_advance(PdsListString **L,char** string_data)
{
	if((*L)==NULL)		    return PDS_WRONG;

    (*string_data)=(*L)->S;

	if((*L)->AddNext!=NULL) 
    {
        (*L)=(PdsListString*)((*L)->AddNext);
    }
    else return PDS_WRONG;

    return PDS_OK;
}

/** \fn int pds_list_string_read(const PdsListString *L,char** string_data)
 *  \brief Lee la cadena de texto del nodo actual de la lista.
 *  \param[in] L Un nodo cualquiera de la lista.
 *  \param[out] string_data La dirección de memoria interna en el PdsListString 
 *  de la cadena a leer (NO liberar esta dirección de memoria).
 *  \return PDS_OK si todo fue bien o PDS_WRONG si (*L)==NULL.
 *  \ingroup PdsListStringGroup
 */
int pds_list_string_read(const PdsListString *L,char** string_data)
{
	if(L==NULL) return PDS_WRONG;

    (*string_data)=L->S;

    return PDS_OK;
}


/** \fn char* pds_list_string_get_string(const PdsListString *L)
 *  \brief Lee una cadena de texto de un nodo en la lista actual y retorna una 
 *  nueva cadena con el contenido.
 *  \param[in] L Un nodo cualquiera de la lista.
 *  \return Una nueva cadena con el contenido del PdsListString. O NULL en caso
 *  de cadena vacia o L==NULL o error de allocacion de memoria. Esta cadena debe
 *  ser liberada despues de usada.
 *  \ingroup PdsListStringGroup
 */
char* pds_list_string_get_string(const PdsListString *L)
{
    char *string=NULL;

	if(L==NULL)		return NULL;
    if(L->S==NULL)  return NULL;


    string=(char *)calloc(strlen(L->S)+1,sizeof(char));
    if(string == NULL)  return NULL;

    strcpy(string,L->S);

    return string;
}


/** \fn int pds_list_string_is_top(const PdsListString *L)
 *  \brief Retorna PDS_OK si L apunta el nodo que esta en el top de la lista.
 *  \param[in] L Un nodo cualquiera de la lista.
 *  \return PDS_OK si L apunta al final de la lista o PDS_WRONG si no.
 *  \ingroup PdsListStringGroup
 */
int pds_list_string_is_top(const PdsListString *L)
{
	if(L==NULL)		    return PDS_WRONG;

	if(L->AddNext==NULL)    return PDS_OK;
    else                    return PDS_WRONG;
}


/** \fn int pds_list_string_is_bottom(const PdsListString *L)
 *  \brief Retorna PDS_OK si L apunta el nodo que esta en el bottom de la lista.
 *  \param[in] L Un nodo cualquiera de la lista.
 *  \return PDS_OK si L apunta al inicio de la lista o PDS_WRONG si no.
 *  \ingroup PdsListStringGroup
 */
int pds_list_string_is_bottom(const PdsListString *L)
{
	if(L==NULL)		    return PDS_WRONG;

	if(L->AddPrev==NULL)    return PDS_OK;
    else                    return PDS_WRONG;
}


/** \fn int pds_list_string_get_number_of_nodes(const PdsListString *L)
 *  \brief Cuenta la cantidad de nodos de la lista.
 *  \param[in] L Un nodo cualquiera de la lista a consultar.
 *  \return Retorna la cantidad de nodos de la lista, 0 si L==NULL o un número  
 *  menor que cero en caso de error.
 *  \ingroup PdsListStringGroup
 */
int pds_list_string_get_number_of_nodes(const PdsListString *L)
{
	PdsListString *Lbottom=NULL;
	int N=0;
	int id;

	if(L==NULL)		return 0;

	Lbottom=(PdsListString *)L;

	id=pds_list_string_go_bottom(&Lbottom);
	if(id==PDS_WRONG)	return -1;

	
	while(1)
	{
        N=N+1;

		if(Lbottom->AddNext!=NULL)	Lbottom=(PdsListString*)Lbottom->AddNext;
		else				break;
	}
		
	return N;
}

/** \fn int pds_list_string_get_number_of_no_null_nodes(const PdsListString *L)
 *  \brief Cuenta la cantidad de nodos con strings no nulos en la lista.
 *  \param[in] L Un nodo cualquiera de la lista a consultar.
 *  \return Retorna la cantidad de nodos no nulos de la lista, 0 si L==NULL o un 
 *  número menor que cero en caso de error.
 *  Si se encuentra un nodo con L->AddNext==L->AddPrev==NULL
 *  es considerado un error (exepto para el nodo inicial). 
 *  \ingroup PdsListStringGroup
 */
int pds_list_string_get_number_of_no_null_nodes(const PdsListString *L)
{
	PdsListString *Lbottom=NULL;
	int N=0;
	int id;

	if(L==NULL)		return 0;

	Lbottom=(PdsListString *)L;

	id=pds_list_string_go_bottom(&Lbottom);
	if(id==PDS_WRONG)	return -1;

	
	while(1)
	{
        if (Lbottom->S==NULL)
        if (Lbottom->AddNext==NULL)
        if (Lbottom->AddPrev==NULL) 
        {
            if (N==0)   return 0;
            else        return -1;
        }

		if (Lbottom->S!=NULL)   N=N+1;

		if(Lbottom->AddNext!=NULL)	Lbottom=(PdsListString*)Lbottom->AddNext;
		else				        break;

	}
		
	return N;
}



/** \fn int pds_list_string_is_empty(const PdsListString *L)
 *  \brief Indica si el nodo está vacío L->S==L->AddNextL->AddPrev==NULL.
 *  \param[in] L Un nodo a consultar.
 *  \return Retorna PDS_OK si L==NULL o si todos los elementos del nodo son nulos.
 *  \ingroup PdsListStringGroup
 */
int pds_list_string_is_empty(const PdsListString *L)
{
	if(L==NULL)		return PDS_OK;
	
	if (L->S==NULL)
    if (L->AddNext==NULL)
    if (L->AddPrev==NULL) 
    {
        return PDS_OK;
    }
		
	return PDS_WRONG;
}
