/*
 * pdsstring.h
 * 
 * Copyright 2011 Fernando Pujaico Rivera <fernando.pujaico.rivera@gmail.com>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 * 
 */

/** \file pdsstring.h
 *  \author Fernando Pujaico Rivera
 *  \date 30-07-2016
 *  \brief Funciones que manipulan archivos y caminos.
 *   
 */

#ifndef __PDSSTRING_H__
#define __PDSSTRING_H__


#ifdef __cplusplus
extern "C" {
#endif 

#include <stdio.h>
#include <pds/pdscstring.h>


/** \defgroup PdsStringGroup Funciones del módulo PdsString.
 *
 * <h1>\#include \<pds/pdsdatafunc.h\></h1>
 *
 * @{
 */

#ifndef PDS_OK
/*!
  \def PDS_OK
  Valor retornado en caso de exito, este es un valor diferente que cero.
*/
	#define PDS_OK 1
#endif

#ifndef PDS_WRONG
/*!
  \def PDS_WRONG
  Valor retornado en caso de fracaso, este valor es igual a cero.
*/
	#define PDS_WRONG 0
#endif


/** \fn char * pds_string_new_from_file(const char *path)
 *  \brief Esta función retorna una cadena de caracteres con el contenido de un archivo.
 *  \param[in] path El archivo a leer.
 *  \return Retorna una cadena de caracteres con el contenido de un archivo. O
 *  NULL en caso de error.
 *  \ingroup PdsStringGroup
 */
char * pds_string_new_from_file(const char *path);





/** @name pds_<std*_variations>
 *  Funciones variantes de stdlib y stdio.
 *
 * @{
 */

/** \fn int isintnumber(const char *str)
 *  \brief Esta función verifica si la cadena contiene un número entero. La función
 *  solo desconsidera espacios en blanco iniciales y finales
 *  \param[in] str Cadena a anlizar.
 *  \return Retorna PDS_OK si es un numero y PDS_WRONG quando no lo es o str==NULL
 *  o str es una cadena vacia o hubo un error de allocacion de memoria.
 *  \ingroup PdsStringGroup
 */
int isintnumber(const char *str);

/** \fn char *pds_strcat_cell_string(const PdsCellString *celldata,const char *separator)
 *  \brief Esta función retorna una cadena de caracteres con el contenido concatenado
 *  usando como separador separator.
 *
 *  "str1"<b>separator</b>"str2"<b>separator</b> ... <b>separator</b>"strN"
 *
 *  \param[in] celldata La estructura a concatenar.
 *  \param[in] separator Separador usado entre celulas.
 *  \return Retorna una nueva cadena de caracteres con el contenido concatenado.
 *  O NULL en caso de error.
 *  \see pds_cell_string_joint()
 *  \see pds_cell_string_joint_with_end()
 *  \ingroup PdsStringGroup
 */
char *pds_strcat_cell_string(const PdsCellString *celldata,const char *separator);


/** \fn char* pds_sprintf(const char *format, ...)
 *  \brief Similar a sprintf pero esta retorna un puntero a una nueva cadena con 
 *  el texto.
 *
 *  \param[in] format Formato de texto de salida.
 *  \param[in] ... Otros argumentos.
 *  \return Retorna un puntero a una nueva cadena con el texto. O NULL en caso de error.
 *  \ingroup PdsStringGroup
 */
char* pds_sprintf(const char *format, ...);


/** \fn PdsCellString *pds_strsep_cell_string(const char *str, const char *delim)
 *  \brief Esta función es similar a la función strsep, donde la cadena str es 
 *  dividida usando algunos de los delimitadores especificados en delim. Los 
 *  pedazos son cargados en una estructura PdsCellString* retornada por la función.
 *  \param[in] str Cadena a dividir, esta cadena no sera modificada.
 *  \param[in] delim Delimitadores a usar. Si delim es NULL la función retorna NULL.
 *  Si delim es una cadena vacia es retornada la cadena entera dentro de una PdsCellString.
 *  \return Retorna un puntero a una nueva estructura de tipo PdsCellString *,
 *  con los tokens de la división.
 *  \ingroup PdsStringGroup
 */
PdsCellString *pds_strsep_cell_string(const char *str, const char *delim);


/** \fn int pds_strcicmp(const char *str0,const char *str1)
 *  \brief Esta función es similar a la función strcmp pero no distinguiendo 
 *  mayúsculas y minúsculas (Case Insensitive), donde si las cadenas son iguales
 *  la funcion retorna 0.
 *  \param[in] str0 Primera cadena a comparar.
 *  \param[in] str1 Segunda cadena a comparar.
 *  \return Retorna 0 si las cadenas son iguales (Case Insensitive).
 *  \ingroup PdsStringGroup
 */
int pds_strcicmp(const char *str0,const char *str1);

/** \fn int pds_strlexcmp(const char *str0,const char *str1)
 *  \brief Esta función es similar a la función strcmp pero compara lexicograficamente
 *  las cadenas, donde si las cadenas son iguales
 *  la funcion retorna 0.
 *  \param[in] str0 Primera cadena a comparar.
 *  \param[in] str1 Segunda cadena a comparar.
 *  \return Retorna 0 si las cadenas son iguales (Case Insensitive).
 *  Retorna < 0 si str0 is less than str1.
 *  Retorna > 0 si str1 is less than str0.
 *  \ingroup PdsStringGroup
 */
int pds_strlexcmp(const char *str0,const char *str1);

/** \fn char *pds_toupper(const char *str)
 *  \brief Esta función es similar a la función toupper() pero transforma una  
 *  cadena de texto (y no un caracter) a letras mayúsculas.
 *  \param[in] str Cadena a convetir en mayúsculas.
 *  \return Retorna una nueva cadena con el resultado de convertir en mayúsculas.
 *  Retorna NULL si str es NULL o si hubo problemas de reserva de memoria.
 *  \ingroup PdsStringGroup
 */
char *pds_toupper(const char *str);

/** \fn int pds_inupper(char *str)
 *  \brief Esta función  modifica una cadena de texto a letras mayúsculas.
 *  \param[in] str Cadena a convetir en mayúsculas.
 *  \return Retorna TRUE si todo fue bien o PDS_WRONG si str es NULL.
 *  \ingroup PdsStringGroup
 */
int pds_inupper(char *str);

/** \fn char *pds_tolower(const char *str)
 *  \brief Esta función es similar a la función tolower() pero transforma una  
 *  cadena de texto (y no un caracter) a letras minúsculas.
 *  \param[in] str Cadena a convetir en minúsculas.
 *  \return Retorna una nueva cadena con el resultado de convertir en minúsculas.
 *  Retorna NULL si str es NULL o si hubo problemas de reserva de memoria.
 *  \ingroup PdsStringGroup
 */
char *pds_tolower(const char *str);

/** \fn int pds_inlower(char *str)
 *  \brief Esta función  modifica una cadena de texto a letras minúsculas.
 *  \param[in] str Cadena a convetir en minúsculas.
 *  \return Retorna TRUE si todo fue bien o PDS_WRONG si str es NULL.
 *  \ingroup PdsStringGroup
 */
int pds_inlower(char *str);

/** \fn char *pds_itoa(int num,int base)
 *  \brief Esta función es similar a la función itoa() pero retorna una nueva
 *  cadena de texto con el número num en la base base.
 *  \param[in] num Número a escribir.
 *  \param[in] base Base del número a escribir.
 *  \return Retorna una nueva cadena con el numero num en la base base..
 *  Retorna NULL si hubo problemas de reserva de memoria o base <=0 .
 *  \ingroup PdsStringGroup
 */
char *pds_itoa(int num,int base);

//@}

/** @name pds_string_<type>_clean_<location>
 *  Funciones para limpiar strings.
 *
 * @{
 */

/** \fn int pds_string_multi_clean_beginning(char *str,const char *pat)
 *  \brief Esta función limpia los primeros caracteres de la cadena str que 
 *  contengan algun caracter en pat, la cadena no es realocada simplemente los datos son
 *  movidos de lugar.
 *  \param[in] str Cadena a limpiar.
 *  \param[in] pat Caracteres a limpiar.
 *  \return Retorna TRUE si las cadena fue limpiada correctamente o retorna PDS_WRONG si
 *  str o pat es igual a NULL;
 *  \ingroup PdsStringGroup
 */
int pds_string_multi_clean_beginning(char *str,const char *pat);



/** \fn int pds_string_multi_clean_ending(char *str,const char *pat)
 *  \brief Esta función limpia los ultimos caracteres de la cadena str que 
 *  contengan algun caracter en pat, la cadena no es realocada simplemente los datos son
 *  cerados.
 *  \param[in] str Cadena a limpiar.
 *  \param[in] pat Caracteres a limpiar.
 *  \return Retorna TRUE si las cadena fue limpiada correctamente o retorna PDS_WRONG si
 *  str o pat es igual a NULL;
 *  \ingroup PdsStringGroup
 */
int pds_string_multi_clean_ending(char *str,const char *pat);


/** \fn int pds_string_clean_beginning(char *str,char c)
 *  \brief Esta función limpia los primeros caracteres de la cadena str que 
 *  contengan el caracter c, la cadena no es realocada simplemente los datos son
 *  movidos de lugar.
 *  \param[in] str Cadena a limpiar.
 *  \param[in] c Caracter a limpiar.
 *  \return Retorna TRUE si las cadena fue limpiada correctamente o retorna PDS_WRONG si
 *  str es igual a NULL;
 *  \ingroup PdsStringGroup
 */
int pds_string_clean_beginning(char *str,char c);


/** \fn int pds_string_clean_ending(char *str,char c)
 *  \brief Esta función limpia los ultimos caracteres de la cadena str que 
 *  contengan el caracter c, la cadena no es realocada simplemente los datos son
 *  cerados.
 *  \param[in] str Cadena a limpiar.
 *  \param[in] c Caracter a limpiar.
 *  \return Retorna TRUE si las cadena fue limpiada correctamente o retorna PDS_WRONG si
 *  str es igual a NULL;
 *  \ingroup PdsStringGroup
 */
int pds_string_clean_ending(char *str,char c);

//@}


/** @name pds_string_<work_with_patterns>
 *  Funciones para limpiar strings.
 *
 * @{
 */


/** \fn int pds_string_get_number_of_patterns(const char *str,const char *pat)
 *  \brief Esta función cuenta la cantidad de coincidencias de la cadena pat
 *  en la cadena str.
 *
 *  Cuando encuentra una coincidencia en la posición p0 de str, continua la
 *  siguiente búsqueda en p0+strlen(path).
 *  \param[in] str Cadena donde se realizará la búsqueda.
 *  \param[in] pat Patrona buscar.
 *  \return Retorna el número de coincidencias o -1 en caso de error.
 *  \ingroup PdsStringGroup
 */
int pds_string_get_number_of_patterns(const char *str,const char *pat);


/** \fn char *pds_string_pattern_replacement(const char *str,const char *pat,const char *rep)
 *  \brief Esta función busca en una cadena str un patrón pat y los remplaza con 
 *  el contenido de la cadena rep, el resultado es retornado en una nueva cadena;
 *
 *  Cuando encuentra una coincidencia en la posición p0 de str, continua la
 *  siguiente búsqueda en p0+strlen(pat).
 *  \param[in] str Cadena donde se realizará la búsqueda.
 *  \param[in] pat Patrón a buscar.
 *  \param[in] rep Patrón a de remplazo.
 *  \return Retorna una nueva cadena con el resultado de remplazar pat por rep en str.
 *  \ingroup PdsStringGroup
 */
char *pds_string_pattern_replacement(const char *str,const char *pat,const char *rep);

/** \fn char *pds_string_content_replacement( const char *str_in, const char *pat_open,const char *pat_close, char *(*func)(const char *))
 *  \brief Esta función busca en una cadena de texto str_in, el contenido
 *  entre un patrón pat_open y pat_close y lo remplaza  por el valor devuelto de 
 *  la función func, el resultado de todo esto es retornado en una nueva cadena 
 *  de texto.
 *
 *  La función func debe ser de la forma :
\code{.c}
char *func(const char* content);
\endcode
 *
\code{.c}
str_out=pds_string_content_replacement(str_in,"<b>","</b>",func);
\endcode
 *  \param[in] str_in Cadena de entrada donde se realizará la búsqueda.
 *  \param[in] pat_open Patrón inicial a buscar.
 *  \param[in] pat_close Patrón final a buscar.
 *  \param[in] func Es una función que se usará para remplazar el contenido 
 *  encontrado. Por ejemplo si entre pat_open y pat_close es encontrado content
\code{.c}
replace=func(content);
\endcode
 *  este será remplazado por el contenido de la cadena replace, luego la cadena 
 *  replace será liberada internamente con free(replace).
 *  Si la funcion func retorna NULL, la busqueda en str_in es detenida y
 *  se retorna NULL.
 *  \return Retorna una nueva cadena de texto si todo fue bien o NULL si no.
 *  \ingroup PdsStringGroup
 */
char *pds_string_content_replacement( const char *str_in,
                                     const char *pat_open,const char *pat_close,
                                     char *(*func)(const char *));


/** \fn char *pds_string_get_match_rawdata(const char *str,const char *pat_open,const char *pat_close,char **str_ptr)
 *  \brief Esta función busca en una cadena str, los datos entre un patrón 
 *  pat_open y uno pat_close. Retorna los datos en una nueva cadena de texto.
 *
 *  Cuando encuentra una coincidencia la posición inmediata después de pat_close
 *  es cargada en str_ptr (solo si esa variable str_ptr existe).
 *
 *  Ejemplos : Busca la coincidencia y además es cargado str_ptr con la dirección
 *  inmediata a la coincidencia con pat_close.
\code{.c}
char str[]="hola <b> Fernando\n</b> como estas";
char *str_ptr=NULL;
char *rawdata=pds_string_get_match_rawdata(str,"<b>","</b>",&str_ptr);
\endcode
 *  Ejemplos : Busca la coincidencia en str.
\code{.c}
char str[]="hola <b> Fernando\n</b> como estas";
char *rawdata=pds_string_get_match_rawdata(str,"<b>","</b>",NULL);
\endcode
 *  \param[in] str Cadena donde se realizará la búsqueda.
 *  \param[in] pat_open Patrón de apertura.
 *  \param[in] pat_close Patrón de finalización.
 *  \param[out] str_ptr puntero en cuyo contenido se grabará la dirección de la
 *  dirección de memoria en str después de pat_close.
 *  Si se retorna un str_ptr==NULL significa que se llegó al final de la cadena.
 *  Este solo es una dirección de memoria de str y no debe ser liberada.
 *  \return Retorna una nueva cadena con el contenido entre pat_open y pat_close.
 *  \ingroup PdsStringGroup
 */
char *pds_string_get_match_rawdata(const char *str,const char *pat_open,const char *pat_close,char **str_ptr);

//@}


/**
 * @}
 */

#ifdef __cplusplus
}
#endif 

#endif	/* __PDSSTRING_H__ */ 

