/*
 * pdsfirnlms.c
 * 
 * Copyright 2011 Fernando Pujaico Rivera <fernando.pujaico.rivera@gmail.com>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 * 
 */


#include <pds/pdsfir.h>
#include <pds/pdsfirnlms.h>
#include <pds/pdsvector.h>


////////////////////////////////////////////////////////////////////////////////
////  Trabajando con PdsFirNlms                                             ////
////////////////////////////////////////////////////////////////////////////////


/** \fn PdsFirNlms *pds_fir_nlms_new(PdsDfReal Mhu, PdsRaNatural M)
 *  \brief Crea un filtro FIR NLMS con parametros h[i] del filtro FIR, con un 
 *  valor inicial de h[i]=1/(1+M). Por defecto el filtro FIR NLMS estará 
 *  auto configurandose continuamente, a no ser que se deshabilite con
 *  pds_fir_nlms_disable .
 * 
 *  \param[in] Mhu Es el paso de la constante de adaptación.
 *  \param[in] M Es el grado del filtro FIR, h[i], 0<= i <=M.
 *  \return Un puntero a una estructura de tipo PdsFirNlms. O NULL en caso de error.
 *  \ingroup PdsFirNlmsGroup
 */
PdsFirNlms *pds_fir_nlms_new(PdsDfReal Mhu, PdsRaNatural M)
{
	PdsFirNlms	*FirNlms=NULL;
	PdsVector	*h=NULL;
	PdsDfNatural	i;


	h=pds_vector_new(M+1);
	if(h==NULL)	return NULL;
	for(i=0;i<=M;i++)	h->V[i]=1.0/(M+1.0);


	FirNlms=(PdsFirNlms *)calloc(1,sizeof(PdsFirNlms));
	if(FirNlms==NULL)
	{	
		pds_vector_free(h);
		return NULL;
	}

	FirNlms->Fir=pds_fir_new(h);
	if(FirNlms->Fir==NULL)
	{	
		pds_vector_free(h);
		pds_fir_nlms_free(FirNlms);
		return NULL;
	}

	FirNlms->Work=1;
	FirNlms->Mhu=Mhu;

	pds_vector_free(h);

	return FirNlms;
}


/** \fn int pds_fir_nlms_disable(PdsFirNlms *FirNlms)
 *  \brief Deshabilita la reconfiguración de los pesos h[i] del filtro FIR NLMS
 *  y se mantienen los ultimos pesos modificados
 * 
 *  \param[in,out] FirNlms El filtro FIR NLMS a usar.
 *  \return TRUE si todo fue bien o FALSE si no;
 *  \ingroup PdsFirNlmsGroup
 */
int pds_fir_nlms_disable(PdsFirNlms *FirNlms)
{

	if(FirNlms==NULL)	return FALSE;

	FirNlms->Work=0;
	return TRUE;
}


/** \fn int pds_fir_nlms_enable(PdsFirNlms *FirNlms)
 *  \brief Habilita la reconfiguración de los pesos h[i] del filtro FIR NLMS.
 * 
 *  \param[in,out] FirNlms El filtro FIR NLMS a usar.
 *  \return TRUE si todo fue bien o FALSE si no;
 *  \ingroup PdsFirNlmsGroup
 */
int pds_fir_nlms_enable(PdsFirNlms *FirNlms)
{

	if(FirNlms==NULL)	return FALSE;

	FirNlms->Work=1;
	return TRUE;
}

/** \fn int pds_fir_nlms_set_mhu(PdsFirNlms *FirNlms,PdsDfReal Mhu)
 *  \brief Coloca el valor Mhu del filtro FIR NLMS.
 * 
 *  \param[in,out] FirNlms El filtro FIR NLMS a usar.
 *  \param[in] Mhu Factor de aprendizaje Mhu.
 *  \return TRUE si todo fue bien o FALSE si no;
 *  \ingroup PdsFirNlmsGroup
 */
int pds_fir_nlms_set_mhu(PdsFirNlms *FirNlms,PdsDfReal Mhu)
{

	if(FirNlms==NULL)	return FALSE;
	FirNlms->Mhu=Mhu;
	return TRUE;
}


/** \fn int pds_fir_nlms_evaluate_value(PdsFirNlms *FirNlms,PdsDfReal d,PdsDfReal x,PdsDfReal *e,PdsDfReal *y)
 *  \brief Evalúa el filtro FIR NLMS con el valor de entrada "d" e "x", el 
 *  resultado es cargado en "e" e "y".
 * 
 *  \param[in,out] FirNlms El filtro FIR NLMS a usar.
 *  \param[in] d Es una entrada adicional al filtro FIR NLMS.
 *  \param[in] x Es la señal de entrada del filtro FIR.
 *  \param[out] e Es la señal de error, es la diferencia entre "d" e "y".
 *  \param[out] y Es el valor de salida del filtro FIR.
 *  \return TRUE si todo fue bien o FALSE si no;
 *  \ingroup PdsFirNlmsGroup
 */
int pds_fir_nlms_evaluate_value(PdsFirNlms *FirNlms,PdsDfReal d,PdsDfReal x,PdsDfReal *e,PdsDfReal *y)
{
	PdsRaNatural i,N;
	PdsRaReal m;

	if(FirNlms==NULL)	return FALSE;

	pds_fir_evaluate_value(FirNlms->Fir,x,y);
	*e=d-(*y);

	pds_vector_norm2_vector (FirNlms->Fir->X,&m);

	N=FirNlms->Fir->N;
	if(FirNlms->Work==1)
	{
		for(i=0;i<N;i++)
		FirNlms->Fir->h->V[i] = 	FirNlms->Fir->h->V[i]+
						FirNlms->Mhu*(*e)*FirNlms->Fir->X->V[i]/m;
	}

	return TRUE;
}


/** \fn int pds_fir_nlms_evaluate_vector(PdsFirNlms *FirNlms,const PdsVector *d,const PdsVector *x,PdsVector *e,PdsVector *y)
 *  \brief Evalúa el filtro FIR NLMS con el vector de entrada x, el resultado
 *  es cargado en el vector y.
 * 
 *  Se recomienda usar esta función solo cuando x es mucho mayo que FIR->h.
 *  Solo se realizan corrimientos de FIR->X al inicio y al final del vector x
 *  en los casos intermediarios se aprovecha tener el vector y no se efectuan
 *  corrimientos, por lo que es un poco mas rápido que pds_fir_nlms_evaluate_value
 *  cuando x es mucho mayo que FIR->h.
 *  \param[in,out] FirNlms El filtro FIR a usar.
 *  \param[in] d Es el vector de una entrada adicional al filtro FIR NLMS.
 *  \param[in] x Es el vector de la señal de entrada del filtro FIR.
 *  \param[out] e Es el vector de la señal de error, es la diferencia entre "d" e "y".
 *  \param[out] y Es el vector del valor de salida del filtro FIR.
 *  \return TRUE si todo fue bien o FALSE si no.
 *  \ingroup PdsFirNlmsGroup
 */
int pds_fir_nlms_evaluate_vector(PdsFirNlms *FirNlms,const PdsVector *d,const PdsVector *x,PdsVector *e,PdsVector *y)
{
	PdsRaNatural i,j,M,N,Nel;
	PdsDfReal S,m;

	if(FirNlms==NULL)	return FALSE;
	if((d==NULL)||(x==NULL)||(e==NULL)||(y==NULL))	return FALSE;
	if((x->Nel!=y->Nel)||(e->Nel!=d->Nel)||(x->Nel!=d->Nel))	return FALSE;

	N=FirNlms->Fir->N;
	Nel=x->Nel;

	// Evaluó salida y tramo1.
	for(j=0;(j<N)&&(j<Nel);j++)
	{
		S=0;
		m=0;
		for(i=0;i<=j;i++)
		{
			S=S+FirNlms->Fir->h->V[i]*x->V[j-i];
			m=m + x->V[j-i]*x->V[j-i];
		}
		for(i=j+1;i<N;i++)
		{
			S=S+FirNlms->Fir->h->V[i]*FirNlms->Fir->X->V[i-(j+1)];
			m=m + FirNlms->Fir->X->V[i-(j+1)]*FirNlms->Fir->X->V[i-(j+1)];
		}	

		y->V[j]=S;
		e->V[j]=d->V[j]-(y->V[j]);
		if(FirNlms->Work==1)
		{
			for(i=0;i<=j;i++)
			FirNlms->Fir->h->V[i] = 	FirNlms->Fir->h->V[i]+
							FirNlms->Mhu*e->V[j]*x->V[j-i]/m;
			for(i=j+1;i<N;i++)
			FirNlms->Fir->h->V[i] = 	FirNlms->Fir->h->V[i]+
							FirNlms->Mhu*e->V[j]*FirNlms->Fir->X->V[i-(j+1)]/m;
		}
	}
	// Evaluó salida y tramo2.
	for(j=N;j<Nel;j++)
	{
		S=0;
		m=0;
		for(i=0;i<=(N-1);i++)
		{
			S=S+FirNlms->Fir->h->V[i]*x->V[j-i];
			m=m + x->V[j-i]*x->V[j-i];
		}
		y->V[j]=S;
		e->V[j]=d->V[j]-(y->V[j]);
		if(FirNlms->Work==1)
		{
			for(i=0;i<N;i++)
			FirNlms->Fir->h->V[i] = 	FirNlms->Fir->h->V[i]+
							FirNlms->Mhu*e->V[j]*x->V[j-i]/m;
		}
	}
	// Ordenamiento final.
	if(N>Nel)	M=N-Nel;
	else		M=0;
	for(j=0;j<M;j++)
	{
		FirNlms->Fir->X->V[N-1-j]=FirNlms->Fir->X->V[M-1-j];
	}
	for(j=0;(j<Nel)&&(j<N);j++)
	{
		FirNlms->Fir->X->V[j]=x->V[(Nel-1)-j];
	}


	return TRUE;
}


/** \fn void pds_fir_nlms_free(PdsFirNlms *FirNlms)
 *  \brief Libera el filtro de tipo PdsFirNlms.
 *  \param[in] FirNlms El filtro a liberar.
 *  \return No retorna valor.
 *  \ingroup PdsFirNlmsGroup
 */
void pds_fir_nlms_free(PdsFirNlms *FirNlms)
{
	if(FirNlms!=NULL)
	{
		pds_fir_free(FirNlms->Fir);
		free(FirNlms);
	}
}


/** \fn void pds_fir_nlms_destroy(PdsFirNlms **FirNlms)
 *  \brief Libera el filtro de tipo PdsFirNlms. y carga la variable con NULL.
 *  \param[in] FirNlms El filtro a liberar.
 *  \return No retorna valor.
 *  \ingroup PdsFirNlmsGroup
 */
void pds_fir_nlms_destroy(PdsFirNlms **FirNlms)
{
	if((*FirNlms)!=NULL)
	{
		pds_fir_free((*FirNlms)->Fir);
		free((*FirNlms));
		(*FirNlms)=NULL;
	}
}


