/*
 * pdsiir.h
 * 
 * Copyright 2011 Fernando Pujaico Rivera <fernando.pujaico.rivera@gmail.com>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 * 
 */

/** \file pdsiir.h
 *  \author Fernando Pujaico Rivera
 *  \date 21-05-2011
 *  \brief Funciones que trabajan con estructuras Filtro Iir.
 *  
 *  <br>
 *  \image html iir-filter.png "Filtro IIR"
 *  \f[ y[n]=\sum_{i=0}^{P}b_ix[n-i]+\sum_{i=1}^{Q}a_iy[n-i] \f].
 *  \f[ \frac{Y[z]}{X[z]}=H[z]=\frac{\sum_{i=0}^{P}b_iz^{-i}}{1-\sum_{i=1}^{Q}a_iz^{-i}} \f].
 *  \f[ a[0]=-1\f]
 *  \f[ 0=\sum_{i=0}^{P}b_ix[n-i]+\sum_{i=0}^{Q}a_iy[n-i] \f].
 *  \f[ \frac{Y[z]}{X[z]}=H[z]=\frac{\sum_{i=0}^{P}h_{num}[i]z^{-i}}{\sum_{i=0}^{Q}h_{den}[i]z^{-i}} \f].
 */

#ifndef __PDSIIR_H__
#define __PDSIIR_H__


#ifdef __cplusplus
extern "C" {
#endif 

#include <pds/pdsdfglobal.h>
#include <pds/pdsvector.h>

#include <stdio.h>

#ifndef TRUE
	#define TRUE 1
#endif

#ifndef FALSE
	#define FALSE 0
#endif

/** \defgroup PdsIirGroup Funciones del módulo PdsIir.
 *
 *  <br>
 *  \image html iir-filter.png "Filtro IIR"
 *  \f[ y[n]=\sum_{i=0}^{P}b_ix[n-i]+\sum_{i=1}^{Q}a_iy[n-i] \f].
 *  \f[ \frac{Y[z]}{X[z]}=H[z]=\frac{\sum_{i=0}^{P}b_iz^{-i}}{1-\sum_{i=1}^{Q}a_iz^{-i}} \f].
 *  \f[ a[0]=-1\f]
 *  \f[ 0=\sum_{i=0}^{P}b_ix[n-i]+\sum_{i=0}^{Q}a_iy[n-i] \f].
 *  \f[ \frac{Y[z]}{X[z]}=H[z]=\frac{\sum_{i=0}^{P}h_{num}[i]z^{-i}}{\sum_{i=0}^{Q}h_{den}[i]z^{-i}} \f].
 * @{
 */


/*! \struct PdsIir
 *  \brief Una estructura tipo  PdsIir .
 *
 *  Esta estructura genera un filtro IIR.
 *  Para usar incluir pds/pdsdf.h.
 *  <br>
 *  \image html iir-filter.png "Filtro IIR"
 *  \f[ y[n]=\sum_{i=0}^{P}b_ix[n-i]+\sum_{i=1}^{Q}a_iy[n-i] \f].
 *  \f[ \frac{Y[z]}{X[z]}=H[z]=\frac{\sum_{i=0}^{P}b_iz^{-i}}{1-\sum_{i=1}^{Q}a_iz^{-i}} \f].
 *  \f[ a[0]=-1\f]
 *  \f[ 0=\sum_{i=0}^{P}b_ix[n-i]+\sum_{i=0}^{Q}a_iy[n-i] \f].
 *  \f[ \frac{Y[z]}{X[z]}=H[z]=\frac{\sum_{i=0}^{P}h_{num}[i]z^{-i}}{\sum_{i=0}^{Q}h_{den}[i]z^{-i}} \f].
 *  \ingroup PdsIirGroup
 *  \author Fernando Pujaico Rivera
 */
typedef struct 
{
	/*! Es el vector de elementos a_i. */
	PdsVector *a;
	/*! Indice del máximo elemento a_i. */
	PdsDfNatural Q;
	/*! Es el vector de elementos b_i. */
	PdsVector *b;
	/*! Indice del máximo elemento b_i. */
	PdsDfNatural P;
	/*! Vector temporal interno del filtro. */
	PdsVector *t;
}PdsIir;


/** \fn PdsIir *pds_iir_new(const PdsVector *hnum,const PdsVector *hden)
 *  \brief Crea un filtro IIR con parametros hnum y hden.
 * 
 *  \f[ \frac{Y[z]}{X[z]}=H[z]=\frac{\sum_{i=0}^{P}h_{num}[i]z^{-i}}{\sum_{i=0}^{Q}h_{den}[i]z^{-i}} \f].
 *  Los vectores hnum y hdem entregados como parametros de esta función nunca 
 *  son modificados.
 *  \param[in] hnum Vector para los coeficientes de numerador del filtro.
 *  \param[in] hden Vector para los coeficientes de denominador del filtro.
 *  \return Un puntero a una estructura de tipo PdsIir. O NULL en caso de error.
 *  Si a[0] es igual a cero la funcion da error y retorna NULL.
 *  \ingroup PdsIirGroup
 */
PdsIir *pds_iir_new(const PdsVector *hnum,const PdsVector *hden);


/** \fn int pds_iir_evaluate_value(PdsIir *IIR,PdsDfReal x,PdsDfReal *y)
 *  \brief Evalua el filtro IIR con el valor de entrada x, el resultado
 *  es cargado en y.
 * 
 *  \param[in,out] IIR El filtro IIR a usar.
 *  \param[in] x El valor de entrada del filtro.
 *  \param[out] y El valor de salida del filtro.
 *  \return TRUE si todo fue bien o FALSE si no.
 *  \ingroup PdsIirGroup
 */
int pds_iir_evaluate_value(PdsIir *IIR,PdsDfReal x,PdsDfReal *y);


/** \fn int pds_iir_evaluate_vector(PdsIir *IIR,const PdsVector *x,PdsVector *y)
 *  \brief Evalua el filtro IIR con el vector de entrada x, el resultado
 *  es cargado en el vector y.
 * 
 *  \param[in,out] IIR El filtro IIR a usar.
 *  \param[in] x El vector de entrada del filtro.
 *  \param[out] y El vector de salida del filtro.
 *  \return TRUE si todo fue bien o FALSE si no. Ejemplo x o y son NULL ó de 
 *  distinto tamaño.
 *  \ingroup PdsIirGroup
 */
int pds_iir_evaluate_vector(PdsIir *IIR,const PdsVector *x,PdsVector *y);


/** \fn void pds_iir_free(PdsIir *IIR)
 *  \brief Libera el filtro de tipo PdsIir.
 *  \param[in] IIR El filtro a liberar.
 *  \return No retorna valor.
 *  \ingroup PdsIirGroup
 */
void pds_iir_free(PdsIir *IIR);


/** \fn void pds_iir_destroy(PdsIir **IIR)
 *  \brief Libera el filtro de tipo PdsIir. y carga la variable con NULL.
 *  \param[in] IIR El filtro a liberar.
 *  \return No retorna valor.
 *  \ingroup PdsIirGroup
 */
void pds_iir_destroy(PdsIir **IIR);


/**
 * @}
 */

#ifdef __cplusplus
}
#endif 

#endif	/* __PDSIIR_H__ */ 

