/*
 * testprog4.c
 * 
 * Copyright 2011 Fernando Pujaico Rivera <fernando.pujaico.rivera@gmail.com>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 * 
 */


/** \example testprog4.c
 *  \author Fernando Pujaico Rivera
 *  \date 28-12-2011
 *  \brief Test del Filtro Fir LMS con la suma de tres señales sinusoidales 
 *  en pi/32 12pi/32 y 20pi/32 en la entrada "x", y una senhal en pi/32 y 
 *  12pi/32 en la señal "d".
 *
 *  <br>
 *  Ademas la señal de entrada "x" tiene una constante igual 1. La frecuencia 
 *  de muestro es 2*pi. El filtro LMS FIR es evaluado con un vector de entrada.
 *  <br>
 *  Test de un filtro FIR.
 *  \f[x[n]=1+sin(\frac{\pi}{32}n)+sin(12\frac{\pi}{32}n)+sin(\frac{20\pi}{32}n+\frac{\pi}{16})\f]
 *  \f[d[n]=  sin(\frac{\pi}{32}n)+sin(12\frac{\pi}{32}n)\f]
 *  <br>
 *  \image html grafico1.png "Señal de entrada d[n]."
 *  \image html grafico2.png "Señal de entrada x[n]."
 *  \image html grafico3.png "Señal de salida e[n]."
 *  \image html grafico4.png "Señal de salida y[n]."
 *  \image html graficofft1.png "FFT de la respuesta al impulso final h[i]."
 */

#include <config.h>
#include <stdlib.h>
#include <pds/pdsdf.h>
#include <pds/pdsvector.h>
#include "extras.h"
#include <math.h>

int main(int argc, char** argv)
{
	///////////////////////////////////////////////////////////////
	PdsFirLms *FirLms=NULL;

	PdsVector *D=NULL;
	PdsVector *X=NULL;
	PdsVector *E=NULL;
	PdsVector *Y=NULL;
	PdsDfReal d,x,y,e;
	PdsDfNatural n,N;

	PdsDfNatural M;
	PdsDfReal Mhu;

	FILE *fd=NULL;
	char DATAFILE[]="datos.txt";
	char OCTAVEFILE[]="plotinoctave.m";
	FILE *fdh=NULL;
	char HFILE[]="hdatos.txt";
	char OCTAVEHFILE[]="plothinoctave.m";
	N=2*1024;

	D=pds_vector_new(N);
	X=pds_vector_new(N);
	Y=pds_vector_new(N);
	E=pds_vector_new(N);


	///////////////////////////////////////////////////////////////////////
	M=24;		Mhu=0.01;
	FirLms=pds_fir_lms_new(Mhu,M);
	y=0;	e=0;
	for(n=0;n<N;n++)
	{
		x=1.0 + sin(M_PI*n/32.0)            + sin(12*M_PI*n/32.0) + sin(20*M_PI*n/32.0+M_PI/16.0);
		d=	sin(M_PI*n/32.0+M_PI/128.0) + sin(12*M_PI*n/32.0) ;
		
		D->V[n]=d;
		X->V[n]=x;
	}

	pds_fir_lms_evaluate_vector(FirLms,D,X,E,Y);

	fdh=fopen(HFILE,"w");
	if(fdh==NULL)	return EXIT_FAILURE;
	pds_vector_fprintf(FirLms->Fir->h,fdh);
	fclose(fdh);

	pds_fir_lms_free(FirLms);
	///////////////////////////////////////////////////////////////////////

	fd=fopen(DATAFILE,"w");
	if(fd==NULL)	return EXIT_FAILURE;
	pds_vector_fprintf(D,fd);
	pds_vector_fprintf(X,fd);
	pds_vector_fprintf(E,fd);
	pds_vector_fprintf(Y,fd);
	fclose(fd);

	pds_octave_plot_lin(OCTAVEFILE,DATAFILE);
	pds_octave_plot_lin_fft(OCTAVEHFILE,HFILE);

	pds_vector_free(D);	
	pds_vector_free(X);	
	pds_vector_free(E);	
	pds_vector_free(Y);	

	return EXIT_SUCCESS;
}
