/*
 * pdsdic2d.h
 * 
 * Copyright 2018 Fernando Pujaico Rivera <fernando.pujaico.rivera@gmail.com>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 * 
 */

/** \file pdsdic2d.h
 *  \author Fernando Pujaico Rivera
 *  \date 01-04-2011
 *  \brief Funciones que trabajan con Digital image crrelation in 2D.
 *  
 *  <br>Existen funciones para buscar una región entre dos imágenes representadas como matriz<br>
 *  <center>
 *  \image html dic2d.png "Digital image correlation"
 *  \image html dic2d2.png "Digital image correlation"
 *  </center>
 */

#ifndef __PDSDIC2D_H__
#define __PDSDIC2D_H__

#ifdef __cplusplus
extern "C" {
#endif 

#include <stdio.h>
#include <stdlib.h>
#include <pds/pdsra.h>

/** \defgroup PdsDic2DGroup Módulo PdsDic2D.
 *  \brief Funciones que trabajan con Digital Image Correlation in 2D.
 *  
 *  <br>Existen funciones para buscar una región entre dos imágenes representadas como matriz<br>
 *  <center>
 *  \image html dic2d.png "Digital image correlation"
 *  \image html dic2d2.png "Digital image correlation"
 *  </center>
 *  
 * Informacion adicional puede ser encontrada en @cite tutorialmatvec
 * @{
 */


#ifndef PDS_MIN
/*!
  \def PDS_MIN(a,b)
  Valor retornado o valor mínimo entre dos números.
*/
    #define PDS_MIN(a,b) \
       ({ typeof (a) _a = (a); \
           typeof (b) _b = (b); \
         (_a < _b) ? _a : _b; })

#endif

#ifndef PDS_DIC2D_NO_MOVED_THRESHOLD
/*!
  \def PDS_DIC2D_NO_MOVED_THRESHOLD
  Valor Umbral de correlación para declarar que el objeto no se ha movido.
*/
	#define PDS_DIC2D_NO_MOVED_THRESHOLD 0.99
#endif

#ifndef PDS_DIC2D_MATCH_THRESHOLD
/*!
  \def PDS_DIC2D_MATCH_THRESHOLD
  Valor Umbral de correlación para declarar un match.
*/
	#define PDS_DIC2D_MATCH_THRESHOLD 0.75
#endif

#ifndef PDS_DIC2D_SEARCH_PIXEL_BY_PIXEL
/*!
  \def PDS_DIC2D_SEARCH_PIXEL_BY_PIXEL
  Valor de el paso de la búsqueda.
*/
	#define PDS_DIC2D_SEARCH_PIXEL_BY_PIXEL 1
#endif

#ifndef PDS_OK
/*!
  \def PDS_OK
  Valor retornado en caso de exito, este es un valor diferente que cero.
*/
	#define PDS_OK 1
#endif

#ifndef PDS_WRONG
/*!
  \def PDS_WRONG
  Valor retornado en caso de fracaso, este valor es igual a cero.
*/
	#define PDS_WRONG 0
#endif


typedef enum
{
    PDS_DIC2D_NOFOUND=-1,
    PDS_DIC2D_ERROR=0,
    PDS_DIC2D_FOUND=1
}PdsDic2DOutput;

/*! \struct PdsDic2D
 *  \brief La estructura tipo  PdsDic2D .
 *  Esta estructura contiene dos matrices de Nlin lineas y Ncol columnas.
 *  Para usar incluir pds/pdsdic2d.h.
 *  \ingroup PdsDic2DGroup
 *  \author Fernando Pujaico Rivera
 */
typedef struct 
{
	/*! Primera matriz*/
	PdsMatrix *M0;
	/*! Segunda matriz */
	PdsMatrix *M1;

	/*! Especifica si regiones con desvío padrón 0 son aceptadas en las búsquedas, 
        por defecto descartad con valor PDS_WRONG. */    
	int zero_std_search;
	/*! Valor mínimo de correlación para aceptar un match, por defecto 
        PDS_DIC2D_MATCH_THRESHOLD */
	PdsRaReal match_threshold;
	/*! Valor mínimo de correlación para aceptar que no hubo disloca-miento de 
        la región, por defecto PDS_DIC2D_NO_MOVED_THRESHOLD*/
	PdsRaReal no_moved_threshold;

	/*! Paso en pixels, para la búsqueda de regiones coincidentes. Por defecto
        es PDS_DIC2D_SEARCH_PIXEL_BY_PIXEL */
	PdsRaNatural search_step_size;
	/*! Distancia máxima en pixels, para la búsqueda de regiones coincidentes. */
	PdsRaNatural search_max_length;
	/*! Variable que indica si información adicional deberá ser mostrada */
	unsigned char debug;
	/*! Ultima máxima correlación si hubo match */
	PdsRaReal last_match_corr;
}PdsDic2D;


/** @name pds_dic2d_new_<methods>
 *  Creando nuevas estructuras dic 2d
 * @{
 */

/** \fn PdsDic2D *pds_dic2d_new(PdsRaNatural Nlin,PdsRaNatural Ncol)
 *  \brief Crea una estructura de tipo PdsDic2D, internamente la estructura 
 *  contiene dos matrices con Nlin lineas y Ncol columnas, con valores iniciados 
 *  con ceros.
 *  \param[in] Nlin Es el número de lineas de las matrices.
 *  \param[in] Ncol Es el número de columnas de las matrices.
 *  \return Un puntero a la estructura de tipo PdsDic2D o NULL en caso de error.
 *  \ingroup PdsDic2DGroup
 */
PdsDic2D *pds_dic2d_new(PdsRaNatural Nlin,PdsRaNatural Ncol);

/** \fn PdsDic2D *pds_dic2d_new_from_matrices(const PdsMatrix *Mat0,const PdsMatrix *Mat1)
 *  \brief Crea una estructura de tipo PdsDic2D, internamente la estructura 
 *  contiene una copia de las dos matrices de entrada.
 *  Los tamanho de las matrices deben ser iguales.
 *  \param[in] Mat0 Primera matriz, matriz de fuente.
 *  \param[in] Mat1 Segunda matriz, matriz de búsqueda.
 *  \return Un puntero a la estructura de tipo PdsDic2D o NULL en caso de error.
 *  \ingroup PdsDic2DGroup
 */
PdsDic2D *pds_dic2d_new_from_matrices(const PdsMatrix *Mat0,const PdsMatrix *Mat1);

//@}

/** @name pds_dic2d_<tracking_region>
 *  Obteniendo datos de correlación
 * @{
 */

/** \fn int pds_dic2d_tracking_region( PdsDic2D *DIC, PdsRegionRect Rin, PdsRegionRect *Rout)
 *  \brief Devuelve la región coincidente en la segunda matriz.
 *  \param[in] DIC La estructura en consulta.
 *  \param[in] Rin La región de la primera matriz que se buscara en la segunda.
 *  \param[out] Rout La región encontrada en la segunda matriz.
 *  Esta variable no es alterada si el objeto no fue encontrado.
 *  \return Puede retornar:
 *  PDS_DIC2D_ERROR cuando hubo un error lectura de memoria.
 *  PDS_DIC2D_FOUND Si el objeto fue encontrado.
 *  PDS_DIC2D_NOFOUND Si el objeto no fue encontrado.
 *  \ingroup PdsDic2DGroup
 */
int pds_dic2d_tracking_region( PdsDic2D *DIC, PdsRegionRect Rin, PdsRegionRect *Rout);

/** \fn PdsMatrix *pds_dic2d_pearson_correlations_matrix(const PdsDic2D *DIC, PdsRegionRect Ro)
 *  \brief Retorna una matriz con las correlaciones de la región Ro en la matriz M0 de DIC,
 *  con una vecindad rectangular con brazo de acción search_max_length y un paso 
 *  de búsqueda de search_step_size, en la matriz M1.
 *
 *  Así, la función retorna una matriz cuadrada de 1+2(search_max_length/search_step_size)
 *  elementos por lado.
 *  \param[in] DIC La estructura en consulta.
 *  \param[in] Ro La región de la matriz M0.
 *  \return A matriz de correlaciones si todo fue bien o NULL si no (ej: DIC==NULL). 
 *  \ingroup PdsDic2DGroup
 */
PdsMatrix *pds_dic2d_pearson_correlations_matrix(const PdsDic2D *DIC, PdsRegionRect Ro);

/** \fn int pds_dic2d_is_the_region_moved(PdsDic2D *DIC, PdsRegionRect Ro)
 *  \brief Indica si la región se ha movido. 
 *
 *  Es cosiderado que hubo movimiento si la correlación de la regiones en M0 y M1
 *  es menor que no_moved_threshold.
 *  \param[in] DIC La estructura en consulta.
 *  \param[in] Ro La región de la matriz M0.
 *  \return PDS_OK si se movimiento o PDS_WRONG si no se movimiento o si hubo algun
 *  error en la consulta. 
 *  \ingroup PdsDic2DGroup
 */
int pds_dic2d_is_the_region_moved(PdsDic2D *DIC, PdsRegionRect Ro);

//@}

/** @name pds_dic2d_<write_matrices>
 *  Estableciendo datos
 * @{
 */

/** \fn int pds_dic2d_push_matrix(PdsDic2D *DIC,const PdsMatrix *Mat)
 *  \brief Actualiza los valores de las matrices en la estructura de tipo PdsDic2D, 
 *  internamente la matriz M1->M0 y mat->M1.
 *  Los tamanho de las matrices deben ser iguales.
 *  \param[in] DIC La estructura en consulta.
 *  \param[in] Mat Matriz a copiar.
 *  \return PDS_OK si todo fue bien o PDS_WRONG si no (ejemplo: DIC==NULL MAT==NULL). 
 *  \ingroup PdsDic2DGroup
 */
int pds_dic2d_push_matrix(PdsDic2D *DIC,const PdsMatrix *Mat);

//@}

/** @name pds_dic2d_set_<dic_data_configuration>
 *  Estableciendo datos
 * @{
 */

/** \fn int pds_dic2d_set_zero_std_search(PdsDic2D *DIC, int zero_std_search)
 *  \brief Establece si una región con desvío padrón cero serán buscadas.
 *  Por defecto está variable está deshabilitada con PDS_WRONG, se habilita con
 *  PDS_OK.
 *  \param[in] DIC La estructura en consulta. 
 *  \param[in] zero_std_search Habilitador, des-habilitador, si se busca regiones
 *  con desvío padrón cero.
 *  Este valor no es aceptado si es diferente de PDS_OK o PDS_WRONG.
 *  \return PDS_OK si todo fue bien o PDS_WRONG si no (ej: DIC==NULL o parámetro fuera de rango). 
 *  \ingroup PdsDic2DGroup
 */
int pds_dic2d_set_zero_std_search(PdsDic2D *DIC, int zero_std_search);

/** \fn int pds_dic2d_set_no_moved_threshold(PdsDic2D *DIC, PdsRaReal no_moved_threshold)
 *  \brief Establece el umbral para aceptar que la región no se ha movido.
 *  Por defecto este umbral es PDS_DIC2D_NO_MOVED_THRESHOLD.
 *  \param[in] DIC La estructura en consulta. 
 *  \param[in] no_moved_threshold Umbral para aceptar que la región no se ha movido.
 *  Este valor no es aceptado si es mayor que 1.0 o menor de 0.0.
 *  \return PDS_OK si todo fue bien o PDS_WRONG si no (ej: DIC==NULL o parámetro fuera de rango). 
 *  \ingroup PdsDic2DGroup
 */
int pds_dic2d_set_no_moved_threshold(PdsDic2D *DIC, PdsRaReal no_moved_threshold);

/** \fn int pds_dic2d_set_match_threshold(PdsDic2D *DIC, PdsRaReal match_threshold)
 *  \brief Establece el umbral para aceptar una coincidencia en un match.
 *  Por defecto este umbral es PDS_DIC2D_MATCH_THRESHOLD.
 *  \param[in] DIC La estructura en consulta.
 *  \param[in] match_threshold Umbral de coincidencia.
 *  Este valor no puede ser mayor que 1.0 o menor de -1.0.
 *  \return PDS_OK si todo fue bien o PDS_WRONG si no (ej: DIC==NULL). 
 *  \ingroup PdsDic2DGroup
 */
int pds_dic2d_set_match_threshold(PdsDic2D *DIC, PdsRaReal match_threshold);

/** \fn int pds_dic2d_set_search_step_size(PdsDic2D *DIC, PdsRaNatural search_step_size)
 *  \brief Establece el paso en pixels, para la búsqueda de regiones coincidentes.
 *  Por defecto es PDS_DIC2D_SEARCH_PIXEL_BY_PIXEL.
 *  \param[in] DIC La estructura en consulta.
 *  \param[in] search_step_size Paso en pixels, para la busqueda de regiones coincidentes.
 *  El valor cero no es aceptado.
 *  \return PDS_OK si todo fue bien o PDS_WRONG si no (ej: DIC==NULL). 
 *  \ingroup PdsDic2DGroup
 */
int pds_dic2d_set_search_step_size(PdsDic2D *DIC, PdsRaNatural search_step_size);

/** \fn int pds_dic2d_set_search_max_length(PdsDic2D *DIC, PdsRaNatural search_max_length)
 *  \brief Establece la distancia máxima en pixels, para la busqueda de regiones coincidentes.
 *  \param[in] DIC La estructura en consulta.
 *  \param[in] search_max_length La distancia máxima en pixels, para la búsqueda de regiones coincidentes.
 *  El valor cero no es aceptado.
 *  \return PDS_OK si todo fue bien o PDS_WRONG si no (ej: DIC==NULL). 
 *  \ingroup PdsDic2DGroup
 */
int pds_dic2d_set_search_max_length(PdsDic2D *DIC, PdsRaNatural search_max_length);

/** \fn int pds_dic2d_set_debug(PdsDic2D *DIC, unsigned char debug)
 *  \brief Establece se  debug esta habilitado, se habilitado se mostrara mas
 *  infrmacion por pantalla.
 *  \param[in] DIC La estructura en consulta.
 *  \param[in] debug Habilita con valor 1, deshabilita con valor diferente de 1.
 *  \return PDS_OK si todo fue bien o PDS_WRONG si no (ej: DIC==NULL). 
 *  \ingroup PdsDic2DGroup
 */
int pds_dic2d_set_debug(PdsDic2D *DIC, unsigned char debug);

/** \fn int pds_dic2d_set_last_match_corr(PdsDic2D *DIC, PdsRaReal last_match_corr)
 *  \brief Establece el valor de la correlación en el ultimo match.
 *  \param[in] DIC La estructura en consulta.
 *  \param[in] last_match_corr Valor de la correlación en el ultimo match.
 *  \return PDS_OK si todo fue bien o PDS_WRONG si no (ej: DIC==NULL). 
 *  \ingroup PdsDic2DGroup
 */
int pds_dic2d_set_last_match_corr(PdsDic2D *DIC, PdsRaReal last_match_corr);

//@}

/** @name pds_dic2d_get_<dic_data_configuration>
 *  Obteniendo datos de matrices
 * @{
 */

/** \fn int pds_dic2d_get_nlines(const PdsDic2D *DIC, PdsRaNatural *Nlines)
 *  \brief Devuelve el número de lineas en las matrices dentro de la estructura DIC.
 *  \param[in] DIC La estructura en consulta.
 *  \param[out] Nlines En donde se guardará el número de lineas de las matrices.
 *  \return PDS_OK si todo fue bien o PDS_WRONG si no (ej: DIC==NULL). 
 *  \ingroup PdsDic2DGroup
 */
int pds_dic2d_get_nlines(const PdsDic2D *DIC, PdsRaNatural *Nlines);

/** \fn int pds_dic2d_get_ncolumns(const PdsDic2D *DIC, PdsRaNatural *Ncolumns)
 *  \brief Devuelve el número de columnas en las matrices dentro de la estructura DIC.
 *  \param[in] DIC La estructura en consulta.
 *  \param[out] Ncolumns En donde se guardará el número de columnas de las matrices.
 *  \return PDS_OK si todo fue bien o PDS_WRONG si no (ej: DIC==NULL). 
 *  \ingroup PdsDic2DGroup
 */
int pds_dic2d_get_ncolumns(const PdsDic2D *DIC, PdsRaNatural *Ncolumns);

/** \fn int pds_dic2d_get_no_moved_threshold(const PdsDic2D *DIC, PdsRaReal *no_moved_threshold)
 *  \brief Devuelve el umbral para aceptar que el objeto no se ha movido.
 *  \param[in] DIC La estructura en consulta.
 *  \param[out] no_moved_threshold Umbral de coincidencia.
 *  \return PDS_OK si todo fue bien o PDS_WRONG si no (ej: DIC==NULL). 
 *  \ingroup PdsDic2DGroup
 */
int pds_dic2d_get_no_moved_threshold(const PdsDic2D *DIC, PdsRaReal *no_moved_threshold);

/** \fn int pds_dic2d_get_match_threshold(const PdsDic2D *DIC, PdsRaReal *match_threshold)
 *  \brief Devuelve el umbral para aceptar una coincidencia en un match.
 *  \param[in] DIC La estructura en consulta.
 *  \param[out] match_threshold Umbral de coincidencia.
 *  \return PDS_OK si todo fue bien o PDS_WRONG si no (ej: DIC==NULL). 
 *  \ingroup PdsDic2DGroup
 */
int pds_dic2d_get_match_threshold(const PdsDic2D *DIC, PdsRaReal *match_threshold);

/** \fn int pds_dic2d_get_search_step_size(const PdsDic2D *DIC, PdsRaNatural *search_step_size)
 *  \brief Devuelve el paso en pixels, para la búsqueda de regiones coincidentes.
 *  Por defecto es PDS_DIC2D_SEARCH_PIXEL_BY_PIXEL.
 *  \param[in] DIC La estructura en consulta.
 *  \param[out] search_step_size Paso en pixels, para la búsqueda de regiones coincidentes.
 *  \return PDS_OK si todo fue bien o PDS_WRONG si no (ej: DIC==NULL). 
 *  \ingroup PdsDic2DGroup
 */
int pds_dic2d_get_search_step_size(const PdsDic2D *DIC, PdsRaNatural *search_step_size);

/** \fn int pds_dic2d_get_search_max_length(const PdsDic2D *DIC, PdsRaNatural *search_max_length)
 *  \brief Devuelve la distancia máxima en pixels, para la búsqueda de regiones coincidentes.
 *  \param[in] DIC La estructura en consulta.
 *  \param[out] search_max_length La distancia máxima en pixels, para la búsqueda de regiones coincidentes.
 *  \return PDS_OK si todo fue bien o PDS_WRONG si no (ej: DIC==NULL). 
 *  \ingroup PdsDic2DGroup
 */
int pds_dic2d_get_search_max_length(const PdsDic2D *DIC, PdsRaNatural *search_max_length);

/** \fn int pds_dic2d_get_zero_std_search(const PdsDic2D *DIC, int *zero_std_search)
 *  \brief Devuelve si esta habilitado que una región con desvío padrón cero sea buscada.
 *  Por defecto está variable está deshabilitada con PDS_WRONG, se habilita con
 *  PDS_OK.
 *  \param[in] DIC La estructura en consulta. 
 *  \param[out] zero_std_search Si se busca regiones con desvío padrón cero.
 *  Este valor es PDS_OK o PDS_WRONG.
 *  \return PDS_OK si todo fue bien o PDS_WRONG si no (ej: DIC==NULL o zero_std_search==NULL). 
 *  \ingroup PdsDic2DGroup
 */
int pds_dic2d_get_zero_std_search(const PdsDic2D *DIC, int *zero_std_search);

//@}

/** @name pds_dic2d_<free>
 *  Liberando matrices
 * @{
 */

/** \fn void pds_dic2d_free(PdsDic2D *DIC)
 *  \brief Libera la estructura de tipo puntero PdsDic2D.
 *  \param[in,out] DIC La estructura a liberar.
 *  \return No retorna valor.
 *  \ingroup PdsDic2DGroup
 */
void pds_dic2d_free(PdsDic2D *DIC);

/** \fn void pds_dic2d_destroy(PdsDic2D **DIC)
 *  \brief Libera la estructura de tipo puntero PdsDic2D, y limpia el puntero con NULL.
 *  \param[in,out] DIC La estructura a liberar y limpiar.
 *  \return No retorna valor.
 *  \ingroup PdsDic2DGroup
 */
void pds_dic2d_destroy(PdsDic2D **DIC);

//@}

/**
 * @}
 */

#ifdef __cplusplus
}
#endif 

#endif

