/*
 * pdsifft.c
 * 
 * Copyright 2011 Fernando Pujaico Rivera <fernando.pujaico.rivera@gmail.com>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 * 
 */


#include <pds/pdsifft.h>
#include <pds/pdscvector.h>
#include <pds/pdsvector.h>
#include <pds/pdscn.h>
#include <stdio.h>
#include <math.h>


////////////////////////////////////////////////////////////////////////////////
////  Trabajando con PdsIFft                                                ////
////////////////////////////////////////////////////////////////////////////////


/** \fn PdsIFft *pds_ifft_new(PdsFtNatural *N)
 *  \brief Crea una estructura de tipo PdsIFft, para generar una IFFT de N puntos.
 *  Si N no es potencia de 2, no da error, y se crea una estructura para una IFFT
 *  con un  N1, que si es potencia de dos y mayor a N, (N1>=N). El valor de N mínimo
 *  es N=2. El nuevo valor N1 será cargado a N. N=N1.
 *  \param[in,out] N Es el número de elementos de la IFFT.
 *  \return Un puntero a una estructura de tipo PdsIFft. En caso de error devuelve
 *  NULL.
 *  \ingroup PdsIFftGroup
 */
PdsIFft *pds_ifft_new(PdsFtNatural *N)
{
	PdsIFft *IFFT=NULL;
	PdsFtNatural i,r,n;

	if(*N<=1)	return NULL;

	for(i=0,r=(*N-1);r>0;i++)	r=r/2;
	n=i;
	for(i=0,r=1;i<n;i++)		r=r*2;

	*N=r;
	
	IFFT=(PdsIFft *)calloc(1,sizeof(PdsIFft));
	if(IFFT==NULL)	return NULL;

	IFFT->N=r;
	IFFT->W=pds_cvector_new(r/2);
	if(IFFT->W==NULL)	
	{	
		pds_cvector_free(IFFT->W);
		return NULL;
	}
	
	for(i=0;i<(IFFT->W->Nel);i++)
	IFFT->W->V[i]=pds_complex_cis((-2.0*M_PI*i)/(IFFT->N*1.0));
	
	return IFFT;
}


/** \fn int pds_ifft_complex_evaluate(const PdsIFft *IFFT,PdsCVector *Out,const PdsCVector *In)
 *  \brief Evalua la inversa de la transformada rápida de fourier a un vetor complejo. El tamaño 
 *  del vector Out debe ser igual que el número de puntos de la IFFT, y mayor que
 *  el número de elementos de In, La diferencia se llenará con ceros.
 *  \param[in] IFFT La estructura a una IFFT de N puntos.
 *  \param[out] Out El vector complejo con la IFFT de N puntos.
 *  \param[in] In El vector de un tamaño menor o igual a N, al que se le desea 
 *  aplicar la IFFT.
 *  \return TRUE si todo fue bien o FALSE si no.
 *  \ingroup PdsIFftGroup
 */
int pds_ifft_complex_evaluate(const PdsIFft *IFFT,PdsCVector *Out,const PdsCVector *In)
{
	PdsFtNatural k,khat,bit;
	PdsFtNatural n,m;
	PdsComplex y,z;

	if(IFFT==NULL)	return FALSE;
	if(Out==NULL)	return FALSE;
	if(In==NULL)	return FALSE;
	if(Out->Nel!=IFFT->N)	return FALSE;
	if(Out->Nel<In->Nel)	return FALSE;

	////////////////////////// Intercambiando Bits /////////////////////////
	for (k=0, khat=0; k<(IFFT->N); k++)
	{
		if(k<(In->Nel))	
		{
			Out->V[khat].Real= In->V[k].Real;
			Out->V[khat].Imag=-In->V[k].Imag;
		}
		else
		{
			Out->V[khat].Real=0.0;
			Out->V[khat].Imag=0.0;
		}

		for (bit=(IFFT->N/2); (khat & bit)!=0; bit >>=1)
		khat ^= bit;
		khat ^= bit;
	}
	
	/////////////////////////// Calculado el IFFT //////////////////////////
	for (n=2; n<=(IFFT->N); n <<= 1)
	{
		for (m=0; m<(IFFT->N); m+=n)
		{
			for (k=0; k<n/2; k++)
			{
				y=Out->V[m+k];
				z=_CMULC(Out->V[m+k+n/2],IFFT->W->V[(k*IFFT->N)/n]);

				Out->V[m+k]=_CADDC(y,z);
				Out->V[m+k+n/2]=_CSUBC(y,z);
			}
		}
	}

	for(k=0;k<IFFT->N;k++)
	{
		Out->V[k].Real=Out->V[k].Real/IFFT->N;
		Out->V[k].Imag=Out->V[k].Imag/IFFT->N;
	}
	pds_cvector_conjugate(Out);

	return TRUE;
}


/** \fn void pds_ifft_free(PdsIFft *IFFT)
 *  \brief Libera una estructura de tipo puntero PdsIFft.
 *  \param[in,out] IFFT La IFFT a liberar.
 *  \return No retorna valor.
 *  \ingroup PdsIFftGroup
 */
void pds_ifft_free(PdsIFft *IFFT)
{
	if(IFFT!=NULL)
	{
		pds_cvector_free(IFFT->W);
		free(IFFT);
	}
}


/** \fn void pds_ifft_destroy(PdsIFft **IFFT)
 *  \brief Libera una estructura de tipo puntero PdsIFft, y carga a la estructura con NULL.
 *  \param[in,out] IFFT La IFFT a liberar.
 *  \return No retorna valor.
 *  \ingroup PdsIFftGroup
 */
void pds_ifft_destroy(PdsIFft **IFFT)
{
	if((*IFFT)!=NULL)
	{
		pds_cvector_free((*IFFT)->W);
		free(*IFFT);
		(*IFFT)=NULL;
	}
}

