/*
 * pdsifft.h
 * 
 * Copyright 2011 Fernando Pujaico Rivera <fernando.pujaico.rivera@gmail.com>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 * 
 */

/** \file pdsifft.h
 *  \author Fernando Pujaico Rivera
 *  \date 10-05-2011
 *  \brief Funciones que trabajan con IFft.
 *  
 *  Estas funciones trabajan con una IFft así.<br>
 *  \image html pdsifft.png "Inversa de la Transformada Rápida de Fourier"
   \f{eqnarray*}
        x[n]    &=& \frac{1}{N}\sum_{i=0}^{N-1}X[i]W_N^{-ik} \\ 
        W_N^{r} &=& e^{-j  2 \pi r / N }\\
	x[n] &=& IFFT\{X[k]\}=\frac{1}{N}FFT\{X^*[k]\}^*
   \f}
 */

#ifndef __PDSIFFT_H__
#define __PDSIFFT_H__

#ifdef __cplusplus
extern "C" {
#endif 


#include <stdio.h>

#include <pds/pdsftglobal.h>
#include <pds/pdscvector.h>
#include <pds/pdsvector.h>

#ifndef TRUE
	#define TRUE 1
#endif

#ifndef FALSE
	#define FALSE 0
#endif


/** \defgroup PdsIFftGroup Funciones del módulo PdsIFft.
 * @{
 */


/*! \struct PdsIFft
 *  \brief La estructura tipo  PdsIFft .
 *  Esta estructura genera una Inversa de la Transformada Rápida de Fourier de N puntos.
 *  Para usar incluir pds/pdsft.h.
 *  \image html pdsifft.png "Inversa de la Transformada Rápida de Fourier"
   \f{eqnarray*}
        x[n]    &=& \frac{1}{N}\sum_{i=0}^{N-1}X[i]W_N^{-ik} \\ 
        W_N^{r} &=& e^{-j  2 \pi r / N }\\
	x[n] &=& IFFT\{X[k]\}=\frac{1}{N}FFT\{X^*[k]\}^*
   \f}
 *  \ingroup PdsIFftGroup
 *  \author Fernando Pujaico Rivera
 */
typedef struct 
{
	/*! Un arreglo de N/2 elementos con W[k]=CIS(-2*M_PI*k/N), k en [0,N/2). */
	PdsCVector *W;
	/*! Número de elementos, N es potencia de 2, 2^r=N  */
	PdsFtNatural N;
}PdsIFft;



/** \fn PdsIFft *pds_ifft_new(PdsFtNatural *N)
 *  \brief Crea una estructura de tipo PdsIFft, para generar una IFFT de N puntos.
 *  Si N no es potencia de 2, no da error, y se crea una estructura para una IFFT
 *  con un  N1, que si es potencia de dos y mayor a N, (N1>=N). El valor de N mínimo
 *  es N=2. El nuevo valor N1 será cargado a N. N=N1.
 *  \param[in,out] N Es el número de elementos de la IFFT.
 *  \return Un puntero a una estructura de tipo PdsIFft. En caso de error devuelve
 *  NULL.
 *  \ingroup PdsIFftGroup
 */
PdsIFft *pds_ifft_new(PdsFtNatural *N);


/** \fn int pds_ifft_complex_evaluate(const PdsIFft *IFFT,PdsCVector *Out,const PdsCVector *In)
 *  \brief Evalua la inversa de la transformada rápida de fourier a un vetor complejo. El tamaño 
 *  del vector Out debe ser igual que el número de puntos de la IFFT, y mayor que
 *  el número de elementos de In, La diferencia se llenará con ceros.
 *  \param[in] IFFT La estructura a una IFFT de N puntos.
 *  \param[out] Out El vector complejo con la IFFT de N puntos.
 *  \param[in] In El vector de un tamaño menor o igual a N, al que se le desea 
 *  aplicar la IFFT.
 *  \return TRUE si todo fue bien o FALSE si no.
 *  \ingroup PdsIFftGroup
 */
int pds_ifft_complex_evaluate(const PdsIFft *IFFT,PdsCVector *Out,const PdsCVector *In);


/** \fn void pds_ifft_free(PdsIFft *IFFT)
 *  \brief Libera una estructura de tipo puntero PdsIFft.
 *  \param[in,out] IFFT La IFFT a liberar.
 *  \return No retorna valor.
 *  \ingroup PdsIFftGroup
 */
void pds_ifft_free(PdsIFft *IFFT);


/** \fn void pds_ifft_destroy(PdsIFft **IFFT)
 *  \brief Libera una estructura de tipo puntero PdsIFft, y carga a la estructura con NULL.
 *  \param[in,out] IFFT La IFFT a liberar.
 *  \return No retorna valor.
 *  \ingroup PdsIFftGroup
 */
void pds_ifft_destroy(PdsIFft **IFFT);


/**
 * @}
 */

#ifdef __cplusplus
}
#endif 

#endif	/* __PDSIFFT_H__ */ 

