/*
 * pdsspectrograph.h
 * 
 * Copyright 2011 Fernando Pujaico Rivera <fernando.pujaico.rivera@gmail.com>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 * 
 */

/** \file pdsspectrograph.h
 *  \author Fernando Pujaico Rivera
 *  \date 10-05-2011
 *  \brief Funciones que trabajan con Fft.
 *  
 *  Estas funciones trabajan con una Fft así.<br>
 *  \image html pdsspectrograph.png "Spectrogram"
 *  \f[ \mathbf{STFT_m} \left \{ x[n] \right \} \equiv Out_m[k] = \sum_{n=0}^{N-1} x_m[n]w[n]e^{-j 2 \pi k \frac{n}{N}} \f]
 *  \f[ \mathbf{SPECTROGRAPH} \left \{ x[n] \right \} \equiv \mathbf{STFT^2} \left \{ x[n] \right \} \f]
 *  \image html graficaM.png "Spectrogram M=N/2=32 puntos de frecuencia y 3 puntos en el tiempo."
 */

#ifndef __PDSSPECTROGRAPH_H__
#define __PDSSPECTROGRAPH_H__

#ifdef __cplusplus
extern "C" {
#endif 


#include <stdio.h>

#include <pds/pdsftglobal.h>
#include <pds/pdscvector.h>
#include <pds/pdsvector.h>

/*! \defgroup PdsSpectrographGroup Funciones del módulo PdsSpectrograph.
 *  \brief Funciones para una SPECTROGRAPH (Spectrograma ).
 *
 *  \image html pdsspectrograph.png "Spectrogram"
 *  \f[ \mathbf{STFT_m} \left \{ x[n] \right \} \equiv Out_m[k] = \sum_{n=0}^{N-1} x_m[n]w[n]e^{-j 2 \pi k \frac{n}{N}} \f]
 *  \f[ \mathbf{SPECTROGRAPH} \left \{ x[n] \right \} \equiv \mathbf{STFT^2} \left \{ x[n] \right \} \f]
 *  \image html graficaM.png "Spectrogram M=N/2=32 puntos de frecuencia y 3 puntos en el tiempo."
 * @{
 */

#ifndef TRUE
/** \def TRUE
 *  Define la palabra TRUE equivalente con 1.
 *  \ingroup PdsSpectrographGroup
 */
	#define TRUE 1
#endif

#ifndef FALSE
/*! \def FALSE
 *  Define la palabra FALSE equivalente con 0.
 *  \ingroup PdsSpectrographGroup
 */
	#define FALSE 0
#endif

/*! \struct PdsSpectrograph
 *  \brief La estructura tipo  PdsSpectrograph .
 *  Esta estructura genera un SPECTROGRAPH de L puntos
 *  de tiempo y M puntos de frecuencia
 *  Para usar incluir pds/pdsspectrograph.h.
 *  \image html pdsspectrograph.png "Spectrogram"
 *  \f[ \mathbf{STFT_m} \left \{ x[n] \right \} \equiv Out_m[k] = \sum_{n=0}^{N-1} x_m[n]w[n]e^{-j 2 \pi k \frac{n}{N}} \f]
 *  \f[ \mathbf{SPECTROGRAPH} \left \{ x[n] \right \} \equiv \mathbf{STFT^2} \left \{ x[n] \right \} \f]
 *  \image html graficaM.png "Spectrogram M=N/2=32 puntos de frecuencia y 3 puntos en el tiempo."
 *  \ingroup PdsSpectrographGroup
 *  \author Fernando Pujaico Rivera
 */
typedef struct 
{
	/*! Número de puntos de tiempo del bloque SPECTROGRAPH.  */
	PdsFtNatural L;

	/*! Vector complejo de entrada de N elementos del bloque FFT. */
	PdsCVector *X;
	/*! Vector de pesos de N elementos para la entrada X. */
	PdsVector *W;
	/*! Vector complejo de salida de N elementos del bloque FFT. */
	PdsCVector *Y;
	/*! Vector modulo cuadrado de Y. */
	PdsVector *V;
	/*! El doble del número de puntos de frecuencia del bloque SPECTROGRAPH.  
	 *  Este valor tiene que ser par y mayor o igual que L
	 */
	PdsFtNatural N;

	/*! El número de puntos de frecuencia del bloque SPECTROGRAPH.  
	 */
	PdsFtNatural M;

	/*! Estructura FFT de N elementos. */
	PdsFft	*FFT;
}PdsSpectrograph;

/** \fn PdsSpectrograph *pds_spectrograph_new(PdsFtNatural L, PdsFtNatural *N)
 *  \brief Crea una estructura de tipo PdsSpectrograph, para generar un SPECTROGRAPH de L puntos
 *  de tiempo y M puntos de frecuencia, (N=2M) >=L.
 *  Si N no es potencia de 2, no da error, y se crea una estructura para una SPECTROGRAPH
 *  con un  N1, que si es potencia de dos y mayor a N1, (N1>=N). El valor de M mínimo
 *  es M=2 y N=4. El nuevo valor N1 afectara a M. M=N1/2.
 *  El vector W (ventana) se incia con todos los elementos con 1.0 .
 *  \param[in] L Es el número de muestras de tiempo tomados por el SPECTROGRAPH 
 *  para generar la FFT de N puntos.
 *  \param[in,out] N Es el número de elementos de fft del SPECTROGRAPH. La matriz
 *  de salida del espectrograma debera tener M=N/2 lineas.
 *  \return Un puntero a una estructura de tipo PdsSpectrograph. En caso de error devuelve
 *  NULL.
 *  \ingroup PdsSpectrographGroup
 */
PdsSpectrograph *pds_spectrograph_new(PdsFtNatural L, PdsFtNatural *N);


/** \fn int pds_spectrograph_rectangular_window(PdsSpectrograph *SPECTROGRAPH)
 *  \brief Copia una ventana rectangular a W de la SPECTROGRAPH.
 *  \param[in,out] SPECTROGRAPH La estructura a una SPECTROGRAPH de N puntos.
 *  \return TRUE si todo fue bien o FALSE si no. (Ejem SPECTROGRAPH==NULL)
 *  \ingroup PdsSpectrographGroup
 */
int pds_spectrograph_rectangular_window(PdsSpectrograph *SPECTROGRAPH);


/** \fn int pds_spectrograph_hann_window(PdsSpectrograph *SPECTROGRAPH)
 *  \brief Copia una ventana de Hann a W de la SPECTROGRAPH.
 *  \param[in,out] SPECTROGRAPH La estructura a una SPECTROGRAPH de N puntos.
 *  \return TRUE si todo fue bien o FALSE si no. (Ejem SPECTROGRAPH==NULL)
 *  \ingroup PdsSpectrographGroup
 */
int pds_spectrograph_hann_window(PdsSpectrograph *SPECTROGRAPH);


/** \fn int pds_spectrograph_hamming_window(PdsSpectrograph *SPECTROGRAPH)
 *  \brief Copia una ventana de Hamming a W de la SPECTROGRAPH.
 *  \param[in,out] SPECTROGRAPH La estructura a una SPECTROGRAPH de N puntos.
 *  \return TRUE si todo fue bien o FALSE si no. (Ejem SPECTROGRAPH==NULL)
 *  \ingroup PdsSpectrographGroup
 */
int pds_spectrograph_hamming_window(PdsSpectrograph *SPECTROGRAPH);


/** \fn int pds_spectrograph_gauss_window(PdsSpectrograph *SPECTROGRAPH, PdsFtReal Sigma)
 *  \brief Copia una ventana de Gauss a W de la SPECTROGRAPH.
 *  \param[in,out] SPECTROGRAPH La estructura a una SPECTROGRAPH de N puntos.
 *  \param[in] Sigma Valor de sigma debe ser <= 0.5, de lo contrario da error.
 *  \return TRUE si todo fue bien o FALSE si no. (Ejem SPECTROGRAPH==NULL)
 *  \ingroup PdsSpectrographGroup
 */
int pds_spectrograph_gauss_window(PdsSpectrograph *SPECTROGRAPH, PdsFtReal Sigma);


/** \fn int pds_spectrograph_real_evaluate(PdsSpectrograph *SPECTROGRAPH,PdsMatrix *Out,const PdsVector *In)
 *  \brief Evalua la SPECTROGRAPH de un vector real. El número de lineas la matriz 
 *  Out debe ser igual N/2=M, Si el tamanho del vector de entrada es menor que
 *  L*Out->Ncol se rrelleno con ceros y se realiza el espectrograma, si es mayor,
 *  solo se usa lo que necesito y el resto se ignora.
 *  \param[in,out] SPECTROGRAPH La estructura a una SPECTROGRAPH de M puntos.
 *  \param[out] Out La matriz real con la SPECTROGRAPH de M puntos.
 *  \param[in] In El vector de entrada al que se le desea aplicar la SPECTROGRAPH.
 *  \return TRUE si todo fue bien o FALSE si no. (Ejem SPECTROGRAPH==NULL, In==NULL,
 *  Out==NULL o Out->Nlin!=N/2)
 *  \ingroup PdsSpectrographGroup
 */
int pds_spectrograph_real_evaluate(PdsSpectrograph *SPECTROGRAPH,PdsMatrix *Out,const PdsVector *In);


/** \fn int pds_spectrograph_complex_evaluate(PdsSpectrograph *SPECTROGRAPH,PdsMatrix *Out,const PdsCVector *In)
 *  \brief Evalua la SPECTROGRAPH de un vector complejo. El número de lineas la matriz 
 *  Out debe ser igual N/2=M, Si el tamanho del vector de entrada es menor que
 *  L*Out->Ncol se rrelleno con ceros y se realiza el espectrograma, si es mayor,
 *  solo se usa lo que necesito y el resto se ignora.
 *  \param[in,out] SPECTROGRAPH La estructura a una SPECTROGRAPH de M puntos.
 *  \param[out] Out La matriz real con la SPECTROGRAPH de M puntos.
 *  \param[in] In El vector de entrada al que se le desea aplicar la SPECTROGRAPH.
 *  \return TRUE si todo fue bien o FALSE si no. (Ejem SPECTROGRAPH==NULL, In==NULL,
 *  Out==NULL o Out->Nlin!=N/2)
 *  \ingroup PdsSpectrographGroup
 */
int pds_spectrograph_complex_evaluate(PdsSpectrograph *SPECTROGRAPH,PdsMatrix *Out,const PdsCVector *In);


/** \fn void pds_spectrograph_free(PdsSpectrograph *SPECTROGRAPH)
 *  \brief Libera una estructura de tipo puntero PdsSpectrograph.
 *  \param[in,out] SPECTROGRAPH La SPECTROGRAPH a liberar.
 *  \return No retorna valor.
 *  \ingroup PdsSpectrographGroup
 */
void pds_spectrograph_free(PdsSpectrograph *SPECTROGRAPH);


/** \fn void pds_spectrograph_destroy(PdsSpectrograph **SPECTROGRAPH)
 *  \brief Libera una estructura de tipo puntero PdsSpectrograph, y carga a la estructura con NULL.
 *  \param[in,out] SPECTROGRAPH La SPECTROGRAPH a liberar.
 *  \return No retorna valor.
 *  \ingroup PdsSpectrographGroup
 */
void pds_spectrograph_destroy(PdsSpectrograph **SPECTROGRAPH);


/**
 * @}
 */

#ifdef __cplusplus
}
#endif 

#endif	/* __PDSSPECTROGRAPH_H__ */ 

