/*
 * pdsfft.c
 * 
 * Copyright 2011 Fernando Pujaico Rivera <fernando.pujaico.rivera@gmail.com>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 * 
 */


#include <pds/pdsfft.h>
#include <pds/pdscvector.h>
#include <pds/pdsvector.h>
#include <pds/pdscn.h>
#include <stdio.h>
#include <math.h>


////////////////////////////////////////////////////////////////////////////////
////  Trabajando con PdsFft                                                 ////
////////////////////////////////////////////////////////////////////////////////


/** \fn PdsFft *pds_fft_new(PdsFtNatural *N)
 *  \brief Crea una estructura de tipo PdsFft, para generar una FFT de N puntos.
 *  Si N no es potencia de 2, no da error, y se crea una estructura para una FFT
 *  con un  N1, que si es potencia de dos y mayor a N, (N1>=N). El valor de N mínimo
 *  es N=2. El nuevo valor N1 será cargado a N. N=N1.
 *  \param[in,out] N Es el número de elementos de la FFT.
 *  \return Un puntero a una estructura de tipo PdsFft. En caso de error devuelve
 *  NULL.
 *  \ingroup PdsFftGroup
 */
PdsFft *pds_fft_new(PdsFtNatural *N)
{
	PdsFft *FFT=NULL;
	PdsFtNatural i,r,n;

	if(*N<=1)	return NULL;

	for(i=0,r=(*N-1);r>0;i++)	r=r/2;
	n=i;
	for(i=0,r=1;i<n;i++)		r=r*2;

	*N=r;	

	FFT=(PdsFft *)calloc(1,sizeof(PdsFft));
	if(FFT==NULL)	return NULL;

	FFT->N=r;
	FFT->W=pds_cvector_new(r/2);
	if(FFT->W==NULL)	
	{	
		pds_cvector_free(FFT->W);
		return NULL;
	}
	
	for(i=0;i<(FFT->W->Nel);i++)
	FFT->W->V[i]=pds_complex_cis((-2.0*M_PI*i)/(FFT->N*1.0));
	
	return FFT;
}


/** \fn int pds_fft_evaluate_real(const PdsFft *FFT,PdsCVector *Out,const PdsVector *In)
 *  \brief Evalua la transformada rapida de fourier a un vetor real. El tamanho 
 *  del vector Out debe ser igual que el número de puntos de la FFT, y mayor que
 *  el número de elementos de In, La diferencia se llevanara con ceros.
 *  \param[in] FFT La estructura a una FFT de N puntos.
 *  \param[out] Out El vector complejo con la FFT de N puntos.
 *  \param[in] In El vector de un tamaño menor o igual a N, al que se le desea 
 *  aplicar la FFT.
 *  \return TRUE si todo fue bien o FALSE si no.
 *  \ingroup PdsFftGroup
 */
int pds_fft_evaluate_real(const PdsFft *FFT,PdsCVector *Out,const PdsVector *In)
{
	PdsFtNatural k,khat,bit;
	PdsFtNatural n,m;
	PdsComplex y,z;

	if(FFT==NULL)	return FALSE;
	if(Out==NULL)	return FALSE;
	if(In==NULL)	return FALSE;
	if(Out->Nel!=FFT->N)	return FALSE;
	if(Out->Nel<In->Nel)	return FALSE;

	////////////////////////// Intercambiando Bits /////////////////////////
	for (k=0, khat=0; k<(FFT->N); k++)
	{
		if(k<(In->Nel))	Out->V[khat].Real=In->V[k];
		else		Out->V[khat].Real=0.0;
		Out->V[khat].Imag=0.0;

		for (bit=(FFT->N/2); (khat & bit)!=0; bit >>=1)
		khat ^= bit;
		khat ^= bit;
	}
	
	/////////////////////////// Calculado el FFT ///////////////////////////
	for (n=2; n<=(FFT->N); n <<= 1)
	{
		for (m=0; m<(FFT->N); m+=n)
		{
			for (k=0; k<n/2; k++)
			{
				y=Out->V[m+k];
				z=PDS_CMULC(Out->V[m+k+n/2],FFT->W->V[(k*FFT->N)/n]);

				Out->V[m+k]=PDS_CADDC(y,z);
				Out->V[m+k+n/2]=PDS_CSUBC(y,z);
			}
		}
	}
	return TRUE;
}


/** \fn int pds_fft_evaluate_complex(const PdsFft *FFT,PdsCVector *Out,const PdsCVector *In)
 *  \brief Evalua la transformada rapida de fourier a un vetor complejo. El tamanho 
 *  del vector Out debe ser igual que el número de puntos de la FFT, y mayor que
 *  el número de elementos de In, La diferencia se llevanara con ceros.
 *  \param[in] FFT La estructura a una FFT de N puntos.
 *  \param[out] Out El vector complejo con la FFT de N puntos.
 *  \param[in] In El vector de un tamaño menor o igual a N, al que se le desea 
 *  aplicar la FFT.
 *  \return TRUE si todo fue bien o FALSE si no.
 *  \ingroup PdsFftGroup
 */
int pds_fft_evaluate_complex(const PdsFft *FFT,PdsCVector *Out,const PdsCVector *In)
{
	PdsFtNatural k,khat,bit;
	PdsFtNatural n,m;
	PdsComplex y,z;

	if(FFT==NULL)	return FALSE;
	if(Out==NULL)	return FALSE;
	if(In==NULL)	return FALSE;
	if(Out->Nel!=FFT->N)	return FALSE;
	if(Out->Nel<In->Nel)	return FALSE;

	////////////////////////// Intercambiando Bits /////////////////////////
	for (k=0, khat=0; k<(FFT->N); k++)
	{
		if(k<(In->Nel))	
		{
			Out->V[khat].Real=In->V[k].Real;
			Out->V[khat].Imag=In->V[k].Imag;
		}
		else
		{
			Out->V[khat].Real=0.0;
			Out->V[khat].Imag=0.0;
		}

		for (bit=(FFT->N/2); (khat & bit)!=0; bit >>=1)
		khat ^= bit;
		khat ^= bit;
	}
	
	/////////////////////////// Calculado el FFT ///////////////////////////
	for (n=2; n<=(FFT->N); n <<= 1)
	{
		for (m=0; m<(FFT->N); m+=n)
		{
			for (k=0; k<n/2; k++)
			{
				y=Out->V[m+k];
				z=PDS_CMULC(Out->V[m+k+n/2],FFT->W->V[(k*FFT->N)/n]);

				Out->V[m+k]=PDS_CADDC(y,z);
				Out->V[m+k+n/2]=PDS_CSUBC(y,z);
			}
		}
	}
	return TRUE;
}


/** \fn void pds_fft_free(PdsFft *FFT)
 *  \brief Libera una estructura de tipo puntero PdsFft.
 *  \param[in,out] FFT La FFT a liberar.
 *  \return No retorna valor.
 *  \ingroup PdsFftGroup
 */
void pds_fft_free(PdsFft *FFT)
{
	if(FFT!=NULL)
	{
		pds_cvector_free(FFT->W);
		free(FFT);
	}
}


/** \fn void pds_fft_destroy(PdsFft **FFT)
 *  \brief Libera una estructura de tipo puntero PdsFft, y carga a la estructura con NULL.
 *  \param[in,out] FFT La FFT a liberar.
 *  \return No retorna valor.
 *  \ingroup PdsFftGroup
 */
void pds_fft_destroy(PdsFft **FFT)
{
	if((*FFT)!=NULL)
	{
		pds_cvector_free((*FFT)->W);
		free(*FFT);
		(*FFT)=NULL;
	}
}

