/*
 * pdsstft.h
 * 
 * Copyright 2011 Fernando Pujaico Rivera <fernando.pujaico.rivera@gmail.com>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 * 
 */

/** \file pdsstft.h
 *  \author Fernando Pujaico Rivera
 *  \date 10-05-2011
 *  \brief Funciones que trabajan con Fft.
 *  
 *  Estas funciones trabajan con una Fft así.<br>
 *  \image html pdsstft.png "Short-Time Fourier Transform"
 *  \f[ \mathbf{STFT_m} \left \{ x[n] \right \} \equiv Out_m[k] = \sum_{n=0}^{N-1} x_m[n]w[n]e^{-j 2 \pi k \frac{n}{N}} \f]
 */

#ifndef __PDSSTFT_H__
#define __PDSSTFT_H__

#ifdef __cplusplus
extern "C" {
#endif 


#include <stdio.h>

#include <pds/pdsftglobal.h>
#include <pds/pdscvector.h>
#include <pds/pdsvector.h>

#ifndef TRUE
	#define TRUE 1
#endif

#ifndef FALSE
	#define FALSE 0
#endif

/*! \defgroup PdsStftGroup Módulo PdsStft.
 *  \brief Cálculo de la STFT (Short-Time Fourier Transform ).
 *
 *  <center>
 *  \image html pdsstft.png "Short-Time Fourier Transform"
 *  </center>
 *  \f[ \mathbf{STFT_m} \left \{ x[n] \right \} \equiv Out_m[k] = \sum_{n=0}^{N-1} x_m[n]w[n]e^{-j 2 \pi k \frac{n}{N}} \f]
 * @{
 */


/*! \struct PdsStft
 *  \brief Estructura tipo  PdsStft.
 *  Esta estructura genera una Transformada de Fourier de Tiempo Reducido de N puntos.
 *  
 *  Para usar incluir pds/pdsstft.h.
 *
 *  <center>
 *  \image html pdsstft.png "Short-Time Fourier Transform"
 *  </center>
 *  \f[ \mathbf{STFT_m} \left \{ x[n] \right \} \equiv Out_m[k] = \sum_{n=0}^{N-1} x_m[n]w[n]e^{-j 2 \pi k \frac{n}{N}} \f]
 *  \ingroup PdsStftGroup
 *  \author Fernando Pujaico Rivera
 */
typedef struct 
{
	/*! Número de elementos del vector de entrada In[n] del bloque STFT.  */
	PdsFtNatural L;

	/*! Vector complejo de entrada de N elementos del bloque FFT. */
	PdsCVector *X;
	/*! Vector de pesos de N elementos para la entrada X. */
	PdsVector *W;
	/*! Número de elementos del vector Out[n] de salida del bloque STFT.  
	 *  Este valor tiene que ser par y mayor o igual que L
	 */
	PdsFtNatural N;

	/*! Estructura FFT de N elementos. */
	PdsFft	*FFT;
}PdsStft;




/** @name pds_stft_new
 *  Creando nuevas estructuras
 * @{
 */

/** \fn PdsStft *pds_stft_new(PdsFtNatural L, PdsFtNatural *N)
 *  \brief Crea una estructura de tipo PdsStft, para generar una STFT de L puntos
 *  de entrada y N puntos de salida, N>=L.
 *  Si N no es potencia de 2, no da error, y se crea una estructura para una STFT
 *  con un  N1, que si es potencia de dos y mayor a N1, (N1>=N). El valor de N mínimo
 *  es N=2. El nuevo valor N1 será cargado a N. N=N1.
 *  El vector W (ventana) se incia con todos los elementos con 1.0 .
 *  \param[in] L Es el número de elementos de entrada de la STFT.
 *  \param[in,out] N Es el número de elementos de salida de la STFT.
 *  \return Un puntero a una estructura de tipo PdsStft. En caso de error devuelve
 *  NULL.
 *  \ingroup PdsStftGroup
 */
PdsStft *pds_stft_new(PdsFtNatural L, PdsFtNatural *N);


//@}

/** @name pds_stft_set
 *  Escribiendo datos
 * @{
 */

/** \fn int pds_stft_set_window_rectangular(PdsStft *STFT)
 *  \brief Copia una ventana rectangular a W de la STFT.
 *  \param[in,out] STFT La estructura a una STFT de N puntos.
 *  \return TRUE si todo fue bien o FALSE si no. (Ejem STFT==NULL)
 *  \ingroup PdsStftGroup
 */
int pds_stft_set_window_rectangular(PdsStft *STFT);


/** \fn int pds_stft_set_window_hanning(PdsStft *STFT)
 *  \brief Copia una ventana de Hann a W de la STFT.
 *  \param[in,out] STFT La estructura a una STFT de N puntos.
 *  \return TRUE si todo fue bien o FALSE si no. (Ejem STFT==NULL)
 *  \ingroup PdsStftGroup
 */
int pds_stft_set_window_hanning(PdsStft *STFT);


/** \fn int pds_stft_set_window_hamming(PdsStft *STFT)
 *  \brief Copia una ventana de Hamming a W de la STFT.
 *  \param[in,out] STFT La estructura a una STFT de N puntos.
 *  \return TRUE si todo fue bien o FALSE si no. (Ejem STFT==NULL)
 *  \ingroup PdsStftGroup
 */
int pds_stft_set_window_hamming(PdsStft *STFT);


/** \fn int pds_stft_set_window_gauss(PdsStft *STFT, PdsFtReal Sigma)
 *  \brief Copia una ventana de Gauss a W de la STFT.
 *  \param[in,out] STFT La estructura a una STFT de N puntos.
 *  \param[in] Sigma Valor de sigma debe ser <= 0.5, de lo contrario da error.
 *  \return TRUE si todo fue bien o FALSE si no. (Ejem STFT==NULL)
 *  \ingroup PdsStftGroup
 */
int pds_stft_set_window_gauss(PdsStft *STFT, PdsFtReal Sigma);


//@}

/** @name pds_stft_evaluate
 *  Evaluando datos
 * @{
 */

/** \fn int pds_stft_evaluate_real(PdsStft *STFT,PdsCVector *Out,const PdsVector *In)
 *  \brief Evalua la STFT de un vetor real. El tamanho del vector Out debe ser 
 *  igual al número de puntos de salida de la STFT, El tamanho del vector In debe ser 
 *  igual al número de puntos de entrada de la STFT.
 *
 *  \f[ \mathbf{STFT_m} \left \{ x[n] \right \} \equiv Out_m[k] = \sum_{n=0}^{N-1} x_m[n]w[n]e^{-j 2 \pi k \frac{n}{N}} \f]
 *  \param[in,out] STFT La estructura a una STFT de N puntos.
 *  \param[out] Out El vector complejo con la STFT de N puntos.
 *  \param[in] In El vector de entrada de tamaño igual a L, al que se le desea 
 *  aplicar la STFT.
 *  \return TRUE si todo fue bien o FALSE si no. (Ejem STFT==NULL, In==NULL,
 *  Out==NULL, In->Nel!=L o Out->Nel!=N)
 *  \ingroup PdsStftGroup
 */
int pds_stft_evaluate_real(PdsStft *STFT,PdsCVector *Out,const PdsVector *In);


/** \fn int pds_stft_evaluate_complex(const PdsStft *STFT,PdsCVector *Out,const PdsCVector *In)
 *  \brief Evalua la STFT de un vetor real. El tamanho del vector Out debe ser 
 *  igual al número de puntos de salida de la STFT, El tamanho del vector In debe ser 
 *  igual al número de puntos de entrada de la STFT.
 *
 *  \f[ \mathbf{STFT_m} \left \{ x[n] \right \} \equiv Out_m[k] = \sum_{n=0}^{N-1} x_m[n]w[n]e^{-j 2 \pi k \frac{n}{N}} \f]
 *  \param[in,out] STFT La estructura a una STFT de N puntos.
 *  \param[out] Out El vector complejo con la STFT de N puntos.
 *  \param[in] In El vector de entrada de tamaño igual a L, al que se le desea 
 *  aplicar la STFT.
 *  \return TRUE si todo fue bien o FALSE si no. (Ejem STFT==NULL, In==NULL,
 *  Out==NULL, In->Nel!=L o Out->Nel!=N)
 *  \ingroup PdsStftGroup
 */
int pds_stft_evaluate_complex(const PdsStft *STFT,PdsCVector *Out,const PdsCVector *In);


//@}

/** @name pds_stft_<free>
 *  Liberando memoria.
 * @{
 */

/** \fn void pds_stft_free(PdsStft *STFT)
 *  \brief Libera una estructura de tipo puntero PdsStft.
 *  \param[in,out] STFT La STFT a liberar.
 *  \return No retorna valor.
 *  \ingroup PdsStftGroup
 */
void pds_stft_free(PdsStft *STFT);


/** \fn void pds_stft_destroy(PdsStft **STFT)
 *  \brief Libera una estructura de tipo puntero PdsStft, y carga a la estructura con NULL.
 *  \param[in,out] STFT La STFT a liberar.
 *  \return No retorna valor.
 *  \ingroup PdsStftGroup
 */
void pds_stft_destroy(PdsStft **STFT);


//@}


/**
 * @}
 */

#ifdef __cplusplus
}
#endif 

#endif	/* __PDSSTFT_H__ */ 

