/*
 * pdsgeneratedata.c
 * 
 * Copyright 2011 Fernando Pujaico Rivera <fernando.pujaico.rivera@gmail.com>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 * 
 */

#include <time.h> 
#include <math.h> 
#include <pds/pdsrv.h>
#include <pds/pdsra.h>
#include <pds/pdsgeneratedata.h>


/** \fn int pds_generate_data_input_func2pi(PdsVector *X,double (*func)(double),PdsNnReal C,PdsNnReal OffsetError)
 *  \brief Genera un vector con C ciclos de la función func() 
 *  con un error de offset de valor OffsetError.
 *
 *  \param[out] X Vector onde se colocaran los datos.
 *  \param[in] func funcion de periodo 2 pi.
 *  \param[in] C Número de veces que se repite la funcion func() en el vector X.
 *  \param[in] OffsetError Error no offset de cada muestra del vector.
 *  \return TRUE si todo fue bien o falso si no (ex: X==NULL ).
 *  \ingroup PdsGenerateDataGroup
 */
int pds_generate_data_input_func2pi(PdsVector *X,double (*func)(double),PdsNnReal C,PdsNnReal OffsetError)
{
	PdsRaNatural i;
	PdsRvReal m,phase;

	if(X==NULL)	return FALSE;

    PdsUniform *RV=pds_uniform_new(-1,1);
	if(RV==NULL)	return FALSE;      
  
    pds_uniform_init(RV,(clock()%512)/512.0); 	

	pds_uniform_get_value(RV,&phase);
    phase=M_PI*phase; // -pi < phase < +pi

	for(i=0;i<X->Nel;i++)	
	{
		pds_uniform_get_value(RV,&m);
		X->V[i]=func((2.0*M_PI*C*i)/(X->Nel)+phase)+OffsetError*m;
	}

    pds_uniform_free(RV);
	return TRUE;
}


/** \fn double pds_generate_data_func1(double x)
 *  \brief Genera una funcion de periodo 2*pi y dominio entre -1 y 1.
 *
 *  \param[out] x Valor de entrada.
 *  \return Retorna la funcion avaliada.
 *  \ingroup PdsGenerateDataGroup
 */
double pds_generate_data_func1(double x)
{
    return 0.89333*(sin(x)+0.2*sin(2*x)+0.3*sin(3*x));
}

/** \fn int pds_generate_data_input_rand(PdsVector *X)
 *  \brief Genera un vector con valores uniformemente distribuidos 
 *  entre -|A| y +|A| para los elementos del vector X.
 *
 *  \param[out] X Vector onde se colocaran los datos.
 *  \param[in] A Valor limite de los números aleatorios.
 *  \return TRUE si todo fue bien o falso si no (ex: X==NULL ).
 *  \ingroup PdsGenerateDataGroup
 */
int pds_generate_data_input_rand(PdsVector *X,PdsNnReal A)
{
	if(X==NULL)	return FALSE;

    PdsUniform *RV=pds_uniform_new(-fabs(A),fabs(A));
	if(RV==NULL)	return FALSE;  
    pds_uniform_init(RV,fabs(A)*(clock()%512)/512.0);
 	
	pds_uniform_get_vector(RV,X);

    pds_uniform_free(RV);

    return TRUE;
}


/** \fn int pds_generate_data_output_negative(PdsVector *Y)
 *  \brief Genera un vector con -1 en todos los elementos del vector Y.
 *
 *  \param[out] Y Vector onde se colocaran los datos.
 *  \return TRUE si todo fue bien o falso si no (ex: Y==NULL ).
 *  \ingroup PdsGenerateDataGroup
 */
int pds_generate_data_output_negative(PdsVector *Y)
{
	if(Y==NULL) return FALSE;
    pds_vector_init_value(Y,-1.0);
    return TRUE;
}

/** \fn int pds_generate_data_output_positive(PdsVector *Y)
 *  \brief Genera un vector con +1 en todos los elementos del vector Y.
 *
 *  \param[out] Y Vector onde se colocaran los datos.
 *  \return TRUE si todo fue bien o falso si no (ex: Y==NULL ).
 *  \ingroup PdsGenerateDataGroup
 */
int pds_generate_data_output_positive(PdsVector *Y)
{
	if(Y==NULL) return FALSE;
    pds_vector_init_value(Y,+1.0);
    return TRUE;
}

/** \fn int pds_generate_data_output_mspace(PdsVector *Y,PdsRaNatural M)
 *  \brief Genera un vector con +1 en la posición M del vector Y,
 *  los otros valores son -1.
 *
 *  \param[out] Y Vector onde se colocaran los datos.
 *  \param[in] M Posición del +1.
 *  \return TRUE si todo fue bien o falso si no (ex: Y==NULL o Y->Nel<M ).
 *  \ingroup PdsGenerateDataGroup
 */
int pds_generate_data_output_mspace(PdsVector *Y,PdsRaNatural M)
{

	if(Y==NULL)     return FALSE;
    if(Y->Nel<M)    return FALSE;

    pds_vector_init_value(Y,-1.0);
    
    Y->V[ M%(Y->Nel) ]=+1;
    
    return TRUE;
}



