/*
 * pdsvector.c
 * 
 * Copyright 2011 Fernando Pujaico Rivera <fernando.pujaico.rivera@gmail.com>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 * 
 */

#include <math.h>
#include <pds/pdsrautils.h>
#include <pds/pdsvector.h>


////////////////////////////////////////////////////////////////////////////////
////  Trabajando con PdsVector                                              ////
////////////////////////////////////////////////////////////////////////////////


/** \fn PdsVector *pds_vector_new(PdsRaNatural Nel)
 *  \brief Crea un vector de tipo PdsVector e inicia con cero todos los elementos.
 *  \param[in] Nel Es el número de elementos del vector.
 *  \return Un puntero al vector de tipo PdsVector.
 *  \ingroup PdsVectorGroup
 */
PdsVector *pds_vector_new(PdsRaNatural Nel)
{
	PdsVector *Vector=NULL;

	Vector=(PdsVector *)calloc(1,sizeof(PdsVector));
	if(Vector==NULL) {return NULL;}

	Vector->Nel=Nel;
	Vector->V=(PdsRaReal *)calloc(Vector->Nel,sizeof(PdsRaReal));
	if(Vector->V==NULL) 
	{
		free(Vector);
		return NULL;
	}

	return Vector;
}


/** \fn PdsVector *pds_vector_new_vector(const PdsVector *VecSrc)
 *  \brief Crea un vector de tipo PdsVector usando como fuente el vector VecSrc.
 *  \param[in] VecSrc Es el vector imagen que se usará como fuente.
 *  \return Un puntero al vector de tipo PdsVector.
 *  \ingroup PdsVectorGroup
 */
PdsVector *pds_vector_new_vector(const PdsVector *VecSrc)
{
	PdsVector *Vector=NULL;
	PdsRaNatural i;

	if(VecSrc==NULL)	return NULL;

	Vector=(PdsVector *)calloc(1,sizeof(PdsVector));
	if(Vector==NULL) {return NULL;}

	Vector->Nel=VecSrc->Nel;
	Vector->V=(PdsRaReal *)calloc(Vector->Nel,sizeof(PdsRaReal));
	if(Vector->V==NULL) 
	{
		free(Vector);
		return NULL;
	}
	else
	{
		for(i=0;i<Vector->Nel;i++)	Vector->V[i]=VecSrc->V[i];
	}

	return Vector;
}


/** \fn PdsVector *pds_vector_new_load_data(const char* datafile)
 *  \brief Crea un nuevo vector e inicia los datos con los elementos del archivo
 *  datafile.
 *  Usa TAB o un salto de linea como delimitador de elemento.
 *  \param[in] datafile Nombre del archivo de donde se cargará los datos iniciales
 *  del vector.
 *  \return Un puntero que apunta a la dirección del nuevo vector, si todo fue bien, 
 *  o NULL si no. (ej. datafile==NULL)
 *  \ingroup PdsVectorGroup
 */
PdsVector *pds_vector_new_load_data(const char* datafile)
{
	PdsRaNatural i,n;
	int id;
	int tmp;
	FILE *fd=NULL;
	char x[__MAX_CHAR_DATA_SIZE__];
	PdsVector *Vector=NULL;
	
	if(datafile==NULL)	return NULL;

	fd=fopen(datafile,"r");
	if(fd==NULL)	return NULL;

	id=pds_ra_nelements_in_file(fd,&n);
	if(id==FALSE)	
	{
		fclose(fd);
		return NULL;
	}

	Vector=pds_vector_new(n);
	if(Vector==NULL)	
	{
		fclose(fd);
		return NULL;
	}

	for(i=0;i<n;i++)
	{
		x[0]=0;
		tmp=fscanf(fd,"%s",x);
		if(strlen(x)==0)	break;

		Vector->V[i]=atof(x);
	}
	
	fclose(fd);
	return Vector;
}


/** \fn PdsVector *pds_vector_new_load_data_line(const char* datafile,PdsRaNatural line)
 *  \brief Crea un nuevo vector e inicia los datos con los elementos de la linea
 *  line del archivo datafile.
 *  Usa TAB como delimitador de elemento y un salto de linea como delimitador de linea.
 *  Si se pide una linea inexistente la funcion retorna NULL
 *  \param[in] datafile Nombre del archivo de donde se cargará los datos iniciales
 *  del vector.
 *  \param[in] line Número de linea que se cargará, line inicia en cero. 
 *  \return Un puntero que apunta a la dirección del nuevo vector, si todo fue bien, 
 *  o NULL si no. (ej. datafile==NULL o line inexistente)
 *  \ingroup PdsVectorGroup
 */
PdsVector *pds_vector_new_load_data_line(const char* datafile,PdsRaNatural line)
{
	PdsRaNatural i,j,nc,nl;
	int id;
	int tmp;
	FILE *fd=NULL;
	char x[__MAX_CHAR_DATA_SIZE__];
	PdsVector *Vector=NULL;
	
	fd=fopen(datafile,"r");
	if(fd==NULL)	return NULL;

	id=pds_ra_ncolumns_nlines_in_file(fd,&nc,&nl);
	if(id==FALSE)	
	{
		fclose(fd);
		return NULL;
	}

	if(line>=nl)
	{
		fclose(fd);
		return NULL;
	}

	Vector=pds_vector_new(nc);
	if(Vector==NULL)	
	{
		fclose(fd);
		return NULL;
	}

	for(j=0;j<nl;j++)
	for(i=0;i<nc;i++)
	{
		x[0]=0;
		tmp=fscanf(fd,"%s",x);
		if(strlen(x)==0)	break;

		if(j==line)
		Vector->V[i]=atof(x);
	}
	
	fclose(fd);
	return Vector;
}


/** \fn PdsVector *pds_vector_new_load_data_col(const char* datafile,PdsRaNatural col)
 *  \brief Crea un nuevo vector e inicia los datos con los elementos de la columna
 *  col del archivo datafile.
 *  Usa TAB como delimitador de columna y un salto de linea como delimitador de linea.
 *  Si se pide una linea inexistente la función retorna NULL
 *  \param[in] datafile Nombre del archivo de donde se cargará los datos iniciales
 *  del vector.
 *  \param[in] col Número de columna que se cargará, col inicia en cero. 
 *  \return Un puntero que apunta a la dirección del nuevo vector, si todo fue bien, 
 *  o NULL si no. (ej. datafile==NULL o col inexistente)
 *  \ingroup PdsVectorGroup
 */
PdsVector *pds_vector_new_load_data_col(const char* datafile,PdsRaNatural col)
{
	PdsRaNatural i,j,nc,nl;
	int id;
	int tmp;
	FILE *fd=NULL;
	char x[__MAX_CHAR_DATA_SIZE__];
	PdsVector *Vector=NULL;
	
	fd=fopen(datafile,"r");
	if(fd==NULL)	return NULL;

	id=pds_ra_ncolumns_nlines_in_file(fd,&nc,&nl);
	if(id==FALSE)	
	{
		fclose(fd);
		return NULL;
	}

	if(col>=nc)
	{
		fclose(fd);
		return NULL;
	}

	Vector=pds_vector_new(nl);
	if(Vector==NULL)	
	{
		fclose(fd);
		return NULL;
	}

	for(j=0;j<nl;j++)
	for(i=0;i<nc;i++)
	{
		x[0]=0;
		tmp=fscanf(fd,"%s",x);
		if(strlen(x)==0)	break;

		if(i==col)
		Vector->V[j]=atof(x);
	}
	
	fclose(fd);
	return Vector;
}

/** \fn int pds_vector_init_value(PdsVector *Vector,PdsRaReal Val)
 *  \brief Inicia el vector de tipo puntero PdsVector. <br><b>Vector=Val</b>.
 *  \param[in,out] Vector El vector a iniciar.
 *  \param[in] Val Es el valor inicial de los elementos.
 *  \return TRUE si todo fue bien o FALSE si no.
 *  \ingroup PdsVectorGroup
 */
int pds_vector_init_value(PdsVector *Vector,PdsRaReal Val)
{
	PdsRaNatural i;

	if(Vector==NULL)	return FALSE;

	for(i=0;i<Vector->Nel;i++)
	{
		Vector->V[i]=Val;
	}
	
	return TRUE;
}


/** \fn int pds_vector_init_vector(PdsVector *Vector,const PdsVector *VecSrc)
 *  \brief Inicia el vector de tipo puntero PdsVector con otro vector.
 *  Si los tamaños son diferentes intersecta los tamaños y hace la copia en la
 *  intersección solamente. <br><b>Vector=VecSrc</b>.
 *  \param[in,out] Vector Es el vector a iniciar.
 *  \param[in] VecSrc Es el vector fuente de la copia.
 *  \return TRUE si todo fue bien o FALSE si no.
 *  \ingroup PdsVectorGroup
 */
int pds_vector_init_vector(PdsVector *Vector,const PdsVector *VecSrc)
{
	PdsRaNatural i;
	PdsRaNatural Nel;

	if(Vector==NULL)	return FALSE;
	if(VecSrc==NULL)	return FALSE;

	if(Vector->Nel < VecSrc->Nel)	Nel=Vector->Nel;
	else				Nel=VecSrc->Nel;

	for(i=0;i<Nel;i++)
	{
		Vector->V[i]=VecSrc->V[i];
	}
	
	return TRUE;
}


/** \fn int pds_vector_add_vector(PdsVector *Vector,const PdsVector *VecSrc)
 *  \brief Suma al vector de tipo puntero PdsVector con otro vector y lo copia en Vector.
 *  Si los tamaños son diferentes intersecta los tamaños y hace la copia en la
 *  intersección solamente. <br><b>Vector=Vector+VecSrc</b>.
 *  \param[in,out] Vector Es el vector de destino y operando.
 *  \param[in] VecSrc Es el Vector fuente del operando.
 *  \return TRUE si todo fue bien o FALSE si no.
 *  \ingroup PdsVectorGroup
 */
int pds_vector_add_vector(PdsVector *Vector,const PdsVector *VecSrc)
{
	PdsRaNatural i;
	PdsRaNatural Nel;

	if(Vector==NULL)	return FALSE;
	if(VecSrc==NULL)	return FALSE;

	if(Vector->Nel < VecSrc->Nel)	Nel=Vector->Nel;
	else				Nel=VecSrc->Nel;

	for(i=0;i<Nel;i++)
	{
		Vector->V[i]=Vector->V[i]+VecSrc->V[i];
	}
	
	return TRUE;
}

/** \fn int pds_vector_add_factor_vector(PdsVector *Vector,PdsRaReal Alpha,const PdsVector *VecSrc)
 *  \brief Suma al vector de tipo puntero PdsVector con otro vector con unf actor 
 *  y lo copia en Vector. Si los tamaños son diferentes intersecta los tamaños y 
 *  hace la copia en la intersección solamente. <br><b>Vector=Vector+Alpha*VecSrc</b>.
 *  \param[in,out] Vector Es el vector de destino y operando.
 *  \param[in] Alpha Factor que multiplica al operando.
 *  \param[in] VecSrc Es el Vector fuente del operando.
 *  \return TRUE si todo fue bien o FALSE si no.
 *  \ingroup PdsVectorGroup
 */
int pds_vector_add_factor_vector(PdsVector *Vector,PdsRaReal Alpha,const PdsVector *VecSrc)
{
	PdsRaNatural i;
	PdsRaNatural Nel;

	if(Vector==NULL)	return FALSE;
	if(VecSrc==NULL)	return FALSE;

	if(Vector->Nel < VecSrc->Nel)	Nel=Vector->Nel;
	else				Nel=VecSrc->Nel;

	for(i=0;i<Nel;i++)
	{
		Vector->V[i]=Vector->V[i]+Alpha*VecSrc->V[i];
	}
	
	return TRUE;
}

/** \fn int pds_vector_add_value(PdsVector *Vector,PdsRaReal Val)
 *  \brief Suma al vector de tipo puntero PdsVector el valor Val. <br><b>Vector=Vector+Val</b>.
 *  \param[in,out] Vector El vector a sumar.
 *  \param[in] Val Es el valor inicial de los elementos.
 *  \return TRUE si todo fue bien o FALSE si no.
 *  \ingroup PdsVectorGroup
 */
int pds_vector_add_value(PdsVector *Vector,PdsRaReal Val)
{
	PdsRaNatural i;

	if(Vector==NULL)	return FALSE;

	for(i=0;i<Vector->Nel;i++)
	{
		Vector->V[i]=Vector->V[i]+Val;
	}
	
	return TRUE;
}


/** \fn int pds_vector_sub_vector(PdsVector *Vector,const PdsVector *VecSrc)
 *  \brief Resta al vector de tipo puntero PdsVector con otro vector y lo copia en Vector.
 *  Si los tamaños son diferentes intersecta los tamaños y hace la copia en la
 *  intersección solamente. <br><b>Vector=Vector-VecSrc</b>.
 *  \param[in,out] Vector Es el vector de destino y operando.
 *  \param[in] VecSrc Es el Vector fuente del operando.
 *  \return TRUE si todo fue bien o FALSE si no.
 *  \ingroup PdsVectorGroup
 */
int pds_vector_sub_vector(PdsVector *Vector,const PdsVector *VecSrc)
{
	PdsRaNatural i;
	PdsRaNatural Nel;

	if(Vector==NULL)	return FALSE;
	if(VecSrc==NULL)	return FALSE;

	if(Vector->Nel < VecSrc->Nel)	Nel=Vector->Nel;
	else				Nel=VecSrc->Nel;

	for(i=0;i<Nel;i++)
	{
		Vector->V[i]=Vector->V[i]-VecSrc->V[i];
	}
	
	return TRUE;
}


/** \fn int pds_vector_mul_vector_elements(PdsVector *Vector,const PdsVector *VecSrc)
 *  \brief Multiplica al vector de tipo puntero PdsVector con otro vector y lo copia en Vector.
 *  Si los tamaños son diferentes intersecta los tamaños y hace la copia en la
 *  intersección solamente. <br><b>Vector=Vector.VecSrc</b>.
 *  \param[in,out] Vector Es el vector de destino y operando.
 *  \param[in] VecSrc Es el Vector fuente del operando.
 *  \return TRUE si todo fue bien o FALSE si no.
 *  \ingroup PdsVectorGroup
 */
int pds_vector_mul_vector_elements(PdsVector *Vector,const PdsVector *VecSrc)
{
	PdsRaNatural i;
	PdsRaNatural Nel;

	if(Vector==NULL)	return FALSE;
	if(VecSrc==NULL)	return FALSE;

	if(Vector->Nel < VecSrc->Nel)	Nel=Vector->Nel;
	else				Nel=VecSrc->Nel;

	for(i=0;i<Nel;i++)
	{
		Vector->V[i]=Vector->V[i]*VecSrc->V[i];
	}
	
	return TRUE;
}


/** \fn int pds_vector_mul_value(PdsVector *Vector,PdsRaReal Val)
 *  \brief Multiplica al vector de tipo puntero PdsVector por un escalar. <br><b>Vector=Val*Vector</b>.
 *  \param[in,out] Vector El vector a multiplicar.
 *  \param[in] Val Es el valor que multiplicará a los elementos del vector.
 *  \return TRUE si todo fue bien o FALSE si no.
 *  \ingroup PdsVectorGroup
 */
int pds_vector_mul_value(PdsVector *Vector,PdsRaReal Val)
{
	PdsRaNatural i;

	if(Vector==NULL)	return FALSE;

	for(i=0;i<Vector->Nel;i++)
	{
		Vector->V[i]=Vector->V[i]*Val;
	}
	
	return TRUE;
}


/** \fn int pds_vector_div_value(PdsVector *Vector,PdsRaReal Val)
 *  \brief Divide al vector de tipo puntero PdsVector por un escalar. <br><b>Vector=Vector/Val</b>.
 *  \param[in,out] Vector El vector a dividir.
 *  \param[in] Val Es el valor que dividirá a los elementos del vector.
 *  \return TRUE si todo fue bien o FALSE si no.
 *  \ingroup PdsVectorGroup
 */
int pds_vector_div_value(PdsVector *Vector,PdsRaReal Val)
{
	PdsRaNatural i;

	if(Vector==NULL)	return FALSE;

	for(i=0;i<Vector->Nel;i++)
	{
		Vector->V[i]=Vector->V[i]/Val;
	}
	
	return TRUE;
}


/** \fn int pds_vector_dot_vector(const PdsVector *Vector1,const PdsVector *Vector2,PdsRaReal *Val)
 *  \brief Ejecuta el producto punto en el vector Vector1 y Vector2, y lo carga en Val.
 *  Los tamaños deben ser iguales. <br><b>Val=(Vector1,Vector2) </b>.
 *  \param[in] Vector1 Es el vector operando.
 *  \param[in] Vector2 Es el Vector operando.
 *  \param[out] Val El producto punto entre Vector1 y Vector2.
 *  \return TRUE si todo fue bien o FALSE si no.
 *  \ingroup PdsVectorGroup
 */
int pds_vector_dot_vector(const PdsVector *Vector1,const PdsVector *Vector2,PdsRaReal *Val)
{
	PdsRaNatural i;

	if(Vector1==NULL)	return FALSE;
	if(Vector2==NULL)	return FALSE;

	if(Vector1->Nel != Vector2->Nel)	return FALSE;

	*Val=0;
	for(i=0;i<Vector1->Nel;i++)
	{
		*Val=*Val+Vector1->V[i]*Vector2->V[i];
	}
	
	return TRUE;
}


/** \fn int pds_vector_printf(const PdsVector *Vector)
 *  \brief Imprime en pantalla un vector de tipo puntero PdsVector.
 *  \param[in] Vector El vector a imprimir en pantalla.
 *  \return TRUE si todo fue bien o FALSE si no.
 *  \ingroup PdsVectorGroup
 */
int pds_vector_printf(const PdsVector *Vector)
{
	PdsRaNatural i,j;

	if(Vector==NULL)	return FALSE;
	//printf("\n");
	for(i=0;i<Vector->Nel;i++)
	{
		printf("%e",Vector->V[i]);
		
		if(i==(Vector->Nel-1))	printf("\n");
		else			printf("\t");
	}
	return TRUE;
}


/** \fn int pds_vector_fprintf(const PdsVector *Vector,FILE *fd)
 *  \brief Imprime en el archivo que apunta fd un vector de tipo puntero PdsVector.
 *  \param[in] Vector El vector a imprimir en pantalla.
 *  \param[in,out] fd El puntero al archivo.
 *  \return TRUE si todo fue bien o FALSE si no.
 *  \ingroup PdsVectorGroup
 */
int pds_vector_fprintf(const PdsVector *Vector,FILE *fd)
{
	PdsRaNatural i,j;

	if(Vector==NULL)	return FALSE;
	//fprintf(fd,"\n");
	for(i=0;i<Vector->Nel;i++)
	{
		fprintf(fd,"%e",Vector->V[i]);
		
		if(i==(Vector->Nel-1))	fprintf(fd,"\n");
		else			fprintf(fd,"\t");
	}
	return TRUE;
}


/** \fn int pds_vector_fscanf(PdsVector *Vector, FILE *fd)
 *  \brief Inicializa un vector con los datos del archivo apuntado por fd.
 *  Usa TAB o un salto de linea como delimitador de elemento.
 *  \param[in] Vector Vector en donde se cargaran los datos.
 *  \param[in] fd Apuntador del archivo de donde se cargará los datos iniciales
 *  del vector.
 *  \return TRUE si todo fue bien o FALSE si no. (ej. Vector==NULL o fd==NULL)
 *  \ingroup PdsVectorGroup
 */
int pds_vector_fscanf(PdsVector *Vector, FILE *fd)
{
	PdsRaNatural i,id;
	int tmp;
	char x[__MAX_CHAR_DATA_SIZE__];
	
	if(Vector==NULL)	return FALSE;
	if(fd==NULL)		return FALSE;

	id=0;
	for(i=0;(i<Vector->Nel+id)&&(feof(fd)==0);i++)
	{
		x[0]=0;
		tmp=fscanf(fd,"%s",x);
		Vector->V[i-id]=atof(x);
		if(strlen(x)==0)	id++;

	}
	i=i-id;
	if(i<Vector->Nel)
	{
		for(;i<Vector->Nel;i++)	Vector->V[i]=0.0;
		return FALSE;
	}
	return TRUE;
}


/** \fn int pds_vector_fwrite(PdsVector *Vector, FILE *fd)
 *  \brief Escribe los datos de un vector en el archivo binario apuntado por fd.
 *  \param[in] Vector Vector de donde se leeran los datos.
 *  \param[in] fd Apuntador del archivo binario de donde se escribiran los datos 
 *  del vector.
 *  \return TRUE si todo fue bien o FALSE si no. (ej. Vector==NULL o fd==NULL)
 *  \ingroup PdsVectorGroup
 */
int pds_vector_fwrite(PdsVector *Vector, FILE *fd)
{
	size_t n;
	
	if(Vector==NULL)	return FALSE;
	if(fd==NULL)		return FALSE;

	n=fwrite(Vector->V,sizeof(Vector->V[0]),Vector->Nel,fd);

	if(n!=Vector->Nel)	return FALSE;

	return TRUE;
}


/** \fn int pds_vector_fread(PdsVector *Vector, FILE *fd)
 *  \brief Inicializa un vector con los datos del archivo binario apuntado por fd.
 *  \param[in] Vector Vector en donde se cargaran los datos.
 *  \param[in] fd Apuntador del archivo binario de donde se cargará los datos 
 *  iniciales del vector.
 *  \return TRUE si todo fue bien o FALSE si no. (ej. Vector==NULL o fd==NULL)
 *  \ingroup PdsVectorGroup
 */
int pds_vector_fread(PdsVector *Vector, FILE *fd)
{
	PdsRaNatural i;
	size_t n;
	
	if(Vector==NULL)	return FALSE;
	if(fd==NULL)		return FALSE;

	for(i=0;(i<Vector->Nel)&&(feof(fd)==0);i++)
	{
		n=fread(&(Vector->V[i]),sizeof(Vector->V[i]),1,fd);
		if(n!=1)	break;
	}

	if(i<Vector->Nel)
	{
		for(;i<Vector->Nel;i++)	Vector->V[i]=0.0;
		return FALSE;
	}
	return TRUE;
}


/** \fn int pds_vector_norm_vector(const PdsVector *Vector, PdsRaReal *m)
 *  \brief Devuelve la norma del vector Vector.
 *  \param[in] Vector El vector en consulta.
 *  \param[out] m El valor de la norma del vector Vector.
 *  \return TRUE si todo fue bien o FALSE si no (ej: Vector==NULL). 
 *  \ingroup PdsVectorGroup
 */
int pds_vector_norm_vector(const PdsVector *Vector, PdsRaReal *m)
{
	PdsRaNatural i;

	*m=0;

	if(Vector==NULL)	return FALSE;

	for(i=0;i<Vector->Nel;i++)	*m=*m+Vector->V[i]*Vector->V[i];

	*m=sqrt(*m);
	return TRUE;
}


/** \fn int pds_vector_norm2_vector(const PdsVector *Vector, PdsRaReal *m)
 *  \brief Devuelve la norma^2 del vector Vector.
 *  \param[in] Vector El vector en consulta.
 *  \param[out] m El valor de la norma^2 del vector Vector.
 *  \return TRUE si todo fue bien o FALSE si no (ej: Vector==NULL). 
 *  \ingroup PdsVectorGroup
 */
int pds_vector_norm2_vector(const PdsVector *Vector, PdsRaReal *m)
{
	PdsRaNatural i;

	*m=0;

	if(Vector==NULL)	return FALSE;

	for(i=0;i<Vector->Nel;i++)	*m=*m+Vector->V[i]*Vector->V[i];

	return TRUE;
}


/** \fn int pds_vector_rms_vector(const PdsVector *Vector, PdsRaReal *m)
 *  \brief Devuelve el valor raíz cuadrático medio del vector Vector.
 *  \param[in] Vector El vector en consulta.
 *  \param[out] m El valor rms del vector Vector.
 *  \return TRUE si todo fue bien o FALSE si no (ej: Vector==NULL). 
 *  \ingroup PdsVectorGroup
 */
int pds_vector_rms_vector(const PdsVector *Vector, PdsRaReal *m)
{
	PdsRaNatural i;

	*m=0;

	if(Vector==NULL)	return FALSE;

	for(i=0;i<Vector->Nel;i++)	*m=*m+Vector->V[i]*Vector->V[i];

	*m=sqrt(*m/Vector->Nel);

	return TRUE;
}


/** \fn int pds_vector_rms_vectors(const PdsVector *VectorA, const PdsVector *VectorB, PdsRaReal *m)
 *  \brief Devuelve el valor raíz cuadrático medio de la diferencia de los vectores.
 *  \param[in] VectorA El vector en consulta.
 *  \param[in] VectorB El vector en consulta.
 *  \param[out] m El valor rms de la diferencias de vectores.
 *  \return TRUE si todo fue bien o FALSE si no (ej: Vector==NULL, o diferentes tamaños). 
 *  \ingroup PdsVectorGroup
 */
int pds_vector_rms_vectors(const PdsVector *VectorA, const PdsVector *VectorB, PdsRaReal *m)
{
	PdsRaNatural i;

	*m=0;

	if(VectorA==NULL)	return FALSE;
	if(VectorB==NULL)	return FALSE;
	if(VectorA->Nel!=VectorB->Nel)	return FALSE;

	for(i=0;i<VectorA->Nel;i++)	
	*m=*m+(VectorA->V[i]-VectorB->V[i])*(VectorA->V[i]-VectorB->V[i]);

	*m=sqrt(*m/VectorA->Nel);

	return TRUE;
}


/** \fn int pds_vector_rms_factor_vectors(const PdsVector *VectorA, const PdsVector *VectorB, PdsRaReal *m)
 *  \brief Devuelve el valor raíz cuadrático medio de la diferencia de los vectores
 *  entre el valor raíz cuadrático medio de VectorB, si este valor es cero, se 
 *  usa entonces el valor raíz cuadrático medio de VectorA.
 *
 *  \f[ 
	\frac{\sqrt{\frac{\sum_{i=0}^{Nel-1}{(VectorA[i]-VectorB[i])^2}}{Nel}}}
	{\sqrt{\frac{\sum_{i=0}^{Nel-1}{VectorB[i]^2}}{Nel}}} 
    \f]
 *  \param[in] VectorA El vector en consulta.
 *  \param[in] VectorB El vector en consulta.
 *  \param[out] m El valor factor rms de la diferencias de vectores.
 *  \return TRUE si todo fue bien o FALSE si no (ej: Vector==NULL, o diferentes tamaños). 
 *  \ingroup PdsVectorGroup
 */
int pds_vector_rms_factor_vectors(const PdsVector *VectorA, const PdsVector *VectorB, PdsRaReal *m)
{
	PdsRaNatural i;
	PdsRaReal d;

	*m=0;

	if(VectorA==NULL)	return FALSE;
	if(VectorB==NULL)	return FALSE;
	if(VectorA->Nel!=VectorB->Nel)	return FALSE;

	for(i=0;i<VectorA->Nel;i++)	
	*m=*m+(VectorA->V[i]-VectorB->V[i])*(VectorA->V[i]-VectorB->V[i]);

	*m=sqrt(*m/VectorA->Nel);

	pds_vector_rms_vector(VectorB,&d);
	if(d<=0)	
	{
		pds_vector_rms_vector(VectorA,&d);
		if(d<=0)
		{
			d=1.0;
		}
	}

	*m=*m/d;

	return TRUE;
}


/** \fn int pds_vector_mean_vector(const PdsVector *Vector, PdsRaReal *m)
 *  \brief Devuelve el valor medio del vector (X) Vector.
 *
 *  \f[ \eta=\frac{\sum_{i=0}^{Nel-1} {X_i}}{Nel} \f]
 *  \param[in] Vector El vector en consulta.
 *  \param[out] m El valor medio del vector Vector.
 *  \return TRUE si todo fue bien o FALSE si no (ej: Vector==NULL). 
 *  \ingroup PdsVectorGroup
 */
int pds_vector_mean_vector(const PdsVector *Vector, PdsRaReal *m)
{
	PdsRaNatural i;

	*m=0;

	if(Vector==NULL)	return FALSE;

	for(i=0;i<Vector->Nel;i++)	*m=*m+Vector->V[i];

	*m=*m/Vector->Nel;

	return TRUE;
}


/** \fn int pds_vector_var_vector(const PdsVector *Vector, PdsRaReal *v)
 *  \brief Devuelve el valor de la varianza muestral \f$\sigma^2\f$ del vector Vector (X).
 *
 *  \f[ \eta=\frac{\sum_{i=0}^{Nel-1} {X_i}}{Nel} \f]
 *  \f[ \sigma^2=\frac{\sum_{i=0}^{Nel-1} {(X_i-\eta)^2}}{Nel-1} \f]
 *  \param[in] Vector El vector en consulta.
 *  \param[out] v El valor de la varianza del vector Vector.
 *  \return TRUE si todo fue bien o FALSE si no (ej: Vector==NULL). 
 *  \ingroup PdsVectorGroup
 */
int pds_vector_var_vector(const PdsVector *Vector, PdsRaReal *v)
{
	PdsRaNatural i,id;
	PdsRaReal    m;
	*v=0;

	if(Vector==NULL)	return FALSE;

	id=pds_vector_mean_vector (Vector,&m);
	if(id==FALSE)		return FALSE;

	for(i=0;i<Vector->Nel;i++)
	{
		(*v)=(*v)+(Vector->V[i]-m)*(Vector->V[i]-m);
	}

	(*v)=(*v)/(Vector->Nel-1);
	return TRUE;
}


/** \fn int pds_vector_cov_vector (const PdsVector *VectorX,const PdsVector *VectorY, PdsRaReal *c)
 *  \brief Devuelve el valor de la covarianza muestral de los vectores VectorX y VectorY.
 *
 *  \f[ \eta_x=\frac{\sum_{i=0}^{Nel-1} {X_i}}{Nel} \f]
 *  \f[ \eta_y=\frac{\sum_{i=0}^{Nel-1} {Y_i}}{Nel} \f]
 *  \f[ cov(X,Y)=\frac{\sum_{i=0}^{Nel-1} {(X_i -{\eta}_x)(Y_i -{\eta}_y)}}{Nel-1} \f]
 *  \param[in] VectorX El vector en consulta.
 *  \param[in] VectorY El vector en consulta.
 *  \param[out] c El valor de la covarianza de los vectores VectorX y VectorY.
 *  \return TRUE si todo fue bien o FALSE si no (ej: VectorX==NULL, VectorY==NULL o
 *  longitudes distintas). 
 *  \ingroup PdsVectorGroup
 */
int pds_vector_cov_vector (const PdsVector *VectorX,const PdsVector *VectorY, PdsRaReal *c)
{
	PdsRaReal mx,my;
	PdsRaNatural i,id;

	*c=0;

	if(VectorX==NULL)		return FALSE;
	if(VectorY==NULL)		return FALSE;
	if(VectorX->Nel!=VectorY->Nel)	return FALSE;

	id=pds_vector_mean_vector (VectorX,&mx);
	if(id==FALSE)			return FALSE;
	id=pds_vector_mean_vector (VectorY,&my);
	if(id==FALSE)			return FALSE;

	for(i=0;i<VectorX->Nel;i++)
	{
		(*c)=(*c)+(VectorX->V[i]-mx)*(VectorY->V[i]-my);
	}

	(*c)=(*c)/(VectorX->Nel-1);
	return TRUE;
}


/** \fn int pds_vector_cor_vector (const PdsVector *VectorX,const PdsVector *VectorY, PdsRaReal *c)
 *  \brief Devuelve el valor del coeficiente de correlación muestral de los vectores VectorX y VectorY.
 *
 *  \f[ \eta_x=\frac{\sum_{i=0}^{Nel-1} {X_i}}{Nel} \f]
 *  \f[ \eta_y=\frac{\sum_{i=0}^{Nel-1} {Y_i}}{Nel} \f]
 *  \f[ cor(X,Y)=\frac{\sum_{i=0}^{Nel-1} {(X_i -{\eta}_x)(Y_i -{\eta}_y)}}{\sqrt{\sum_{i=0}^{Nel-1} {(X_i -{\eta}_x)^2}}\sqrt{\sum_{i=0}^{Nel-1} {(Y_i -{\eta}_y)^2}}} \f]
 *  \param[in] VectorX El vector en consulta.
 *  \param[in] VectorY El vector en consulta.
 *  \param[out] c El valor de la correlación de los vectores VectorX y VectorY.
 *  \return TRUE si todo fue bien o FALSE si no (ej: VectorX==NULL, VectorY==NULL o
 *  longitudes distintas). 
 *  \ingroup PdsVectorGroup
 */
int pds_vector_cor_vector (const PdsVector *VectorX,const PdsVector *VectorY, PdsRaReal *c)
{
	PdsRaReal mx,my,txy,tx,ty;
	PdsRaNatural i,id;

	*c=0;

	if(VectorX==NULL)		return FALSE;
	if(VectorY==NULL)		return FALSE;
	if(VectorX->Nel!=VectorY->Nel)	return FALSE;

	id=pds_vector_mean_vector (VectorX,&mx);
	if(id==FALSE)			return FALSE;
	id=pds_vector_mean_vector (VectorY,&my);
	if(id==FALSE)			return FALSE;

	for(txy=0,i=0;i<VectorX->Nel;i++)
	{
		txy=txy+(VectorX->V[i]-mx)*(VectorY->V[i]-my);
	}

	for(tx=0,i=0;i<VectorX->Nel;i++)
	{
		tx=tx+(VectorX->V[i]-mx)*(VectorX->V[i]-mx);
	}

	for(ty=0,i=0;i<VectorY->Nel;i++)
	{
		ty=ty+(VectorY->V[i]-my)*(VectorY->V[i]-my);
	}

	(*c)=txy/sqrt(tx*ty);
	return TRUE;
}


/** \fn int pds_vector_get_value(const PdsVector *Vector, PdsRaNatural x, PdsRaReal *m)
 *  \brief Devuelve el valor en la posición (x) del vector Vector.
 *  (x) inicia con (0).
 *  \param[in] Vector El vector en consulta.
 *  \param[in] x Posición x, el primer valor de x es cero.
 *  \param[out] m El valor en la posición (x), en caso de error por fuera de rango 
 *  (x) entonces carga 0 en m, en caso de error de vector nulo carga cero en m.
 *  \return TRUE si todo fue bien o FALSE si no (ej: Vector==NULL). 
 *  \ingroup PdsVectorGroup
 */
int pds_vector_get_value(const PdsVector *Vector, PdsRaNatural x, PdsRaReal *m)
{
	*m=0;

	if(Vector==NULL)	return FALSE;

	if((x>=0)&&(x<Vector->Nel))	*m=Vector->V[x];
	else				*m=0;

	return TRUE;
}


/** \fn int pds_vector_set_value(PdsVector *Vector, PdsRaNatural x, PdsRaReal m)
 *  \brief Escribe el valor m en la posición (x) del vector Vector.
 *  (x) inicia con (0).
 *  \param[in,out] Vector El vector a escribir.
 *  \param[in] x Posición x, el primer valor de x es cero.
 *  \param[in] m El valor en la posición (x), en caso de error por fuera de rango 
 *  (x) entonces no hace nada y no se considera como error.
 *  \return TRUE si todo fue bien o FALSE si no (ej: Vector==NULL). 
 *  \ingroup PdsVectorGroup
 */
int pds_vector_set_value(PdsVector *Vector, PdsRaNatural x, PdsRaReal m)
{
	if(Vector==NULL)	return FALSE;

	if((x>=0)&&(x<Vector->Nel))	Vector->V[x]=m;

	return TRUE;
}


/** \fn int pds_vector_get_nelements(const PdsVector *Vector,PdsRaReal *Nelements)
 *  \brief Devuelve el número de elementos del vector Vector.
 *  \param[in] Vector El vector en consulta.
 *  \param[out] Nelements En donde se guardará el número de elementos del vector Vector.
 *  \return TRUE si todo fue bien o FALSE si no (ej: Vector==NULL). 
 *  \ingroup PdsVectorGroup
 */
int pds_vector_get_nelements(const PdsVector *Vector,PdsRaReal *Nelements)
{
	*Nelements=0;

	if(Vector==NULL)	return FALSE;

	*Nelements=Vector->Nel;


	return TRUE;
}


/** \fn int pds_vector_get_min_value(const PdsVector *Vector, PdsRaReal *m)
 *  \brief Devuelve el valor mínimo de todos los elementos del vector.
 *  \param[in] Vector El vector en consulta.
 *  \param[out] m El valor mínimo de todos los elementos del vector Vector. En 
 *  caso de error por vector nulo carga cero en m.
 *  \return TRUE si todo fue bien o FALSE si no (ej: Vector==NULL). 
 *  \ingroup PdsVectorGroup
 */
int pds_vector_get_min_value(const PdsVector *Vector, PdsRaReal *m)
{
	PdsRaNatural i;
	*m=0;

	if(Vector==NULL)	return FALSE;

	*m=Vector->V[0];
	for(i=1;i<Vector->Nel;i++) 
	{
		if(Vector->V[i]<*m) *m=Vector->V[i];
	}

	return TRUE;
}


/** \fn int pds_vector_get_max_value(const PdsVector *Vector, PdsRaReal *m)
 *  \brief Devuelve el valor máximo de todos los elementos del vector.
 *  \param[in] Vector El vector en consulta.
 *  \param[out] m El valor máximo de todos los elementos del vector Vector. En 
 *  caso de error por vector nulo carga cero en m.
 *  \return TRUE si todo fue bien o FALSE si no (ej: Vector==NULL). 
 *  \ingroup PdsVectorGroup
 */
int pds_vector_get_max_value(const PdsVector *Vector, PdsRaReal *m)
{
	PdsRaNatural i;
	*m=0;

	if(Vector==NULL)	return FALSE;

	*m=Vector->V[0];
	for(i=1;i<Vector->Nel;i++) 
	{
		if(Vector->V[i]>*m) *m=Vector->V[i];
	}

	return TRUE;
}


/** \fn int pds_vector_swap_elements(PdsVector *Vector, PdsRaNatural X, PdsRaNatural Y)
 *  \brief Intercambia los elementos de la posición X e Y del vector.
 *  \param[in] Vector El vector en consulta.
 *  \param[out] X Posicion del vector. X in [0,Nel-1].
 *  \param[out] Y Posicion del vector. Y in [0,Nel-1].
 *  \return TRUE si todo fue bien o FALSE si no (ej: Vector==NULL o X,Y fuera de rango). 
 *  \ingroup PdsVectorGroup
 */
int pds_vector_swap_elements(PdsVector *Vector, PdsRaNatural X, PdsRaNatural Y)
{
	PdsRaNatural i;
	PdsRaReal tmp;

	if(Vector==NULL)	return FALSE;
	if((X<0)||(X>=Vector->Nel))	return FALSE;
	if((Y<0)||(Y>=Vector->Nel))	return FALSE;
	
	tmp=Vector->V[X];
	Vector->V[X]=Vector->V[Y];
	Vector->V[Y]=tmp;

	return TRUE;
}


/** \fn int pds_vector_asc_order(PdsVector *Vector)
 *  \brief Ordena ascendentemente todos los elementos del vector.
 *  \param[in,out] Vector El vector en consulta.
 *  \return TRUE si todo fue bien o FALSE si no (ej: Vector==NULL). 
 *  \ingroup PdsVectorGroup
 */
int pds_vector_asc_order(PdsVector *Vector)
{
	PdsRaNatural i,j;
	PdsRaReal tmp;

	if(Vector==NULL)	return FALSE;

	for(i=0;i<Vector->Nel;i++) 
	{
		//Al final el maximo esta en la posicion (Vector->Nel-i-1)
		for(j=1;j<(Vector->Nel-i);j++)
		{
			//Al final V[j-1]<V[j]
			if(Vector->V[j]<Vector->V[j-1])
			{
				tmp=Vector->V[j];
				Vector->V[j]=Vector->V[j-1];
				Vector->V[j-1]=tmp;
			}
			//printf("[%d] V:\t",i);pds_vector_printf(Vector);	
		}
	}

	return TRUE;
}


/** \fn int pds_vector_desc_order(PdsVector *Vector)
 *  \brief Ordena descendentemente todos los elementos del vector.
 *  \param[in,out] Vector El vector en consulta.
 *  \return TRUE si todo fue bien o FALSE si no (ej: Vector==NULL). 
 *  \ingroup PdsVectorGroup
 */
int pds_vector_desc_order(PdsVector *Vector)
{
	PdsRaNatural i,j;
	PdsRaReal tmp;

	if(Vector==NULL)	return FALSE;

	for(i=0;i<Vector->Nel;i++) 
	{
		//Al final el minimo esta en la posicion (Vector->Nel-i-1)
		for(j=1;j<(Vector->Nel-i);j++)
		{
			//Al final V[j-1]>V[j]
			if(Vector->V[j]>Vector->V[j-1])
			{
				tmp=Vector->V[j];
				Vector->V[j]=Vector->V[j-1];
				Vector->V[j-1]=tmp;
			}
		}
	}

	return TRUE;
}


/** \fn int pds_vector_set_bcd(PdsVector *Vector,PdsRaNatural n,PdsRaNatural b0,PdsRaNatural b1)
 *  \brief Escribe en formato BCD el numero n en el vector Vector.
 *  \param[in,out] Vector El vector a escribir.
 *  \param[in] n EL número decimal e escribir en BCD en el vector Vector
 *  \param[in] b0 El valor que será escrito cuando se representa a un bit 0.
 *  \param[in] b1 El valor que será escrito cuando se representa a un bit 1.
 *  \return TRUE si todo fue bien o FALSE si no (ej: Vector==NULL). 
 *  \ingroup PdsVectorGroup
 */
int pds_vector_set_bcd(PdsVector *Vector,PdsRaNatural n,PdsRaNatural b0,PdsRaNatural b1)
{
	PdsRaNatural i;
	
	if(Vector==NULL)			return FALSE;

	//creando bcd
	for(i=0;i<Vector->Nel;i++)
	{
		if(n%2==0)	Vector->V[i]=b0;
		else		Vector->V[i]=b1;
		n=n/2;
	}

	return TRUE;

}


/** \fn int pds_vector_set_gray(PdsVector *Vector,PdsRaNatural n,PdsRaNatural b0,PdsRaNatural b1)
 *  \brief Escribe en formato Gray el numero n en el vector Vector.
 *  \param[in,out] Vector El vector a escribir.
 *  \param[in] n EL número decimal e escribir en Gray en el vector Vector
 *  \param[in] b0 El valor que será escrito cuando se representa a un bit 0.
 *  \param[in] b1 El valor que será escrito cuando se representa a un bit 1.
 *  \return TRUE si todo fue bien o FALSE si no (ej: Vector==NULL). 
 *  \ingroup PdsVectorGroup
 */
int pds_vector_set_gray(PdsVector *Vector,PdsRaNatural n,PdsRaNatural b0,PdsRaNatural b1)
{
	PdsRaNatural i;
	
	if(Vector==NULL)			return FALSE;

	//creando bcd
	for(i=0;i<Vector->Nel;i++)
	{
		Vector->V[i]=n%2;
		n=n/2;
	}

	//Pasando a Gray
	for(i=0;i<(Vector->Nel-1);i++)
	{
		if(Vector->V[i]==Vector->V[i+1])	Vector->V[i]=b0;
		else					Vector->V[i]=b1;
	}
	return TRUE;
}


/** \fn int pds_vector_cmp(const PdsVector *Vector1,const PdsVector *Vector2,PdsRaNatural *n)
 *  \brief Compara los vectores Vector1 y Vector2 y retorna la cantidad de elementos distintos.
 *  \param[in] Vector1 Un vector a comparar.
 *  \param[in] Vector2 Un vector a comparar.
 *  \param[out] n EL número de elementos diferentes entre los vectores.
 *  \return TRUE si todo fue bien o FALSE si no (ej: Vector1==NULL o Vector2==NULL). 
 *  \ingroup PdsVectorGroup
 */
int pds_vector_cmp(const PdsVector *Vector1,const PdsVector *Vector2,PdsRaNatural *n)
{
	PdsRaNatural i,j;
	
	if(Vector1==NULL)			return FALSE;
	if(Vector2==NULL)			return FALSE;
	if(Vector1->Nel!=Vector2->Nel)		return FALSE;

	for(i=0,j=0;i<Vector1->Nel;i++)
	{
		if(Vector1->V[i]!=Vector2->V[i])	j++;
	}

	*n=j;

	return TRUE;
}

/** \fn void pds_vector_free(PdsVector *Vector)
 *  \brief Libera un vector de tipo puntero PdsVector.
 *  \param[in,out] Vector El vector a liberar.
 *  \return No retorna valor.
 *  \ingroup PdsVectorGroup
 */
void pds_vector_free(PdsVector *Vector)
{
	if(Vector!=NULL)
	{
		free(Vector->V);
		free(Vector);
	}
}


/** \fn void pds_vector_destroy(PdsVector **Vector)
 *  \brief Libera un vector de tipo puntero PdsVector, y limpia el puntero con NULL.
 *  \param[in,out] Vector El vector a liberar y limpiar.
 *  \return No retorna valor.
 *  \ingroup PdsVectorGroup
 */
void pds_vector_destroy(PdsVector **Vector)
{
	if((*Vector)!=NULL)
	{
		free((*Vector)->V);
		free(*Vector);
		(*Vector)=NULL;
	}
}

