/*
 * pdsfifo.h
 * 
 * Copyright 2011 Fernando Pujaico Rivera <fernando.pujaico.rivera@gmail.com>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 * 
 */

/** \file pdsfifo.h
 *  \author Fernando Pujaico Rivera
 *  \date 02-02-2012
 *  \brief Funciones que trabajan con fifos.
 *  
 *  <br>Estas funciones trabajan con una fifo de la forma.<br>
 *  \image html fifo.png "Fifo de N elementos como máximo y ancho M."
 */

#ifndef __PDSFIFO_H__
#define __PDSFIFO_H__

#ifdef __cplusplus
extern "C" {
#endif 

#include <stdio.h>
#include <stdlib.h>
#include <pds/pdsraglobal.h>
#include <pds/pdsvector.h>

#ifndef TRUE
	#define TRUE 1
#endif

#ifndef FALSE
	#define FALSE 0
#endif

/** \defgroup PdsFifoGroup Funciones del módulo PdsFifo.
 *  \brief Funciones que trabajan con fifos, puede usarse para lectura y escritura 
 *  de forma asincrona, por ejemplo cuando se usa hilos.
 *  
 *  <br>Estas funciones trabajan con una fifo de la forma.<br>
 *  \image html fifo.png "Fifo de N elementos como máximo y ancho M."
 * @{
 */

/*! \def pds_fifo_nnodes(FIFO)
 *  Devuelve el número de elementos en la fifo.
 *  \ingroup PdsFifoGroup
 */
#define pds_fifo_nnodes(FIFO) FIFO->Nnodes

/*! \struct PdsFifo
 *  \brief La estructura tipo  PdsFifo .
 *  Esta estructura genera una fifo de N elementos como máximo y ancho M, puede 
 *  usarse para lectura y escritura de forma asincrona, por ejemplo cuando se usa hilos.
 *  Para usar incluir pds/pdsfifo.h.
 *
 *  <br>Estas funciones trabajan con una fifo de la forma.<br>
 *  \image html fifo.png "Fifo de N elementos como máximo y ancho M."
 *  \ingroup PdsFifoGroup
 *  \author Fernando Pujaico Rivera
 */
typedef struct
{
	// Arreglo de vectores reales.
	PdsVector	**V;

	// Arreglo que indica si el vector V[i] esta lleno (E[i]==1) o vacio (E[i]==0).
	PdsRaNatural	*E;

	// Número máximo de elementos en la fifo.
	PdsRaNatural	N;

	// Tamaño del vector correspondiente a cada elemento de la fifo.
	PdsRaNatural	M;

	// ID de la siguiente posición a leer.
	PdsRaNatural	ReadID;

	// ID de la siguiente posición a escribir.
	PdsRaNatural	WriteID;

	// Número de elementos usados en la fifo.
	PdsRaNatural	Nnodes;
}PdsFifo;

/** \fn PdsFifo *pds_fifo_new(PdsRaNatural N,PdsRaNatural M)
 *  \brief Crea una fifo de tipo PdsFifo, la fifo estará vacía y aceptará como
 *  máximo N vectores en la cola de tamaño M. La fifo está preparada para atender
 *  dos procesos que intenten simultáneamente leer y escribir en ella.
 *  \param[in] N Es el número máximo de elementos en la cola.
 *  \param[in] M Es el número de elementos de cada vector de la fifo.
 *  \return Un puntero a la Fifo de tipo PdsFifo.
 *  \ingroup PdsFifoGroup
 */
PdsFifo *pds_fifo_new(PdsRaNatural N,PdsRaNatural M);


/** \fn int pds_fifo_write(PdsFifo *FIFO,const PdsVector *V)
 *  \brief Escribe en la fifo FIFO el vector V. Si el tamaño del vector V y el 
 *  ancho de la fifo son diferentes se intersectan los tamaños y se hace la copia 
 *  en la intersección solamente. 
 *  \param[in,out] FIFO La fifo a escribir.
 *  \param[in] V El vector con los datos a escribir.
 *  \return TRUE si todo fue bien o FALSE si no. (ej. FIFO==NULL, V==NULL o Número 
 *  máximo de elementos )
 *  \ingroup PdsFifoGroup
 */
int pds_fifo_write(PdsFifo *FIFO,const PdsVector *V);


/** \fn int pds_fifo_queue(const PdsFifo *FIFO,PdsRaNatural n)
 *  \brief Indica si existen como mínimo n vectores en el buffer. 
 *  \param[in] FIFO La fifo a leer.
 *  \param[in] n El número de vectores a leer.
 *  \return TRUE si todo fue bien y existen como mínimo n elementos o
 *  FALSE si no. (ej. FIFO==NULL, n==0, o fifo vacía )
 *  \ingroup PdsFifoGroup
 */
int pds_fifo_queue(const PdsFifo *FIFO,PdsRaNatural n);


/** \fn int pds_fifo_read(PdsFifo *FIFO,PdsVector *V)
 *  \brief Lee en la fifo FIFO el vector V. Si el tamaño del vector V y el 
 *  ancho de la fifo son diferentes se intersectan los tamaños y se hace la copia 
 *  en la intersección solamente. 
 *  \param[in,out] FIFO La fifo a leer.
 *  \param[in] V El vector donde se copiará los datos.
 *  \return TRUE si todo fue bien o FALSE si no. (ej. FIFO==NULL, V==NULL )
 *  \ingroup PdsFifoGroup
 */
int pds_fifo_read(PdsFifo *FIFO,PdsVector *V);


/** \fn int pds_fifo_printf(const PdsFifo *FIFO)
 *  \brief Imprime en pantalla los vectores de la fifo FIFO, desde el primero en
 *  ser atendido en la cola hasta el último.
 *  \param[in] FIFO La fifo a imprimir en pantalla.
 *  \return TRUE si todo fue bien o FALSE si no. (ejem FIFO==NULL)
 *  \ingroup PdsFifoGroup
 */
int pds_fifo_printf(const PdsFifo *FIFO);


/** \fn int pds_fifo_isfull(const PdsFifo *FIFO)
 *  \brief Indica si la fifo esta llena. 
 *  \param[in] FIFO La fifo a leer.
 *  \return TRUE si todo fue bien y esta llena o
 *  FALSE si no. (ej. FIFO==NULL)
 *  \ingroup PdsFifoGroup
 */
int pds_fifo_isfull(const PdsFifo *FIFO);


/** \fn int pds_fifo_isempty(const PdsFifo *FIFO)
 *  \brief Indica si la fifo esta vacia. 
 *  \param[in] FIFO La fifo a leer.
 *  \return TRUE si todo fue bien y esta vacia o
 *  FALSE si no. (ej. FIFO==NULL)
 *  \ingroup PdsFifoGroup
 */
int pds_fifo_isempty(const PdsFifo *FIFO);


/** \fn void pds_fifo_free(PdsFifo *FIFO)
 *  \brief Libera una fifo de tipo puntero PdsFifo.
 *  \param[in,out] FIFO La fifo a liberar.
 *  \return No retorna valor.
 *  \ingroup PdsFifoGroup
 */
void pds_fifo_free(PdsFifo *FIFO);


/** \fn void pds_fifo_destroy(PdsFifo **FIFO)
 *  \brief Libera una fifo de tipo puntero PdsFifo, y limpia el puntero con NULL.
 *  \param[in,out] FIFO La fifo a liberar y limpiar.
 *  \return No retorna valor.
 *  \ingroup PdsFifoGroup
 */
void pds_fifo_destroy(PdsFifo **FIFO);


/**
 * @}
 */

#ifdef __cplusplus
}
#endif 

#endif


