/*
 * pdsmatrix.h
 * 
 * Copyright 2011 Fernando Pujaico Rivera <fernando.pujaico.rivera@gmail.com>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 * 
 */

/** \file pdsmatrix.h
 *  \author Fernando Pujaico Rivera
 *  \date 01-04-2011
 *  \brief Funciones que trabajan con matrices.
 *  
 *  <br>Estas funciones trabajan con una matriz de la forma.<br>
 *  \image html matrix.png "Matriz de NlinxNcol"
 *  \b Nlin es el número de lineas y \b Ncol es el número de columnas.
 */

#ifndef __PDSMATRIX_H__
#define __PDSMATRIX_H__

#ifdef __cplusplus
extern "C" {
#endif 

#include <stdio.h>
#include <stdlib.h>
#include <pds/pdsraglobal.h>
#include <pds/pdsvector.h>

#ifndef TRUE
	#define TRUE 1
#endif

#ifndef FALSE
	#define FALSE 0
#endif

/** \defgroup PdsMatrixGroup Módulo PdsMatrix.
 *  \brief Funciones que trabajan con matrices.
 *  
 *  <br>Estas funciones trabajan con una matriz de la forma.<br>
 *  \image html matrix.png "Matriz de NlinxNcol"
 *  \b Nlin es el número de lineas y \b Ncol es el número de columnas.
 *  
 * Informacion adicional puede ser encontrada en @cite tutorialmatvec
 * @{
 */

/*! \def _MATSETVAL(Matrix,X,Y,m)
 *  Es equivalente a Matrix(X,Y)=m.
 *  \ingroup PdsMatrixGroup
*/
#define _MATSETVAL(Matrix,X,Y,m) pds_matrix_set_value(Matrix,X,Y,m)

/*! \struct PdsMatrix
 *  \brief La estructura tipo  PdsMatrix .
 *  Esta estructura genera una matriz de Nlin lineas y Ncol columnas.
 *  Para usar incluir pds/pdsmatrix.h.
 *  \ingroup PdsMatrixGroup
 *  \author Fernando Pujaico Rivera
 */
typedef struct 
{
	/*! Un arreglo de Nlin lineas y Ncol elementos por linea. */
	PdsRaReal **M;
	/*! Número de lineas. */
	PdsRaNatural Nlin;
	/*! Número de columnas. */
	PdsRaNatural Ncol;
}PdsMatrix;


/** \fn PdsMatrix *pds_matrix_new(PdsRaNatural Nlin,PdsRaNatural Ncol)
 *  \brief Crea una matriz de tipo PdsMatrix e inicia con cero todos los elementos.
 *  \param[in] Nlin Es el número de lineas de la matriz.
 *  \param[in] Ncol Es el número de columnas de la matriz.
 *  \return Un puntero a la matriz de tipo PdsMatrix.
 *  \ingroup PdsMatrixGroup
 */
PdsMatrix *pds_matrix_new(PdsRaNatural Nlin,PdsRaNatural Ncol);


/** \fn PdsMatrix *pds_matrix_new_matrix(const PdsMatrix *MatSrc)
 *  \brief Crea una matriz de tipo PdsMatrix a partir de la matriz MatSrc.
 *  \param[in] MatSrc Matriz que se usará como imagen como fuente.
 *  \return Un puntero a la matriz de tipo PdsMatrix.
 *  \ingroup PdsMatrixGroup
 */
PdsMatrix *pds_matrix_new_matrix(const PdsMatrix *MatSrc);


/** \fn PdsMatrix *pds_matrix_new_load_data(const char* datafile)
 *  \brief Crea un nuevo vector e inicia los datos con los elementos de la linea
 *  line del archivo datafile.
 *  Usa TAB como delimitador de elemento y un salto de linea como delimitador de linea.
 *  Si se pide una linea inexistente la funcion retorna NULL
 *  \param[in] datafile Nombre del archivo de donde se cargará los datos iniciales
 *  del vector.
 *  \return Un puntero que apunta a la dirección del nuevo vector, si todo fue bien, 
 *  o NULL si no. (ej. datafile==NULL o line inexistente)
 *  \ingroup PdsMatrixGroup
 */
PdsMatrix *pds_matrix_new_load_data(const char* datafile);

/** \fn int pds_matrix_copy_matrix(PdsMatrix *Matrix,PdsRaNatural X1,PdsRaNatural Y1,const PdsMatrix *MatSrc,PdsRaNatural X2,PdsRaNatural Y2)
 *  \brief Copia en la matriz Matrix en la posicion (X1,Y1) la matriz MatSrc 
 *  desde la posicion (X2,Y2).
 *  Si los tamaños son diferentes intersecta los tamaños y hace la copia en la
 *  intersección solamente. <br><b>Matrix(X1,Y1)=MatSrc(X2,Y2)</b>.
 *  \param[in,out] Matrix La matriz a copiar.
 *  \param[in] X1 Linea en la que se copiará.
 *  \param[in] Y1 Columna en la que se copiará.
 *  \param[in] MatSrc Matriz que se usara como fuente.
 *  \param[in] X2 Linea que se copiará.
 *  \param[in] Y2 Columna que se copiará.
 *  \return TRUE si las matrices existen o FALSE si no.
 *  \ingroup PdsMatrixGroup
 */
int pds_matrix_copy_matrix(PdsMatrix *Matrix,PdsRaNatural X1,PdsRaNatural Y1,const PdsMatrix *MatSrc,PdsRaNatural X2,PdsRaNatural Y2);


/** \fn int pds_matrix_copy_vector_col(PdsMatrix *Matrix,const PdsVector *VecSrc,PdsRaNatural col)
 *  \brief Copia vector VecSrc en la columna col de la matriz Matrix.
 *  Si los tamaños son diferentes intersecta los tamaños y hace la copia en la
 *  intersección solamente. <br><b>Matrix(:,col)=VecSrc</b>.
 *  \param[in,out] Matrix La matriz a copiar.
 *  \param[in] VecSrc Vector que se usará como fuente.
 *  \param[in] col Columna a escribir.
 *  \return TRUE si las matrices existen o FALSE si no.
 *  \ingroup PdsMatrixGroup
 */
int pds_matrix_copy_vector_col(PdsMatrix *Matrix,const PdsVector *VecSrc,PdsRaNatural col);


/** \fn int pds_matrix_copy_vector_lin(PdsMatrix *Matrix,const PdsVector *VecSrc,PdsRaNatural lin)
 *  \brief Copia vector VecSrc en la linea lin de la matriz Matrix.
 *  Si los tamaños son diferentes intersecta los tamaños y hace la copia en la
 *  intersección solamente. <br><b>Matrix(:,col)=VecSrc</b>.
 *  \param[in,out] Matrix La matriz a copiar.
 *  \param[in] VecSrc Vector que se usará como fuente.
 *  \param[in] lin Linea a escribir.
 *  \return TRUE si las matrices existen o FALSE si no.
 *  \ingroup PdsMatrixGroup
 */
int pds_matrix_copy_vector_lin(PdsMatrix *Matrix,const PdsVector *VecSrc,PdsRaNatural lin);


/** \fn int pds_matrix_copy_identity(PdsMatrix *Matrix,PdsRaNatural X,PdsRaNatural Y,PdsRaReal Val)
 *  \brief Copia en la matriz Matrix una matriz identidad(Val) en la posicion (X,Y).
 *  Si los tamaños son diferentes intersecta los tamaños y hace la copia en la
 *  intersección solamente. <br><b>Matrix(NLin,NCol)=MatSrc</b>.
 *  \param[in,out] Matrix La matriz a copiar.
 *  \param[in] X Linea en la que se copiará.
 *  \param[in] Y Columna en la que se copiará.
 *  \param[in] Val El valor que será escrito en la diagonal.
 *  \return TRUE si las matrices existen o FALSE si no.
 *  \ingroup PdsMatrixGroup
 */
int pds_matrix_copy_identity(PdsMatrix *Matrix,PdsRaNatural X,PdsRaNatural Y,PdsRaReal Val);


/** \fn int pds_matrix_init_identity(PdsMatrix *Matrix,PdsRaReal Val)
 *  \brief Inicia la matriz de tipo puntero PdsMatrix con una diagonal de valor 
 *  Val y llena el resto de ceros. <br><b>Matrix=Val*I</b>.
 *  \param[in,out] Matrix La matriz a iniciar.
 *  \param[in] Val El valor que será escrito en la diagonal.
 *  \return TRUE si todo fue bien o FALSE si no.
 *  \ingroup PdsMatrixGroup
 */
int pds_matrix_init_identity(PdsMatrix *Matrix,PdsRaReal Val);


/** \fn int pds_matrix_init_matrix(PdsMatrix *Matrix,const PdsMatrix *MatSrc)
 *  \brief Inicia la matriz de tipo puntero PdsMatrix con una matriz.
 *  Si los tamaños son diferentes intersecta los tamaños y hace la copia en la
 *  intersección solamente. <br><b>Matrix=MatSrc</b>.
 *  \param[in,out] Matrix La matriz a iniciar.
 *  \param[in] MatSrc Matriz que se usara como fuente.
 *  \return TRUE si todo fue bien o FALSE si no.
 *  \ingroup PdsMatrixGroup
 */
int pds_matrix_init_matrix(PdsMatrix *Matrix,const PdsMatrix *MatSrc);


/** \fn int pds_matrix_init_value(PdsMatrix *Matrix,PdsRaReal Val)
 *  \brief Inicia la matriz de tipo puntero PdsMatrix con un valor. <br><b>Matrix=Val</b>.
 *  \param[in,out] Matrix La matriz a iniciar con un valor.
 *  \param[in] Val Es el valor inicial de los elementos.
 *  \return TRUE si todo fue bien o FALSE si no.
 *  \ingroup PdsMatrixGroup
 */
int pds_matrix_init_value(PdsMatrix *Matrix,PdsRaReal Val);


/** \fn int pds_matrix_add_identity(PdsMatrix *Matrix,PdsRaReal Val)
 *  \brief Suma a la matriz de tipo puntero PdsMatrix con una diagonal de valor 
 *  Val y suma cero al resto. <br><b>Matrix=Matrix+Val*I</b>.
 *  \param[in,out] Matrix La matriz destino y operando de la suma.
 *  \param[in] Val El valor que será sumado a la diagonal.
 *  \return TRUE si todo fue bien o FALSE si no.
 *  \ingroup PdsMatrixGroup
 */
int pds_matrix_add_identity(PdsMatrix *Matrix,PdsRaReal Val);


/** \fn int pds_matrix_add_matrix(PdsMatrix *Matrix,const PdsMatrix *MatSrc)
 *  \brief Suma la matriz de tipo puntero PdsMatrix con una matriz MatSrc y lo 
 *  carga en Matrix. Si los tamaños son diferentes intersecta los tamaños y hace 
 *  la copia en la  intersección solamente. <br><b>Matrix=Matrix+MatSrc</b>.
 *  \param[in,out] Matrix La matriz que se usará como destino y operando.
 *  \param[in] MatSrc Matriz que se sumará a Matrix.
 *  \return TRUE si todo fue bien o FALSE si no.
 *  \ingroup PdsMatrixGroup
 */
int pds_matrix_add_matrix(PdsMatrix *Matrix,const PdsMatrix *MatSrc);


/** \fn int pds_matrix_add_value(PdsMatrix *Matrix,PdsRaReal Val)
 *  \brief Suma a la matriz de tipo puntero PdsMatrix un valor. <br><b>Matrix=Matrix+Val</b>.
 *  \param[in,out] Matrix La matriz a la cual se le sumará un valor.
 *  \param[in] Val Es el valor que se sumará a los elementos.
 *  \return TRUE si todo fue bien o FALSE si no.
 *  \ingroup PdsMatrixGroup
 */
int pds_matrix_add_value(PdsMatrix *Matrix,PdsRaReal Val);


/** \fn int pds_matrix_sub_matrix(PdsMatrix *Matrix,const PdsMatrix *MatSrc)
 *  \brief Resta la matriz de tipo puntero PdsMatrix con una matriz MatSrc y lo 
 *  carga en Matrix. Si los tamaños son diferentes intersecta los tamaños y hace 
 *  la copia en la  intersección solamente. <br><b>Matrix=Matrix-MatSrc</b>.
 *  \param[in,out] Matrix La matriz que se usará como destino y operando.
 *  \param[in] MatSrc Matriz que se sumará a Matrix.
 *  \return TRUE si todo fue bien o FALSE si no.
 *  \ingroup PdsMatrixGroup
 */
int pds_matrix_sub_matrix(PdsMatrix *Matrix,const PdsMatrix *MatSrc);


/** \fn int pds_matrix_mul_matrix_elements(PdsMatrix *Matrix,const PdsMatrix *MatSrc)
 *  \brief Multiplica la matriz de tipo puntero PdsMatrix con una matriz MatSrc 
 *  elemento a elemento y lo carga en Matrix. Si los tamaños son diferentes
 *  intersecta los tamaños y hace la copia en la  intersección solamente.  <br><b>Matrix=Matrix.MatSrc</b>.
 *  \param[in,out] Matrix La matriz que se usará como destino y operando.
 *  \param[in] MatSrc Matriz que se multiplicará a Matrix.
 *  \return TRUE si todo fue bien o FALSE si no.
 *  \ingroup PdsMatrixGroup
 */
int pds_matrix_mul_matrix_elements(PdsMatrix *Matrix,const PdsMatrix *MatSrc);


/** \fn int pds_matrix_mul_matrix(PdsMatrix *Matrix,const PdsMatrix *MatSrc1,const PdsMatrix *MatSrc2)
 *  \brief Multiplica la matriz MatSrc1 (Nlin1xNcol1) con la matriz MatSrc2 (Nlin2xNcol2)
 *  y lo carga en Matrix (Nlin1xNcol2). <br><b>Matrix=MatSrc1*MatSrc2</b>.
 *  \param[in,out] Matrix La matriz que se usará como destino, esta matriz ya debe 
 *  estar creada, todo su contenido será sobrescrito.
 *  \param[in] MatSrc1 Matriz que se multiplicará a MatSrc2.
 *  \param[in] MatSrc2 Matriz que se multiplicará a MatSrc1.
 *  \return TRUE si todo fue bien o FALSE si no.
 *  Puede retornar FALSE si alguna de las matrices es NULL, o si el número de 
 *  lineas (Nlin1) de Matrix no corresponde al número de lineas (Nlin1) de MatSrc1, 
 *  o si el número de columnas (Ncol2) de Matrix no corresponde al número de 
 *  columnas (Ncol2) de MatSrc2, o si Ncol1!=Nlin2, e cualquiera de esos errores
 *  la matriz Matrix no es alterada.
 *  \ingroup PdsMatrixGroup
 */
int pds_matrix_mul_matrix(PdsMatrix *Matrix,const PdsMatrix *MatSrc1,const PdsMatrix *MatSrc2);


/** \fn PdsMatrix *pds_matrix_mul_matrix_new(const PdsMatrix *MatSrc1,const PdsMatrix *MatSrc2)
 *  \brief Multiplica la matriz MatSrc1 (Nlin1xNcol1) con la matriz MatSrc2 (Nlin2xNcol2)
 *  y devuelve una matriz Matrix (Nlin1xNcol2). <br><b>Matrix=MatSrc1*MatSrc2</b>.
 *  \param[in] MatSrc1 Matriz que se multiplicará a MatSrc2.
 *  \param[in] MatSrc2 Matriz que se multiplicará a MatSrc1.
 *  \return La multiplicación de las matrices si todo fue bien o NULL si no.
 *  Puede retornar NULL si alguna de las matrices es NULL, o si Ncol1!=Nlin2.
 *  \ingroup PdsMatrixGroup
 */
PdsMatrix *pds_matrix_mul_matrix_new(const PdsMatrix *MatSrc1,const PdsMatrix *MatSrc2);


/** \fn int pds_matrix_mul_vector(PdsVector *A,const PdsMatrix *M,const PdsVector *V)
 *  \brief Multiplica la matriz M con el vector V y devuelve un vector A. 
 *  <br><b>A=M*V</b>.
 *  \param[out] A Vector donde se cargará la multiplicación.
 *  \param[in] M Matriz que se multiplicará.
 *  \param[in] V Vector que se multiplicará.
 *  \return TRUE si todo fue bien o FALSE si no.
 *  Puede retornar FALSE si la matriz no tiene el mismo número de columnas que elementos
 *  tiene V. O si el numero de elementos de A es distinto al número de lineas de M.
 *  \ingroup PdsMatrixGroup
 */
int pds_matrix_mul_vector(PdsVector *A,const PdsMatrix *M,const PdsVector *V);


/** \fn PdsVector *pds_matrix_mul_vector_new(const PdsMatrix *M,const PdsVector *V)
 *  \brief Multiplica la matriz M con el vector V y devuelve un vector A. 
 *  <br><b>A=M*V</b>.
 *  \param[in] M Matriz que se multiplicará.
 *  \param[in] V Vector que se multiplicará.
 *  \return La multiplicación de la matriz por el vector si todo fue bien o NULL si no.
 *  Puede retornar NULL si la matriz no tiene el mismo número de columnas que elementos
 *  tiene V. O si el numero de elementos de A es distinto al número de lineas de M.
 *  \ingroup PdsMatrixGroup
 */
PdsVector *pds_matrix_mul_vector_new(const PdsMatrix *M,const PdsVector *V);


/** \fn int pds_matrix_mul_value(PdsMatrix *Matrix,PdsRaReal Val)
 *  \brief Multiplica a la matriz de tipo puntero PdsMatrix por un valor.  <br><b>Matrix=Val*Matrix</b>.
 *  \param[in,out] Matrix La matriz a la cual se le multiplicará un valor.
 *  \param[in] Val Es el valor que se sumará a los elementos.
 *  \return TRUE si todo fue bien o FALSE si no.
 *  \ingroup PdsMatrixGroup
 */
int pds_matrix_mul_value(PdsMatrix *Matrix,PdsRaReal Val);


/** \fn PdsMatrix *pds_matrix_transpose_new(const PdsMatrix *M)
 *  \brief Genera la transpuesta de la matriz.  <br><b>Mt=M^T</b>.
 *  \param[in] M La matriz a la cual se le aplicará la transpuesta.
 *  \return La transpuesta de la matriz o NULL en caso de error.(ej. M==NULL).
 *  \ingroup PdsMatrixGroup
 */
PdsMatrix *pds_matrix_transpose_new(const PdsMatrix *M);


/** \fn int pds_matrix_transpose(PdsMatrix *Mt,const PdsMatrix *M)
 *  \brief Genera la transpuesta de la matriz.  <br><b>Mt=M^T</b>.
 *  \param[in] Mt La transpuesta de la matriz M.
 *  \param[in] M La matriz a la cual se le aplicará la transpuesta.
 *  \return RUE si todo fue bien o FALSE si no.(ej. M==NULL Mt==NULL).
 *  \ingroup PdsMatrixGroup
 */
int pds_matrix_transpose(PdsMatrix *Mt,const PdsMatrix *M);


/** \fn PdsMatrix *pds_matrix_mtm_new(const PdsMatrix *M)
 *  \brief Genera  <br><b>M^T * M</b>.
 *  \param[in] M La matriz a la cual se le aplicará la operación.
 *  \return La <br><b>M^T * M</b> de la matriz o NULL en caso de error.(ej. M==NULL).
 *  \ingroup PdsMatrixGroup
 */
PdsMatrix *pds_matrix_mtm_new(const PdsMatrix *M);


/** \fn PdsMatrix *pds_matrix_mtm_ai_new(const PdsMatrix *M,PdsRaReal Alpha)
 *  \brief Genera  <br><b>M^T * M + Alpha * I</b>.
 *  \param[in] M La matriz a la cual se le aplicará la operación.
 *  \param[in] Alpha Factor que se multiplicará a la identidad.
 *  \return La <br><b>M^T * M + Alpha * I</b> de la matriz o NULL en caso de 
 *  error.(ej. M==NULL).
 *  \ingroup PdsMatrixGroup
 */
PdsMatrix *pds_matrix_mtm_ai_new(const PdsMatrix *M,PdsRaReal Alpha);


/** \fn int pds_matrix_linear_least_square(const PdsVector *U,const PdsMatrix *H,PdsVector *Z)
 *  \brief Encuentra Z  <br><b>U=HZ</b><br><b>Z=(H'H)^(-1)H'U</b>.
 *
 *  \f[ e^2=||U - H Z||^2 \f]
 *  \f[ \frac{\partial e^2}{\partial Z}=0 \f]
 *  \f[ Z=( H^T H )^{-1} H^T U \f]
 *  \param[in] U Valor de salida del sistema de ecuaciones <b>U=HZ</b>.
 *  \param[in] H Matriz del sistema.
 *  \param[out] Z Incognita del sistema, aquí es cargado el resultado.
 *  \return TRUE si todo fue bien o FALSE si no. (ej U==NULL,H==NULL o Z==NULL)
 *  \ingroup PdsMatrixGroup
 */
int pds_matrix_linear_least_square(const PdsVector *U,const PdsMatrix *H,PdsVector *Z);


/** \fn int pds_matrix_tikhonov_nolinear_least_square(const PdsVector *U,const PdsMatrix *J,const PdsVector *F,PdsRaReal Alpha,PdsVector *Z)
 *  \brief Encuentra Z  una iteración de <br><b>U=F(Z)
 *  </b><br><b>Z=Z+(J'J + Alpha I)^(-1)(J'(U-F(Z))-Alpha Z)</b>.
 *
 *  \f[ e^2={||U-F(Z)||}^2+\alpha {||Z||}^2 \f]
 *  \f[ \frac{\partial e^2}{\partial Z}=0 \f]
 *  \f[ Z=Z+ {[ {J(Z)}^T {J(Z)} + \alpha I]}^{-1} [{J(Z)}^T (U-F(Z))-\alpha Z] \f]
 *  \param[in] U Valor de salida del sistema de ecuaciones <b>U=HZ</b>.
 *  \param[in] J matriz jacobiano de F con respecto a Z.
 *  \param[in] F Vector que depende de Z.
 *  \param[in] Alpha Factor de regularización.
 *  \param[out] Z Incognita del sistema, aquí es cargado el resultado.
 *  \return TRUE si todo fue bien o FALSE si no. (ej U==NULL,H==NULL o Z==NULL)
 *  \ingroup PdsMatrixGroup
 */
int pds_matrix_tikhonov_nolinear_least_square(const PdsVector *U,const PdsMatrix *J,const PdsVector *F,PdsRaReal Alpha,PdsVector *Z);


/** \fn int pds_matrix_printf(const PdsMatrix *Matrix)
 *  \brief Imprime en pantalla una matriz de tipo puntero PdsMatrix.
 *  \param[in] Matrix La matriz a imprimir en pantalla.
 *  \return TRUE si todo fue bien o FALSE si no.
 *  \ingroup PdsMatrixGroup
 */
int pds_matrix_printf(const PdsMatrix *Matrix);


/** \fn int pds_matrix_fprintf(const PdsMatrix *Matrix,FILE *fd)
 *  \brief Imprime en el archivo que apunta fd una matriz de tipo puntero PdsMatrix.
 *  \param[in] Matrix La matriz a imprimir en fd.
 *  \param[in,out] fd El puntero al archivo.
 *  \return TRUE si todo fue bien o FALSE si no.
 *  \ingroup PdsMatrixGroup
 */
int pds_matrix_fprintf(const PdsMatrix *Matrix,FILE *fd);


/** \fn int pds_matrix_fscanf(PdsMatrix *Matrix, FILE *fd)
 *  \brief Inicializa una matriz con los datos del archivo apuntado por fd.
 *  Usa TAB o un salto de linea como delimitador de elemento.
 *  \param[in] Matrix Matriz en donde se cargaran los datos.
 *  \param[in] fd Apuntador del archivo de donde se cargará los datos iniciales
 *  de la matriz.
 *  \return TRUE si todo fue bien o FALSE si no. (ej. Matrix==NULL o fd==NULL)
 *  \ingroup PdsMatrixGroup
 */
int pds_matrix_fscanf(PdsMatrix *Matrix, FILE *fd);


/** \fn int pds_matrix_fwrite(PdsMatrix *Matrix, FILE *fd)
 *  \brief Escribe los datos de un vector en el archivo binario apuntado por fd.
 *  \param[in] Matrix Matrix de donde se leeran los datos.
 *  \param[in] fd Apuntador del archivo binario de donde se escribiran los datos 
 *  del vector.
 *  \return TRUE si todo fue bien o FALSE si no. (ej. Matrix==NULL o fd==NULL)
 *  \ingroup PdsMatrixGroup
 */
int pds_matrix_fwrite(PdsMatrix *Matrix, FILE *fd);


/** \fn int pds_matrix_fread(PdsMatrix *Matrix, FILE *fd)
 *  \brief Inicializa un vector con los datos del archivo binario apuntado por fd.
 *  \param[in] Matrix Matrix en donde se cargaran los datos.
 *  \param[in] fd Apuntador del archivo binario de donde se cargará los datos 
 *  iniciales del vector.
 *  \return TRUE si todo fue bien o FALSE si no. (ej. Matrix==NULL o fd==NULL)
 *  \ingroup PdsMatrixGroup
 */
int pds_matrix_fread(PdsMatrix *Matrix, FILE *fd);


/** \fn int pds_matrix_get_value(const PdsMatrix *Matrix, PdsRaNatural x,PdsRaNatural y, PdsRaReal *m)
 *  \brief Devuelve el valor en la posición (x,y) de la matriz  Matrix.
 *  (x,y) inicia con (0,0).
 *  \param[in] Matrix La matriz de en consulta.
 *  \param[in] x Linea de posición x, el primer valor es cero.
 *  \param[in] y Columna de posición y, el primer valor es cero.
 *  \param[out] m El valor en la posición (x,y), en caso de error por fuera de rango 
 *  (x,y) entonces carga 0, en caso de error de matriz nula carga cero.
 *  \return TRUE si todo fue bien o FALSE si no (ej: Matrix==NULL). 
 *  \ingroup PdsMatrixGroup
 */
int pds_matrix_get_value(const PdsMatrix *Matrix, PdsRaNatural x,PdsRaNatural y, PdsRaReal *m);


/** \fn int pds_matrix_set_value(PdsMatrix *Matrix, PdsRaNatural x,PdsRaNatural y,PdsRaReal m)
 *  \brief Escribe el valor en la posición (x,y) de la matriz  Matrix.
 *  En caso de que (x,y) estuvieran fuera de rango, no se considera como error
 *  simplemente no se escribe nada. (x,y) inicia con (0,0).
 *  \param[in,out] Matrix La matriz a modificar.
 *  \param[in] x Linea de posición x, el primer valor es cero.
 *  \param[in] y Columna de posición y, el primer valor es cero.
 *  \param[in] m Valor real que se colocara en (x,y).
 *  \return TRUE si todo fue bien o FALSE si no (ej: Matrix==NULL). 
 *  \ingroup PdsMatrixGroup
 */
int pds_matrix_set_value(PdsMatrix *Matrix, PdsRaNatural x,PdsRaNatural y,PdsRaReal m);


/** \fn int pds_matrix_get_nlines(const PdsMatrix *Matrix, PdsRaNatural *Nlines)
 *  \brief Devuelve el número de lineas de la matriz  Matrix.
 *  \param[in] Matrix La matriz de en consulta.
 *  \param[out] Nlines En donde se guardará el número de lineas de la matriz  Matrix.
 *  \return TRUE si todo fue bien o FALSE si no (ej: Matrix==NULL). 
 *  \ingroup PdsMatrixGroup
 */
int pds_matrix_get_nlines(const PdsMatrix *Matrix, PdsRaNatural *Nlines);


/** \fn int pds_matrix_get_ncolumns(const PdsMatrix *Matrix, PdsRaNatural *Ncolumns)
 *  \brief Devuelve el número de columnas de la matriz  Matrix.
 *  \param[in] Matrix La matriz de en consulta.
 *  \param[out] Ncolumns En donde se guardará el número de columnas de la matriz  Matrix.
 *  \return TRUE si todo fue bien o FALSE si no (ej: Matrix==NULL). 
 *  \ingroup PdsMatrixGroup
 */
int pds_matrix_get_ncolumns(const PdsMatrix *Matrix, PdsRaNatural *Ncolumns);


/** \fn int pds_matrix_get_min_value(const PdsMatrix *Matrix, PdsRaReal *m)
 *  \brief Devuelve el valor mínimo de los elementos de la matriz  Matrix.
 *  \param[in] Matrix La matriz de en consulta.
 *  \param[out] m El valor mínimo de los elementos de la matriz  Matrix.
 *  \return TRUE si todo fue bien o FALSE si no (ej: Matrix==NULL). 
 *  \ingroup PdsMatrixGroup
 */
int pds_matrix_get_min_value(const PdsMatrix *Matrix, PdsRaReal *m);


/** \fn int pds_matrix_get_max_value(const PdsMatrix *Matrix, PdsRaReal *m)
 *  \brief Devuelve el valor máximo de los elementos de la matriz  Matrix.
 *  \param[in] Matrix La matriz de en consulta.
 *  \param[out] m El valor máximo de los elementos de la matriz  Matrix.
 *  \return TRUE si todo fue bien o FALSE si no (ej: Matrix==NULL). 
 *  \ingroup PdsMatrixGroup
 */
int pds_matrix_get_max_value(const PdsMatrix *Matrix, PdsRaReal *m);


/** \fn int pds_matrix_swap_rows(PdsMatrix *Matrix, PdsRaNatural r1, PdsRaNatural r2)
 *  \brief Intercambia dos lineas de la matriz  Matrix.
 *  \param[in,out] Matrix La matriz de en consulta.
 *  \param[in] r1 La linea a intercambiar con r2.
 *  \param[in] r2 La linea a intercambiar con r1.
 *  \return TRUE si todo fue bien o FALSE si no (ej: Matrix==NULL). Puede retornar
 *  FALSE por causa de r1>=Nlin, r2>=Nlin o Matrix==NULL. En todos los casos de 
 *  error Matrix no es alterado.
 *  \ingroup PdsMatrixGroup
 */
int pds_matrix_swap_rows(PdsMatrix *Matrix, PdsRaNatural r1, PdsRaNatural r2);


/** \fn int pds_matrix_row_add_row_factor(PdsMatrix *Matrix, PdsRaNatural r1, PdsRaNatural r2, PdsRaReal factor)
 *  \brief Multiplica la linea r2 por un factor y el resulltado lo resta a linea r1.
 *  <br><b>Matrix{r1}=Matrix{r1}+factor*Matrix{r2}</b>.
 *  \param[in,out] Matrix La matriz de en consulta.
 *  \param[in] r1 La linea r1.
 *  \param[in] r2 La linea r2.
 *  \param[in] factor Factor que multiplicará a la linea r2.
 *  \return TRUE si todo fue bien o FALSE si no (ej: Matrix==NULL). Puede retornar
 *  FALSE por causa de r1>=Nlin, r2>=Nlin o Matrix==NULL. En todos los casos de 
 *  error Matrix no es alterado.
 *  \ingroup PdsMatrixGroup
 */
int pds_matrix_row_add_row_factor(PdsMatrix *Matrix, PdsRaNatural r1, PdsRaNatural r2, PdsRaReal factor);


/** \fn int pds_matrix_row_mul_factor(PdsMatrix *Matrix, PdsRaNatural r1, PdsRaReal factor)
 *  \brief Multiplica la linea r1 por un factor y el resulltado lo carga a linea r1.
 *  <br><b>Matrix{r1}=Matrix{r1}*factor</b>.
 *  \param[in,out] Matrix La matriz de en consulta.
 *  \param[in] r1 La linea r1.
 *  \param[in] factor Factor que multiplicará a la linea r1.
 *  \return TRUE si todo fue bien o FALSE si no (ej: Matrix==NULL). Puede retornar
 *  FALSE por causa de r1>=Nlin o Matrix==NULL. En todos los casos de 
 *  error Matrix no es alterado.
 *  \ingroup PdsMatrixGroup
 */
int pds_matrix_row_mul_factor(PdsMatrix *Matrix, PdsRaNatural r1, PdsRaReal factor);


/** \fn int pds_matrix_row_div_factor(PdsMatrix *Matrix, PdsRaNatural r1, PdsRaReal factor)
 *  \brief Divide la linea r1 por un factor y el resulltado lo carga a linea r1.
 *  <br><b>Matrix{r1}=Matrix{r1}/factor</b>.
 *  \param[in,out] Matrix La matriz de en consulta.
 *  \param[in] r1 La linea r1.
 *  \param[in] factor Factor que dividirá a la linea r1.
 *  \return TRUE si todo fue bien o FALSE si no (ej: Matrix==NULL). Puede retornar
 *  FALSE por causa de r1>=Nlin o Matrix==NULL. En todos los casos de 
 *  error Matrix no es alterado.
 *  \ingroup PdsMatrixGroup
 */
int pds_matrix_row_div_factor(PdsMatrix *Matrix, PdsRaNatural r1, PdsRaReal factor);


/** \fn int pds_matrix_reduced_matrix(const PdsMatrix *Matrix, PdsMatrix *MatReduced)
 *  \brief Reduce la matriz Matrix y la guarda en MatReduced. 
 *  <br><b>MatReduced=Reduced{Matrix}</b>.
 *
 *  Una matriz es reducida cuando:
 *  Es una matriz escalonada.
 *  Sus pivotes son todos iguales a 1
 *  En cada fila el pivote es el único elemento no nulo de su columna
 *  \param[in] Matrix La matriz de en consulta.
 *  \param[out] MatReduced Matriz reducida.
 *  \return TRUE si todo fue bien o FALSE si no (ej: Matrix==NULL). Puede retornar
 *  FALSE por causa de MatReduced tenga distinto tamaño que Matrix, 
 *  MatReduced==NULL o Matrix==NULL. 
 *  \ingroup PdsMatrixGroup
 */
int pds_matrix_reduced_matrix(const PdsMatrix *Matrix, PdsMatrix *MatReduced);


/** \fn int pds_matrix_inverse_matrix(const PdsMatrix *Matrix, PdsMatrix *MatInv)
 *  \brief Invierte la matriz Matrix y la guarda en MatInv. Ambas matrices deben
 *  existir y tener el mismo tamanho. <br><b>MatInv=Matrix^{-1}</b>.
 *  \param[in] Matrix La matriz de en consulta.
 *  \param[out] MatInv Matriz inversa.
 *  \return TRUE si todo fue bien o FALSE si no (ej: Matrix==NULL). Puede retornar
 *  FALSE por causa de MatInv tenga distinto tamaño que Matrix, 
 *  MatInv==NULL o Matrix==NULL. 
 *  \ingroup PdsMatrixGroup
 */
int pds_matrix_inverse_matrix(const PdsMatrix *Matrix, PdsMatrix *MatInv);


/** \fn PdsMatrix *pds_matrix_inverse_matrix_new(const PdsMatrix *Matrix)
 *  \brief Invierte la matriz Matrix y la guarda en MatInv. Ambas matrices deben
 *  existir y tener el mismo tamanho. <br><b>MatInv=Matrix^{-1}</b>.
 *  \param[in] Matrix La matriz de en consulta.
 *  \return Un puntero que apunta a una nueva estructura con la multiplicación.
 *  Retorna NULL si fallo en la inversión, o si Matrix==NULL. 
 *  \ingroup PdsMatrixGroup
 */
PdsMatrix *pds_matrix_inverse_matrix_new(const PdsMatrix *Matrix);

/** \fn int pds_matrix_is_equal (PdsMatrix *M1,PdsMatrix *M2)
 *  \brief Compara dos matrices.
 *  \param[in] M1 La matriz a comparar com M2.
 *  \param[in] M2 La matriz a comparar com M1.
 *  \return Retorna TRUE si las matrices son iguales, caso contrario retorna FALSE. 
 *  \ingroup PdsMatrixGroup
 */
int pds_matrix_is_equal (PdsMatrix *M1,PdsMatrix *M2);

/** \fn int pds_matrix_is_quasiequal (PdsMatrix *M1,PdsMatrix *M2,double u)
 *  \brief Compara dos matrices, y verifica si la diferencia absoluta de cada 
 *  elemento es menor que un umbral u.
 *  \param[in] M1 La matriz a comparar com M2.
 *  \param[in] M2 La matriz a comparar com M1.
 *  \param[in] u Es el umbrarl a testar.
 *  \return Retorna TRUE si las matrices son quasi iguales 
 *  (diferencia de elementos menoro igual a u), caso contrario retorna FALSE. 
 *  \ingroup PdsMatrixGroup
 */
int pds_matrix_is_quasiequal (PdsMatrix *M1,PdsMatrix *M2,double u);


/** \fn int pds_matrix_is_diagonal (PdsMatrix *M1)
 *  \brief Verifica si una matriz es diagonal.
 *
 *  Una matriz es diagonal cuando sus elementos exteriores a la diagonal 
 *  principal son nulos. Una matriz nula (llena de ceros) tambien es diagonal.
 *
 *  \param[in] M1 La matriz a verificar su diagonalidad.
 *  \return Retorna TRUE si las matriz es diagonal, caso contrario retorna FALSE.
 *  Tambien retorna FALSE si la matriz no es cuadrada. 
 *  \ingroup PdsMatrixGroup
 */
int pds_matrix_is_diagonal (PdsMatrix *M1);


/** \fn int pds_matrix_is_quasidiagonal (PdsMatrix *M1,double u)
 *  \brief Verifica si una matriz es casi diagonal. Usa un umbral u, qualquier 
 *  valor menor o igual que u es considerado cero.
 *
 *  Una matriz es diagonal cuando sus elementos exteriores a la diagonal 
 *  principal son nulos. Una matriz nula (llena de ceros) tambien es diagonal.
 *
 *  \param[in] M1 La matriz a verificar su diagonalidad.
 *  \param[in] u Es el umbral bajo el cual todo lo demas se considera cero.
 *  \return Retorna TRUE si las matriz es diagonal, caso contrario retorna FALSE.
 *  Tambien retorna FALSE si la matriz no es cuadrada. 
 *  \ingroup PdsMatrixGroup
 */
int pds_matrix_is_quasidiagonal (PdsMatrix *M1,double u);

/** \fn int pds_matrix_is_symetric (PdsMatrix *M1)
 *  \brief Verifica si una matriz es simétrica.
 *
 *  Una matriz es simétrica si coincide con su transpuesta.
 *
 *  \param[in] M1 La matriz a verificar su simetria.
 *  \return Retorna TRUE si las matriz es simetrica, caso contrario retorna FALSE. 
 *  Tambien retorna FALSE si la matriz no es cuadrada.
 *  \ingroup PdsMatrixGroup
 */
int pds_matrix_is_symetric (PdsMatrix *M1);

/** \fn int pds_matrix_is_antisymetric (PdsMatrix *M1)
 *  \brief Verifica si una matriz es anti simétrica.
 *
 *  Una matriz es anti simétrica cuando el opuesto de esta  coincide con su transpuesta.
 *  De esto se deduce que los elementos de la diagonal son nulos.
 *  <br><b>M1^t=-M1</b>.
 *
 *  \param[in] M1 La matriz a verificar su anti simetria.
 *  \return Retorna TRUE si las matriz es anti simetrica, caso contrario retorna FALSE. 
 *  Tambien retorna FALSE si la matriz no es cuadrada.
 *  \ingroup PdsMatrixGroup
 */
int pds_matrix_is_antisymetric (PdsMatrix *M1);

/** \fn void pds_matrix_free(PdsMatrix *Matrix)
 *  \brief Libera una matriz de tipo puntero PdsMatrix.
 *  \param[in,out] Matrix La matriz a liberar.
 *  \return No retorna valor.
 *  \ingroup PdsMatrixGroup
 */
void pds_matrix_free(PdsMatrix *Matrix);


/** \fn void pds_matrix_destroy(PdsMatrix **Matrix)
 *  \brief Libera una matriz de tipo puntero PdsMatrix, y limpia el puntero con NULL.
 *  \param[in,out] Matrix La matriz a liberar y limpiar.
 *  \return No retorna valor.
 *  \ingroup PdsMatrixGroup
 */
void pds_matrix_destroy(PdsMatrix **Matrix);


/**
 * @}
 */

#ifdef __cplusplus
}
#endif 

#endif

