/*
 * pdslvector.c
 * 
 * Copyright 2011 Fernando Pujaico Rivera <fernando.pujaico.rivera@gmail.com>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 * 
 */

#include <pds/pdsvector.h>
#include <pds/pdslvector.h>


////////////////////////////////////////////////////////////////////////////////
////  Trabajando con PdsListVector                                              ////
////////////////////////////////////////////////////////////////////////////////


/** \fn PdsListVector *pds_list_vector_new(void)
 *  \brief Crea una lista de tipo PdsListVector vacia.
 *  \return Un puntero al vector de tipo PdsListVector.
 *  \ingroup PdsListVectorGroup
 */
PdsListVector *pds_list_vector_new(void)
{
	PdsListVector *L=NULL;
	L=(PdsListVector *)calloc(1,sizeof(PdsListVector));
	if(L==NULL)	return NULL;
	
	L->AddPrev=NULL;
	L->V=NULL;
	L->AddNext=NULL;

	return L;
}


/** \fn int pds_list_vector_push(PdsListVector **Lprev,PdsRaNatural N)
 *  \brief Agrega un elemento a la cima de la lista.
 *  \param[in,out] Lprev El nodo de la cima de la lista.
 *  \param[in] N La cantidad de elementos del vector.
 *  \return PDS_OK si todo fue bien o PDS_WRONG si no. Si no le das el último nodo da error
 *  \ingroup PdsListVectorGroup
 */
int pds_list_vector_push(PdsListVector **Lprev,PdsRaNatural N)
{
	PdsListVector *L=NULL;

	if((*Lprev)==NULL)		return PDS_WRONG;
	if((*Lprev)->AddNext!=NULL)	return PDS_WRONG;

	if((*Lprev)->V==NULL)
	{
		(*Lprev)->V=pds_vector_new(N);
		return PDS_OK;
	}
	else
	{
		L=(PdsListVector *)calloc(1,sizeof(PdsListVector));
		if(L==NULL)	return PDS_WRONG;
	
		L->AddPrev=*Lprev;
		L->V=pds_vector_new(N);
		if(L->V==NULL)
		{
			free(L);
			return PDS_WRONG;
		}
		L->AddNext=NULL;

		(*Lprev)->AddNext=L;

		(*Lprev)=L;
	
		return PDS_OK;
	}
}


/** \fn int pds_list_vector_pop(PdsListVector **L)
 *  \brief Quita un elemento de la lista. Si no hay elementos retorna PDS_WRONG.
 *  \param[in,out] L El nodo de la cima de la lista.
 *  \return PDS_OK si todo fue bien o PDS_WRONG si no. Si no le das el último nodo da error
 *  \ingroup PdsListVectorGroup
 */
int pds_list_vector_pop(PdsListVector **L)
{
	PdsListVector *Lprev=NULL;

	if((*L)==NULL)		return PDS_WRONG;
	if((*L)->AddNext!=NULL)	return PDS_WRONG;
	
	if((*L)->AddPrev==NULL)
	{
		if((*L)->V==NULL)
		{
			return PDS_WRONG;
		}
		else
		{
			(*L)->AddNext=NULL;
			pds_vector_free((*L)->V);
			(*L)->V=NULL;
			(*L)->AddPrev=NULL;
			return PDS_OK;
		}
	}
	else
	{
		Lprev=(PdsListVector*)((*L)->AddPrev);
	
		(*L)->AddNext=NULL;
		pds_vector_free((*L)->V);
		(*L)->V=NULL;
		(*L)->AddPrev=NULL;
		free((*L));

		(*L)=Lprev;
		(*L)->AddNext=NULL;
		return PDS_OK;
	}
}


/** \fn int pds_list_vector_shift(PdsListVector **L)
 *  \brief Quita el elemento inicial;mas antiguo; de la lista, si no hay elementos retorna PDS_WRONG.
 *  \param[in,out] L El primer nodo de la lista.
 *  \return PDS_OK si todo fue bien o PDS_WRONG si no. Si no le das el primer nodo da error.
 *  \ingroup PdsListVectorGroup
 */
int pds_list_vector_shift(PdsListVector **L)
{
	PdsListVector *Lnext=NULL;

	if((*L)==NULL)		return PDS_WRONG;
	if((*L)->AddPrev!=NULL)	return PDS_WRONG;
	
	if((*L)->AddNext==NULL)
	{
		if((*L)->V==NULL)
		{
			return PDS_WRONG;
		}
		else
		{
			pds_vector_free((*L)->V);
			(*L)->V=NULL;
			return PDS_OK;
		}
	}
	else
	{
		Lnext=(*L)->AddNext;
	
		pds_vector_free((*L)->V);
		(*L)->V=NULL;
		(*L)->AddNext=NULL;
		free((*L));

		(*L)=Lnext;
		(*L)->AddPrev=NULL;
		return PDS_OK;
	}
}


/** \fn int pds_list_vector_unshift(PdsListVector **Lnext,PdsRaNatural N)
 *  \brief Agrega un elemento al inicio de la lista.
 *  \param[in,out] Lnext El primer nodo de la lista.
 *  \param[in] N La cantidad de elementos del vector.
 *  \return PDS_OK si todo fue bien o PDS_WRONG si no. Si no le das el primer nodo da error.
 *  \ingroup PdsListVectorGroup
 */
int pds_list_vector_unshift(PdsListVector **Lnext,PdsRaNatural N)
{
	PdsListVector *L=NULL;

	if((*Lnext)==NULL)		return PDS_WRONG;
	if((*Lnext)->AddPrev!=NULL)	return PDS_WRONG;

	if((*Lnext)->V==NULL)
	{
		(*Lnext)->V=pds_vector_new(N);
		return PDS_OK;
	}
	else
	{
		L=(PdsListVector *)calloc(1,sizeof(PdsListVector));
		if(L==NULL)	return PDS_WRONG;
	
		L->AddNext=*Lnext;
		L->V=pds_vector_new(N);
		if(L->V==NULL)
		{
			free(L);
			return PDS_WRONG;
		}
		L->AddPrev=NULL;

		(*Lnext)->AddPrev=L;

		(*Lnext)=L;
	
		return PDS_OK;
	}
}


/** \fn int pds_list_vector_top(PdsListVector **L)
 *  \brief Busca el elemento final; superior; de la lista.
 *  \param[in,out] L Un nodo de la lista, en donde se cargará el último nodo.
 *  \return PDS_OK si todo fue bien o PDS_WRONG si no. 
 *  \ingroup PdsListVectorGroup
 */
int pds_list_vector_top(PdsListVector **L)
{
	if((*L)==NULL)	return PDS_WRONG;

	while((*L)->AddNext!=NULL)
	{
		(*L)=(*L)->AddNext;
	}
	return PDS_OK;
}


/** \fn int pds_list_vector_bottom(PdsListVector **L)
 *  \brief Busca el elemento inicial; inferior; de la lista.
 *  \param[in,out] L Un nodo de la lista, en donde se cargará el primer nodo.
 *  \return PDS_OK si todo fue bien o PDS_WRONG si no. 
 *  \ingroup PdsListVectorGroup
 */
int pds_list_vector_bottom(PdsListVector **L)
{
	if((*L)==NULL)	return PDS_WRONG;

	while((*L)->AddPrev!=NULL)
	{
		(*L)=(*L)->AddPrev;
	}
	return PDS_OK;
}


/** \fn int pds_list_vector_printf(const PdsListVector *L)
 *  \brief Imprime en pantalla los datos de un nodo de tipo puntero PdsListVector.
 *  \param[in] L El nodo a imprimir en pantalla.
 *  \return PDS_OK si todo fue bien o PDS_WRONG si no.
 *  \ingroup PdsListVectorGroup
 */
int pds_list_vector_printf(const PdsListVector *L)
{
	if(L==NULL)		return PDS_WRONG;

	printf("%p\t",L);
	printf("AddPrev : %p\n",L->AddPrev);
	if(L->V!=NULL)	
	{
		printf("\t\tV       : ");
		pds_vector_printf(L->V);
	}
	else		printf("\t\tV       :\n");
	printf("\t\tAddNext : %p\n",L->AddNext);
	
	return PDS_OK;
}


/** \fn int pds_list_vector_all_printf(const PdsListVector *L)
 *  \brief Imprime en pantalla todos los datos de la lista.
 *  \param[in] L Un nodo de la lista a imprimir en pantalla.
 *  \return PDS_OK si todo fue bien o PDS_WRONG si no.
 *  \ingroup PdsListVectorGroup
 */
int pds_list_vector_all_printf(const PdsListVector *L)
{
	PdsListVector *Lbottom=NULL;
	//PdsRaNatural i;
	int id;

	if(L==NULL)		return PDS_WRONG;

	Lbottom=(PdsListVector *)L;

	id=pds_list_vector_bottom(&Lbottom);
	if(id==PDS_WRONG)	return PDS_WRONG;

	
	while(1)
	{
		id=pds_list_vector_printf(Lbottom);
		if(id==PDS_WRONG)	return PDS_WRONG;

		if(Lbottom->AddNext!=NULL)	Lbottom=Lbottom->AddNext;
		else				break;
	}
		
	return PDS_OK;
}


/** \fn void pds_list_vector_free(PdsListVector *L)
 *  \brief Libera una lista entera de tipo puntero PdsListVector. 
 *  \param[in,out] L La lista a liberar.
 *  \return No retorna valor.
 *  \ingroup PdsListVectorGroup
 */
void pds_list_vector_free(PdsListVector *L)
{
	PdsListVector *Lbottom=NULL;
	int id;

	if(L!=NULL)
	{
		Lbottom=L;

		id=pds_list_vector_bottom(&Lbottom);
		if(id==PDS_OK)
		{
			while(1)
			{
				pds_list_vector_shift(&Lbottom);
				if(Lbottom->AddNext==NULL)	break;
			}

			free(Lbottom);
		}
	}
}


/** \fn void pds_list_vector_destroy(PdsListVector **L)
 *  \brief Libera una lista de tipo puntero PdsListVector, y limpia el puntero con NULL.
 *  \param[in,out] L La lista a liberar y limpiar.
 *  \return No retorna valor.
 *  \ingroup PdsListVectorGroup
 */
void pds_list_vector_destroy(PdsListVector **L)
{
	PdsListVector *Lbottom=NULL;
	int id;

	if(L!=NULL)
	{
		Lbottom=*L;

		id=pds_list_vector_bottom(&Lbottom);
		if(id==PDS_OK)
		{
			while(1)
			{
				pds_list_vector_shift(&Lbottom);
				if(Lbottom->AddNext==NULL)	break;
			}

			free(Lbottom);
			*L=NULL;
		}
	}
}

