/*
 * pdsneuron.c
 * 
 * Copyright 2011 Fernando Pujaico Rivera <fernando.pujaico.rivera@gmail.com>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 * 
 */

#include <math.h>
#include <pds/pdsninput.h>
#include <pds/pdsneuron.h>


////////////////////////////////////////////////////////////////////////////////
////  Trabajando con PdsNeuron                                              ////
////////////////////////////////////////////////////////////////////////////////


/** \fn PdsNeuron *pds_neuron_new(PdsSnNatural Nel)
 *  \brief Crea un vector de tipo PdsNeuron e inicia con cero todos los elementos
 *  excepto Sigma \$ = \sigma \$ que toma valor uno.
 *  \param[in] Nel Es el número de elementos del vector.
 *  \return Un puntero al vector de tipo PdsNeuron.
 *  \ingroup PdsNeuronGroup
 */
PdsNeuron *pds_neuron_new(PdsSnNatural Nel)
{
	PdsNeuron *Neuron=NULL;
	PdsSnNatural i;

	if(Nel<=0)		return NULL;
	
	Neuron=(PdsNeuron *)calloc(1,sizeof(PdsNeuron));
	if(Neuron==NULL)	return NULL;

	Neuron->Nel=Nel;
	Neuron->X=(PdsSnReal **)calloc(Neuron->Nel,sizeof(PdsSnReal*));
	if(Neuron->X==NULL) 
	{
		free(Neuron);
		return NULL;
	}
	for(i=0;i<Neuron->Nel;i++)	Neuron->X[i]=NULL;

	Neuron->W=(PdsSnReal *)calloc(Neuron->Nel,sizeof(PdsSnReal));
	if(Neuron->W==NULL) 
	{
		free(Neuron);
		return NULL;
	}

	for(i=0;i<Neuron->Nel;i++)	Neuron->W[i]=0.0;
	
	Neuron->U=0.0;
	Neuron->Sigma=1.0;
	Neuron->Theta=0.0;
	Neuron->F=0.0;
	Neuron->Y[0]=0.0;
	Neuron->Y[1]=0.0;

	return Neuron;
}


/** \fn int pds_neuron_get_synaptic_weight(const PdsNeuron *Neuron, PdsSnNatural x, PdsSnReal *m)
 *  \brief Devuelve el valor en la posición (x) del vector de pesos de la Neurona.
 *  (x) inicia con (0).
 *  \param[in] Neuron La neurona en consulta.
 *  \param[in] x Posición x, el primer valor de x es cero.
 *  \param[out] m El valor en la posición (x), en caso de error por fuera de rango 
 *  (x) entonces carga 0 en m, en caso de error de vector nulo carga cero en m.
 *  \return TRUE si todo fue bien o FALSE si no (ej: Neuron==NULL). 
 *  \ingroup PdsNeuronGroup
 */
int pds_neuron_get_synaptic_weight(const PdsNeuron *Neuron, PdsSnNatural x, PdsSnReal *m)
{
	*m=0;

	if(Neuron==NULL)	return FALSE;

	if((x>=0)&&(x<Neuron->Nel))	*m=Neuron->W[x];
	else				*m=0;

	return TRUE;
}


/** \fn int pds_neuron_set_synaptic_weight(PdsNeuron *Neuron, PdsSnNatural x, PdsSnReal m)
 *  \brief Escribe el valor m en la posición (x) del vector de pesos de la Neurona.
 *  (x) inicia con (0).
 *  \param[in,out] Neuron La neurona a escribir.
 *  \param[in] x Posición x, el primer valor de x es cero.
 *  \param[in] m El valor en la posición (x), en caso de error por fuera de rango 
 *  (x) entonces no hace nada y no se considera como error.
 *  \return TRUE si todo fue bien o FALSE si no (ej: Neuron==NULL). 
 *  \ingroup PdsNeuronGroup
 */
int pds_neuron_set_synaptic_weight(PdsNeuron *Neuron, PdsSnNatural x, PdsSnReal m)
{
	if(Neuron==NULL)	return FALSE;

	if((x>=0)&&(x<Neuron->Nel))	Neuron->W[x]=m;

	return TRUE;
}


/** \fn int pds_neuron_init_synaptic_weight(PdsNeuron *Neuron, PdsSnReal m)
 *  \brief Inicia el vector de pesos de la Neurona.
 *  \param[in,out] Neuron La neurona a escribir.
 *  \param[in] m El valor de los pesos de la neurona.
 *  \return TRUE si todo fue bien o FALSE si no (ej: Neuron==NULL). 
 *  \ingroup PdsNeuronGroup
 */
int pds_neuron_init_synaptic_weight(PdsNeuron *Neuron, PdsSnReal m)
{
	PdsSnNatural i;

	if(Neuron==NULL)	return FALSE;

	for(i=0;i<Neuron->Nel;i++)	Neuron->W[i]=m;

	return TRUE;
}


/** \fn int pds_neuron_get_number_of_inputs(const PdsNeuron *Neuron,PdsSnNatural *Nelements)
 *  \brief Devuelve el número de entradas de la Neurona.
 *  \param[in] Neuron La neurona en consulta.
 *  \param[out] Nelements En donde se guardará el número de entradas de la neurona.
 *  \return TRUE si todo fue bien o FALSE si no (ej: Neuron==NULL). 
 *  \ingroup PdsNeuronGroup
 */
int pds_neuron_get_number_of_inputs(const PdsNeuron *Neuron,PdsSnNatural *Nelements)
{
	*Nelements=0;

	if(Neuron==NULL)	return FALSE;

	*Nelements=Neuron->Nel;


	return TRUE;
}


/** \fn int pds_neuron_get_sigma(const PdsNeuron *Neuron,PdsSnReal *Sigma)
 *  \brief Devuelve el valor de  sigma de la Neurona.
 *  \param[in] Neuron La neurona en consulta.
 *  \param[out] Sigma En donde se guardará el valor de sigma de la neurona.
 *  \return TRUE si todo fue bien o FALSE si no (ej: Neuron==NULL). 
 *  \ingroup PdsNeuronGroup
 */
int pds_neuron_get_sigma(const PdsNeuron *Neuron,PdsSnReal *Sigma)
{
	*Sigma=0;

	if(Neuron==NULL)	return FALSE;

	*Sigma=Neuron->Sigma;

	return TRUE;
}


/** \fn int pds_neuron_set_sigma(PdsNeuron *Neuron,PdsSnReal Sigma)
 *  \brief Coloca el valor de  sigma de la Neurona.
 *  \param[in,out] Neuron La neurona a escribir.
 *  \param[in] Sigma El valor de sigma de la neurona.
 *  \return TRUE si todo fue bien o FALSE si no (ej: Neuron==NULL). 
 *  \ingroup PdsNeuronGroup
 */
int pds_neuron_set_sigma(PdsNeuron *Neuron,PdsSnReal Sigma)
{

	if(Neuron==NULL)	return FALSE;

	Neuron->Sigma=Sigma;

	return TRUE;
}


/** \fn int pds_neuron_get_u(const PdsNeuron *Neuron,PdsSnReal *U)
 *  \brief Devuelve el valor de  U de la Neurona.
 *  \param[in] Neuron La neurona en consulta.
 *  \param[out] U En donde se guardará el valor de U de la neurona.
 *  \return TRUE si todo fue bien o FALSE si no (ej: Neuron==NULL). 
 *  \ingroup PdsNeuronGroup
 */
int pds_neuron_get_u(const PdsNeuron *Neuron,PdsSnReal *U)
{
	*U=0;

	if(Neuron==NULL)	return FALSE;

	*U=Neuron->U;

	return TRUE;
}


/** \fn int pds_neuron_set_u(PdsNeuron *Neuron,PdsSnReal U)
 *  \brief Coloca el valor de  U de la Neurona.
 *  \param[in,out] Neuron La neurona a escribir.
 *  \param[in] U El valor de U de la neurona.
 *  \return TRUE si todo fue bien o FALSE si no (ej: Neuron==NULL). 
 *  \ingroup PdsNeuronGroup
 */
int pds_neuron_set_u(PdsNeuron *Neuron,PdsSnReal U)
{

	if(Neuron==NULL)	return FALSE;

	Neuron->U=U;

	return TRUE;
}


/** \fn PdsSnReal *pds_neuron_get_dendrite(const PdsNeuron *Neuron,PdsSnNatural id)
 *  \brief Devuelve la dirección de la entrada X[id] de la Neurona.
 *  \param[in] Neuron La neurona en consulta.
 *  \param[in] id La posición a consultar, id tiene que ser menor que Nel.
 *  \return La dirección de la entrada X[id] o NULL en caso de error (ej: Neuron==NULL). 
 *  \ingroup PdsNeuronGroup
 */
PdsSnReal *pds_neuron_get_dendrite(const PdsNeuron *Neuron,PdsSnNatural id)
{
	if(Neuron==NULL)		return NULL;
	if((id>=Neuron->Nel)||(id<0))	return NULL;

	return Neuron->X[id];
}


/** \fn int pds_neuron_get_input(const PdsNeuron *Neuron,PdsSnNatural id,PdsSnReal *m)
 *  \brief Devuelve el valor de la entrada X[id] de la Neurona.
 *  \param[in] Neuron La neurona en consulta.
 *  \param[in] id La posición a consultar, id tiene que ser menor que Nel.
 *  \param[out] m Donde se cargará el valor de la entrada X[id].
 *  \return TRUE si todo fue bien o FALSE si no (ej: Neuron==NULL, Neuron->X[id]==NULL). 
 *  \ingroup PdsNeuronGroup
 */
int pds_neuron_get_input(const PdsNeuron *Neuron,PdsSnNatural id,PdsSnReal *m)
{
	*m=0;
	if(Neuron==NULL)		return FALSE;
	if((id>=Neuron->Nel)||(id<0))	return FALSE;
	if(Neuron->X[id]==NULL)		return FALSE;

	*m=*(Neuron->X[id]);

	return TRUE;
}


/** \fn const PdsSnReal *pds_neuron_get_axon(const PdsNeuron *Neuron)
 *  \brief Devuelve la dirección del axon (salida) de la Neurona.
 *  \param[in] Neuron La neurona en consulta.
 *  \return La dirección del axon (salida) o NULL en caso de error (ej: Neuron==NULL). 
 *  \ingroup PdsNeuronGroup
 */
const PdsSnReal *pds_neuron_get_axon(const PdsNeuron *Neuron)
{
	if(Neuron==NULL)		return NULL;

	return Neuron->Y;
}


/** \fn int pds_neuron_get_output(const PdsNeuron *Neuron,PdsSnReal *m)
 *  \brief Devuelve el valor de la salida Y de la Neurona.
 *  \param[in] Neuron La neurona en consulta.
 *  \param[out] m Donde se cargará el valor de la salida Y.
 *  \return TRUE si todo fue bien o FALSE si no (ej: Neuron==NULL). 
 *  \ingroup PdsNeuronGroup
 */
int pds_neuron_get_output(const PdsNeuron *Neuron,PdsSnReal *m)
{
	if(Neuron==NULL)		return FALSE;

	*m=*(Neuron->Y);

	return TRUE;
}


/** \fn int pds_neuron_connect_input(PdsNeuron *Neuron,PdsSnNatural id,PdsNeuron *NeuronX)
 *  \brief Conecta el valor de salida Y de la Neurona NeuronX con la entrada X[id]
 *  de la neurona Neuron.
 *  \param[in,out] Neuron La neurona a trabajar.
 *  \param[in] id La posición a conectar, id tiene que ser menor que Nel.
 *  \param[in] NeuronX Neurona que se conectará a la entrada X[id] de Neuron.
 *  \return TRUE si todo fue bien o FALSE si no (ej: Neuron==NULL). 
 *  \ingroup PdsNeuronGroup
 */
int pds_neuron_connect_input(PdsNeuron *Neuron,PdsSnNatural id,PdsNeuron *NeuronX)
{
	if(Neuron ==NULL)		return FALSE;
	if(NeuronX==NULL)		return FALSE;
	if((id>=Neuron->Nel)||(id<0))	return FALSE;

	Neuron->X[id]=NeuronX->Y;

	return TRUE;
}


/** \fn int pds_neuron_connect_input_data(PdsNeuron *Neuron,PdsSnNatural id,PdsNInput *NInput)
 *  \brief Conecta el valor de salida Y de la neurona de entrada NInput con la 
 *  entrada X[id] de la neurona Neuron.
 *  \param[in,out] Neuron La neurona a trabajar.
 *  \param[in] id La posición a conectar, id tiene que ser menor que Nel.
 *  \param[in] NInput La neurona de entrada que se conectará a la entrada X[id] de Neuron.
 *  \return TRUE si todo fue bien o FALSE si no (ej: Neuron==NULL). 
 *  \ingroup PdsNeuronGroup
 */
int pds_neuron_connect_input_data(PdsNeuron *Neuron,PdsSnNatural id,PdsNInput *NInput)
{
	if(Neuron==NULL)		return FALSE;
	if(NInput==NULL)		return FALSE;
	if((id>=Neuron->Nel)||(id<0))	return FALSE;

	Neuron->X[id]=NInput->Y;

	return TRUE;
}


/** \fn int pds_neuron_evaluate_theta(PdsNeuron *Neuron)
 *  \brief Evalua la variable theta de la Neurona Neuron.
 *  \f[ \theta=\sum_{i=0}^{N_{el}-1}{W[i]X[i]} - U \f].
 *  \param[in,out] Neuron La neurona a evaluar.
 *  \return TRUE si todo fue bien o FALSE si no (ej: Neuron==NULL). 
 *  \ingroup PdsNeuronGroup
 */
int pds_neuron_evaluate_theta(PdsNeuron *Neuron)
{
	PdsSnReal theta;
	PdsSnNatural i;

	if(Neuron==NULL)		return FALSE;

	theta=-Neuron->U;
	for(i=0;i<Neuron->Nel;i++)	
	{
		if(Neuron->X[i]!=NULL)
		theta=theta+(*(Neuron->X[i]))*Neuron->W[i];
	}
	
	Neuron->Theta=theta;

	return TRUE;
}


/** \fn PdsSnReal pds_neuron_func(PdsSnReal x)
 *  \brief Función de activación de la neurona de tipo PdsNeuron.
 *  \f[ f(x)=\frac{2}{1+e^{-x}} -1 \f].
 *  \param[in] x Parámetro a evaluar.
 *  \return El resultado de la función. 
 *  \ingroup PdsNeuronGroup
 */
PdsSnReal pds_neuron_func(PdsSnReal x)
{
	return (2.0/(1.0+exp(-x)))-1.0;
}


/** \fn PdsSnReal pds_neuron_dfunc(PdsSnReal x)
 *  \brief Derivada de la función de activación de la neurona de tipo PdsNeuron.
 *  \f[ \frac{\partial  f(x)}{\partial x}=\frac{1-f(x)^2}{2} \f].
 *  \param[in] x Parámetro a evaluar.
 *  \return El resultado de la derivada de la función. 
 *  \ingroup PdsNeuronGroup
 */
PdsSnReal pds_neuron_dfunc(PdsSnReal x)
{
	PdsSnReal m;

	m=(2.0/(1.0+exp(-x)))-1.0;

	return 0.5-0.5*m*m;
}


/** \fn PdsSnReal pds_neuron_invdfunc(PdsSnReal x)
 *  \brief Uno sobre la derivada de la función de activación de la neurona de 
 *  tipo PdsNeuron.
 *  \f[ ( \frac{\partial  f(x)}{\partial x} )^{-1}=\frac{2}{1-f(x)^2} \f].
 *  \param[in] x Parámetro a evaluar.
 *  \return El resultado de uno sobre la derivada de la función. 
 *  \ingroup PdsNeuronGroup
 */
PdsSnReal pds_neuron_invdfunc(PdsSnReal x)
{
	PdsSnReal m;

	m=(2.0/(1.0+exp(-x)))-1.0;

	return 0.5-0.5*m*m;
}

/** \fn int pds_neuron_evaluate_f(PdsNeuron *Neuron)
 *  \brief Evalua la funcion de activación de la Neurona Neuron.
 *  \f[ f(\theta)=\frac{2}{1+e^{-\frac{\theta}{\sigma}}} -1 \f].
 *  \param[in,out] Neuron La neurona a evaluar.
 *  \return TRUE si todo fue bien o FALSE si no (ej: Neuron==NULL). 
 *  \ingroup PdsNeuronGroup
 */
int pds_neuron_evaluate_f(PdsNeuron *Neuron)
{
	PdsSnReal x;

	if(Neuron==NULL)		return FALSE;

	x=Neuron->Theta/Neuron->Sigma;

	Neuron->F=pds_neuron_func(x);

	return TRUE;
}


/** \fn int pds_neuron_evaluate(PdsNeuron *Neuron)
 *  \brief Evalua la Neurona Neuron.
 *  \f[ \theta=\sum_{i=0}^{N_{el}-1}{W[i]X[i]} - U \f].
 *  \f[ f(\theta)=\frac{2}{1+e^{-\frac{\theta}{\sigma}}} -1 \f].
 *  \param[in,out] Neuron La neurona a evaluar.
 *  \return TRUE si todo fue bien o FALSE si no (ej: Neuron==NULL). 
 *  \ingroup PdsNeuronGroup
 */
int pds_neuron_evaluate(PdsNeuron *Neuron)
{
	int m;
	if(Neuron==NULL)		return FALSE;

	m=pds_neuron_evaluate_theta(Neuron);
	if(m==FALSE)	return FALSE;
	m=pds_neuron_evaluate_f(Neuron);
	if(m==FALSE)	return FALSE;

	return TRUE;
}


/** \fn int pds_neuron_update(PdsNeuron *Neuron)
 *  \brief Actualiza el valor a la salida de la Neurona Neuron.
 *  \f[ Y=f(\theta) \f].
 *  \param[in,out] Neuron La neurona a trabajar.
 *  \return TRUE si todo fue bien o FALSE si no (ej: Neuron==NULL). 
 *  \ingroup PdsNeuronGroup
 */
int pds_neuron_update(PdsNeuron *Neuron)
{
	if(Neuron==NULL)		return FALSE;

	*(Neuron->Y)=Neuron->F;
	Neuron->Y[1]=0.0;

	return TRUE;
}


/** \fn int pds_neuron_iterate(PdsNeuron *Neuron)
 *  \brief Itera la Neurona Neuron.
 *  \f[ \theta=\sum_{i=0}^{N_{el}-1}{W[i]X[i]} - U \f].
 *  \f[ f(\theta)=\frac{2}{1+e^{-\frac{\theta}{\sigma}}} -1 \f].
 *  \f[ Y=f(\theta) \f].
 *  \param[in,out] Neuron La neurona a iterar.
 *  \return TRUE si todo fue bien o FALSE si no (ej: Neuron==NULL). 
 *  \ingroup PdsNeuronGroup
 */
int pds_neuron_iterate(PdsNeuron *Neuron)
{
	int m;

	if(Neuron==NULL)		return FALSE;

	m=pds_neuron_evaluate_theta(Neuron);
	if(m==FALSE)	return FALSE;
	m=pds_neuron_evaluate_f(Neuron);
	if(m==FALSE)	return FALSE;

	*(Neuron->Y)=Neuron->F;
	Neuron->Y[1]=0.0;

	return TRUE;
}


/** \fn int pds_neuron_evaluate_e(PdsNeuron *Neuron,PdsSnReal y)
 *  \brief Compara el valor de salida de la neurona con "y" y luego lo carga en
 *  la variable Y[1] de la neurona. 
 *  Neuron->Y[1]=y-Neuron->Y[0]
 *  \param[in,out] Neuron La neurona a evaluar.
 *  \param[in] y Valor a comparar.
 *  \return TRUE si todo fue bien o FALSE si no (ej: Neuron==NULL). 
 *  \ingroup PdsNeuronGroup
 */
int pds_neuron_evaluate_e(PdsNeuron *Neuron,PdsSnReal y)
{

	if(Neuron==NULL)		return FALSE;

	Neuron->Y[1]=y-Neuron->Y[0];

	return TRUE;
}


/** \fn int pds_neuron_backpropagate_error(PdsNeuron *Neuron)
 *  \brief Propaga el error de la salida de la neurona hacia las neuronas conectadas
 * a sus entradas. E_input (i)=W[i] E_salida
 *  \param[in,out] Neuron La neurona a retropropagar el error.
 *  \return TRUE si todo fue bien o FALSE si no (ej: Neuron==NULL). 
 *  \ingroup PdsNeuronGroup
 */
int pds_neuron_backpropagate_error(PdsNeuron *Neuron)
{
	PdsSnReal e;
	PdsSnNatural i;

	if(Neuron==NULL)		return FALSE;

	e=Neuron->Y[1];
	for(i=0;i<Neuron->Nel;i++)	
	{
		if(Neuron->X[i]!=NULL)
		(*(Neuron->X[i]+1))=(*(Neuron->X[i]+1))+Neuron->W[i]*e;
	}

	return TRUE;
}


/** \fn int pds_neuron_update_weight(PdsNeuron *Neuron, PdsSnReal Alpha)
 *  \brief Actualiza los pesos W[i] de la neurona.
 *  \f[ \triangle W[i]= \alpha e f'(\theta) X[i] \f].
 *  "e" es el error de la salida da neurona Neuron :Neuron->Y[1].
 *  "f" es la función de activación da neurona Neuron.
 *  \param[in,out] Neuron La neurona a actualizar los pesos.
 *  \param[in] Alpha Factor de aprendizaje de los pesos.
 *  \return TRUE si todo fue bien o FALSE si no (ej: Neuron==NULL). 
 *  \ingroup PdsNeuronGroup
 */
int pds_neuron_update_weight(PdsNeuron *Neuron, PdsSnReal Alpha)
{
	PdsSnReal e,x,Df;
	PdsSnNatural i;

	if(Neuron==NULL)		return FALSE;

	Df=pds_neuron_dfunc(Neuron->Theta);
	e=Neuron->Y[1];

	for(i=0;i<Neuron->Nel;i++)	
	{
		if(Neuron->X[i]!=NULL)
		{
			x=(*(Neuron->X[i]));
			Neuron->W[i]=Neuron->W[i]+Alpha*e*Df*x;
		}
	}

	return TRUE;
}


/** \fn int pds_neuron_get_xtx(PdsNeuron *Neuron,PdsSnReal *XtX)
 *  \brief Devuelve la suma cuadrática de los valores de entrada de la neurona.
 *  \f[ X^T X=\sum^{Nel-1}_{i=0} X[i]^2 \f].
 *  \param[in,out] Neuron La neurona en consulta.
 *  \param[in] XtX Donde se cargará la suma cuadrática.
 *  \return TRUE si todo fue bien o FALSE si no (ej: Neuron==NULL). 
 *  \ingroup PdsNeuronGroup
 */
int pds_neuron_get_xtx(PdsNeuron *Neuron,PdsSnReal *XtX)
{
	PdsSnNatural i;

	*XtX=0;
	if(Neuron==NULL)		return FALSE;
	
	for(i=0;i<Neuron->Nel;i++)	
	if(Neuron->X[i]!=NULL)	*XtX=*XtX+(*(Neuron->X[i]))*(*(Neuron->X[i]));

	return TRUE;
}

/** \fn int pds_neuron_update_weight_norm(PdsNeuron *Neuron, PdsSnReal Alpha,PdsSnReal XtX)
 *  \brief Actualiza los pesos W[i] de la neurona.
 *  \f[ \triangle W[i]= \alpha e f'(\theta) \frac{X[i]}{X^TX} \f].
 *  "e" es el error de la salida da neurona Neuron :Neuron->Y[1].
 *  "f" es la función de activación da neurona Neuron.
 *  \param[in,out] Neuron La neurona a actualizar los pesos.
 *  \param[in] Alpha Factor de aprendizaje de los pesos.
 *  \param[in] XtX Suma cuadrática de los valores de entrada X[i].
 *  \return TRUE si todo fue bien o FALSE si no (ej: Neuron==NULL). 
 *  \ingroup PdsNeuronGroup
 */
int pds_neuron_update_weight_norm(PdsNeuron *Neuron, PdsSnReal Alpha,PdsSnReal XtX)
{
	PdsSnReal e,x,Df;
	PdsSnNatural i;

	if(Neuron==NULL)		return FALSE;

	Df=pds_neuron_dfunc(Neuron->Theta);
	e=Neuron->Y[1];

	for(i=0;i<Neuron->Nel;i++)	
	{
		if((Neuron->X[i]!=NULL)&&(XtX>0))
		{
			x=(*(Neuron->X[i]));
			Neuron->W[i]=Neuron->W[i]+Alpha*e*Df*x/XtX;
		}
	}

	return TRUE;
}


/** \fn int pds_neuron_update_weight_normalized(PdsNeuron *Neuron, PdsSnReal Alpha)
 *  \brief Actualiza los pesos W[i] de la neurona.
 *  \f[ \triangle W[i]= \alpha e f'(\theta) \frac{X[i]}{X^TX} \f].
 *  "e" es el error de la salida da neurona Neuron :Neuron->Y[1].
 *  "f" es la función de activación da neurona Neuron.
 *  \param[in,out] Neuron La neurona a actualizar los pesos.
 *  \param[in] Alpha Factor de aprendizaje de los pesos.
 *  \return TRUE si todo fue bien o FALSE si no (ej: Neuron==NULL). 
 *  \ingroup PdsNeuronGroup
 */
int pds_neuron_update_weight_normalized(PdsNeuron *Neuron, PdsSnReal Alpha)
{
	PdsSnReal e,x,Df,XtX;
	PdsSnNatural i;

	if(Neuron==NULL)		return FALSE;

	Df=pds_neuron_dfunc(Neuron->Theta);
	e=Neuron->Y[1];
	
	for(XtX=0,i=0;i<Neuron->Nel;i++)	
	if(Neuron->X[i]!=NULL)	XtX=XtX+(*(Neuron->X[i]))*(*(Neuron->X[i]));

	for(i=0;i<Neuron->Nel;i++)	
	{
		if((Neuron->X[i]!=NULL)&&(XtX>0))
		{
			x=(*(Neuron->X[i]));
			Neuron->W[i]=Neuron->W[i]+Alpha*e*Df*x/XtX;
		}
	}

	return TRUE;
}


/** \fn int pds_neuron_fprintf(const PdsNeuron *Neuron, FILE *fd)
 *  \brief Guarda en un archivo de texto los pesos W[i] y el valor de U.
 *  Ocupando una linea cada uno, y separando los elementos por un TAB.
 *  \param[in] Neuron La neurona a leer.
 *  \param[in,out] fd Manejador del fichero a escribir.
 *  \return TRUE si todo fue bien o FALSE si no (ej: NVector==NULL o fd==NULL). 
 *  \ingroup PdsNeuronGroup
 */
int pds_neuron_fprintf(const PdsNeuron *Neuron, FILE *fd)
{
	PdsSnNatural i;
	int tmp;

	if(Neuron==NULL)	return FALSE;
	if(fd==NULL)		return FALSE;


	for(i=0;i<Neuron->Nel;i++)	
	{
		if(i!=(Neuron->Nel-1))	tmp=fprintf(fd,"%e\t",Neuron->W[i]);
		else			tmp=fprintf(fd,"%e\n",Neuron->W[i]);
	}

	tmp=fprintf(fd,"%e\t",Neuron->U);
	tmp=fprintf(fd,"%e\n",Neuron->Sigma);

	return TRUE;
}


/** \fn int pds_neuron_fscanf(PdsNeuron *Neuron, FILE *fd)
 *  \brief Lee de un archivo de texto los pesos W[i] y el valor de U.
 *  Ocupando una linea cada uno, y separando los elementos por un TAB.
 *  \param[out] Neuron La neurona a escribir.
 *  \param[in,out] fd Manejador del fichero a leer.
 *  \return TRUE si todo fue bien o FALSE si no (ej: NVector==NULL o fd==NULL). 
 *  \ingroup PdsNeuronGroup
 */
int pds_neuron_fscanf(PdsNeuron *Neuron, FILE *fd)
{
	PdsSnNatural i;
	int tmp;

	if(Neuron==NULL)	return FALSE;
	if(fd==NULL)		return FALSE;


	for(i=0;i<Neuron->Nel;i++)	
	{
		tmp=fscanf(fd,"%e",&(Neuron->W[i]));
		if(tmp==0)	return FALSE;
	}

	tmp=fscanf(fd,"%e",&(Neuron->U));
	if(tmp==0)	return FALSE;

	tmp=fscanf(fd,"%e",&(Neuron->Sigma));
	if(tmp==0)	return FALSE;

	return TRUE;
}


/** \fn int pds_neuron_fwrite(const PdsNeuron *Neuron, FILE *fd)
 *  \brief Guarda en un archivo binario los pesos W[i] y el valor de U.
 *  \param[in] Neuron La neurona a leer.
 *  \param[in,out] fd Manejador del fichero binario a escribir.
 *  \return TRUE si todo fue bien o FALSE si no (ej: NVector==NULL o fd==NULL). 
 *  \ingroup PdsNeuronGroup
 */
int pds_neuron_fwrite(const PdsNeuron *Neuron, FILE *fd)
{
	PdsSnNatural i;
	size_t tmp;

	if(Neuron==NULL)	return FALSE;
	if(fd==NULL)		return FALSE;


	for(i=0;i<Neuron->Nel;i++)	
	{
		tmp=fwrite(&(Neuron->W[i]),sizeof(Neuron->W[i]),1,fd);
		if(tmp!=1)	return	FALSE;
	}

	tmp=fwrite(&(Neuron->U),sizeof(Neuron->U),1,fd);
	if(tmp!=1)	return	FALSE;

	tmp=fwrite(&(Neuron->Sigma),sizeof(Neuron->Sigma),1,fd);
	if(tmp!=1)	return	FALSE;

	return TRUE;
}


/** \fn int pds_neuron_fread(PdsNeuron *Neuron, FILE *fd)
 *  \brief Lee de un archivo binario los pesos W[i] y el valor de U.
 *  \param[out] Neuron La neurona a escribir.
 *  \param[in,out] fd Manejador del fichero binario a leer.
 *  \return TRUE si todo fue bien o FALSE si no (ej: NVector==NULL o fd==NULL). 
 *  \ingroup PdsNeuronGroup
 */
int pds_neuron_fread(PdsNeuron *Neuron, FILE *fd)
{
	PdsSnNatural i;
	int tmp;

	if(Neuron==NULL)	return FALSE;
	if(fd==NULL)		return FALSE;


	for(i=0;i<Neuron->Nel;i++)	
	{
		tmp=fread(&(Neuron->W[i]),sizeof(Neuron->W[i]),1,fd);
		if(tmp!=1)	return FALSE;
	}

	tmp=fread(&(Neuron->U),sizeof(Neuron->U),1,fd);
	if(tmp!=1)	return FALSE;

	tmp=fread(&(Neuron->Sigma),sizeof(Neuron->Sigma),1,fd);
	if(tmp!=1)	return FALSE;

	return TRUE;
}


/** \fn void pds_neuron_free(PdsNeuron *Neuron)
 *  \brief Libera una neurona de tipo puntero PdsNeuron.
 *  \param[in,out] Neuron La neurona a liberar.
 *  \return No retorna valor.
 *  \ingroup PdsNeuronGroup
 */
void pds_neuron_free(PdsNeuron *Neuron)
{
	if(Neuron!=NULL)
	{
		free(Neuron->X);
		free(Neuron->W);
		free(Neuron);
	}
}


/** \fn void pds_neuron_destroy(PdsNeuron **Neuron)
 *  \brief Libera una neurona de tipo puntero PdsNeuron, y limpia el puntero con NULL.
 *  \param[in,out] Neuron La neurona a liberar y limpiar.
 *  \return No retorna valor.
 *  \ingroup PdsNeuronGroup
 */
void pds_neuron_destroy(PdsNeuron **Neuron)
{
	if((*Neuron)!=NULL)
	{
		free((*Neuron)->X);
		free((*Neuron)->W);
		free(*Neuron);
		(*Neuron)=NULL;
	}
}

