/*
 * pdsninput.h
 * 
 * Copyright 2011 Fernando Pujaico Rivera <fernando.pujaico.rivera@gmail.com>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 * 
 */

/** \file pdsninput.h
 *  \author Fernando Pujaico Rivera
 *  \date 03-01-2012
 *  \brief Funciones que trabajan con vectores con elementos de tipo PdsNInput.
 *  
 *  <br>Estas funciones trabajan con una neurona de la forma.<br>
 *  \image html PdsNInput.png "Neurona de entrada de tipo PdsNInput."
 *  \f[ NInput=\{Y[0], Y[1]\} \f].
 */

#ifndef __PDSNINPUT_H__
#define __PDSNINPUT_H__

#ifdef __cplusplus
extern "C" {
#endif 

#include <stdio.h>
#include <stdlib.h>
#include <pds/pdssnglobal.h>

#ifndef TRUE
	#define TRUE 1
#endif

#ifndef FALSE
	#define FALSE 0
#endif

/** \defgroup PdsNInputGroup Módulo PdsNInput.
 *  \brief Funciones que trabajan con neuronas de entrada.
 *  
 *  <br>Estas funciones trabajan con una neurona de la forma.<br>
 *  \image html PdsNInput.png "Neurona de entrada de tipo PdsNInput."
 * @{
 */


/*! \struct PdsNInput
 *  \brief La estructura tipo  PdsNInput .
 *  Esta estructura genera una neurona de entrada.
 *  \image html PdsNInput.png "Neurona de entrada de tipo PdsNInput."
 *  Para usar incluir pds/pdssn.h.
 *  \ingroup PdsNInputGroup
 *  \author Fernando Pujaico Rivera
 */
typedef struct 
{
	/*! Valor de salida *Y, y un espacio extra Y[1]. */
	PdsSnReal Y[2];
}PdsNInput;



/** \fn PdsNInput *pds_ninput_new(void)
 *  \brief Crea una neurona de tipo PdsNInput e inicia con cero todos los elementos.
 *  \return Un puntero al vector de tipo PdsNInput.
 *  \ingroup PdsNInputGroup
 */
PdsNInput *pds_ninput_new(void);


/** \fn int pds_ninput_get_input(const PdsNInput *NInput,PdsSnReal *m)
 *  \brief Devuelve el valor de la entrada  de la neurona NInput.
 *  \param[in] NInput La neurona en consulta.
 *  \param[out] m Donde se cargará el valor de la entrada.
 *  \return TRUE si todo fue bien o FALSE si no (ej: NInput==NULL). 
 *  \ingroup PdsNInputGroup
 */
int pds_ninput_get_input(const PdsNInput *NInput,PdsSnReal *m);


/** \fn int pds_ninput_get_output(const PdsNInput *NInput,PdsSnReal *m)
 *  \brief Devuelve el valor de la salida Y de la neurona NInput.
 *  \param[in] NInput La neurona en consulta.
 *  \param[out] m Donde se cargará el valor de la salida Y.
 *  \return TRUE si todo fue bien o FALSE si no (ej: NInput==NULL). 
 *  \ingroup PdsNInputGroup
 */
int pds_ninput_get_output(const PdsNInput *NInput,PdsSnReal *m);


/** \fn int pds_ninput_set_input(PdsNInput *NInput,PdsSnReal m)
 *  \brief Escribe el valor en la entrada de la neurona NInput.
 *  \param[in,out] NInput La neurona en consulta.
 *  \param[in] m El valor que se cargará en la entrada.
 *  \return TRUE si todo fue bien o FALSE si no (ej: NInput==NULL). 
 *  \ingroup PdsNInputGroup
 */
int pds_ninput_set_input(PdsNInput *NInput,PdsSnReal m);


/** \fn void pds_ninput_free(PdsNInput *NInput)
 *  \brief Libera una neurona de tipo puntero PdsNInput.
 *  \param[in,out] NInput La neurona a liberar.
 *  \return No retorna valor.
 *  \ingroup PdsNInputGroup
 */
void pds_ninput_free(PdsNInput *NInput);


/** \fn void pds_ninput_destroy(PdsNInput **NInput)
 *  \brief Libera una neurona de tipo puntero PdsNInput, y limpia el puntero con NULL.
 *  \param[in,out] NInput La neurona a liberar y limpiar.
 *  \return No retorna valor.
 *  \ingroup PdsNInputGroup
 */
void pds_ninput_destroy(PdsNInput **NInput);

/**
 * @}
 */

#ifdef __cplusplus
}
#endif 

#endif


