/*
 * pdsneuron.h
 * 
 * Copyright 2011 Fernando Pujaico Rivera <fernando.pujaico.rivera@gmail.com>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 * 
 */

/** \file pdsneuron.h
 *  \author Fernando Pujaico Rivera
 *  \date 01-04-2011
 *  \brief Funciones que trabajan con vectores.
 *  
 *  <br>Estas funciones trabajan con una neurona de la forma.<br>
 *  \image html PdsNeuron.png "Neurona de n=Nel entradas"
 *  \b Nel es el número de elementos.
 */

#ifndef __PDSNEURON_H__
#define __PDSNEURON_H__

#ifdef __cplusplus
extern "C" {
#endif 

#include <stdio.h>
#include <stdlib.h>
#include <pds/pdsninput.h>
#include <pds/pdsnivector.h>
#include <pds/pdssnglobal.h>

#ifndef TRUE
	#define TRUE 1
#endif

#ifndef FALSE
	#define FALSE 0
#endif

/** \defgroup PdsNeuronGroup Módulo PdsNeuron.
 *  \brief Funciones que trabajan con neuronas.
 *  
 *  <br>Estas funciones trabajan con una neurona de la forma.<br>
 *  <center>
 *  \image html PdsNeuron.png "Neurona de n=Nel entradas."
 *  </center>
 * @{
 */



/*! \struct PdsNeuron
 *  \brief La estructura tipo  PdsNeuron .
 *  Esta estructura genera una neurona de Nel entradas.
 *
 *  <center>
 *  \image html PdsNeuron.png "Neurona de n=Nel entradas."
 *  </center>
 *  Para usar incluir pds/pdssn.h.
 *  \ingroup PdsNeuronGroup
 *  \author Fernando Pujaico Rivera
 */
typedef struct 
{
	/*! Un arreglo de punteros con Nel elementos. */
	PdsSnReal **X;

	/*! Un arreglo de Nel elementos. */
	PdsSnReal *W;

	/*! Número de elementos. */
	PdsSnNatural Nel;

	/*! Valor de umbral. */
	PdsSnReal U;

	/*! Valor de ponderacion. */
	PdsSnReal Theta;

	/*! Función de activación.*/
    double (*func)(double);

	/*! Derivada de la función de activación.*/
    double (*dfunc)(double);

	/*! Valor de salida *Y, y un espacio extra Y[1]. */
	PdsSnReal Y[2];
}PdsNeuron;




/** @name pds_neuron_new
 *  Creando nuevas neuronas
 *
 *  <center>
 *  \image html PdsNeuron.png "Neurona de n=Nel entradas."
 *  </center>
 *  <br>
 * @{
 */


/** \fn PdsNeuron *pds_neuron_new(PdsSnNatural Nel)
 *  \brief Crea una neurona de tipo PdsNeuron de Nel entradas.
 *
 *  Los valores por defecto son :
 *  \f[ U=1.0 \f]
 *  \f[ func(x)=tanh\left(\frac{x}{2}\right) \f]
 *  \param[in] Nel Es el número de entradas de la neurona.
 *  \return Un puntero al vector de tipo PdsNeuron.
 *  \ingroup PdsNeuronGroup
 */
PdsNeuron *pds_neuron_new(PdsSnNatural Nel);


/** \fn PdsNeuron *pds_neuron_new_with_params(PdsSnNatural Nel,PdsSnReal W,PdsSnReal U,double (*func)(double),double (*dfunc)(double));
 *  \brief Crea una neurona de tipo PdsNeuron de Nel entradas.
 *
 *  \param[in] Nel Es el número de entradas de la neurona.
 *  \param[in] W Peso inicial de todas las entradas
 *  \param[in] U Umbral de disparo.
 *  \param[in] func Función de activación.
 *  \param[in] dfunc Derivada de la función de activación.
 *  \return Un puntero al vector de tipo PdsNeuron.
 *  \ingroup PdsNeuronGroup
 */
PdsNeuron *pds_neuron_new_with_params(PdsSnNatural Nel,PdsSnReal W,PdsSnReal U,double (*func)(double),double (*dfunc)(double));

//@}

/** @name pds_neuron_<connect_input_methods>
 *  Conectando datos.
 *
 * @{
 */


/** \fn int pds_neuron_connect_input(PdsNeuron *Neuron,PdsSnNatural id,PdsNeuron *NeuronX)
 *  \brief Conecta el valor de salida Y de la Neurona NeuronX con la entrada X[id]
 *  de la neurona Neuron.
 *  \param[in,out] Neuron La neurona a trabajar.
 *  \param[in] id La posición a conectar, id tiene que ser menor que Nel.
 *  \param[in] NeuronX Neurona que se conectará a la entrada X[id] de Neuron.
 *  \return TRUE si todo fue bien o FALSE si no (ej: Neuron==NULL). 
 *  \ingroup PdsNeuronGroup
 */
int pds_neuron_connect_input(PdsNeuron *Neuron,PdsSnNatural id,PdsNeuron *NeuronX);




/** \fn int pds_neuron_connect_input_with_ninput(PdsNeuron *Neuron,PdsSnNatural id,PdsNInput *NInput)
 *  \brief Conecta el valor de salida Y de la neurona de entrada NInput con la 
 *  entrada X[id] de la neurona Neuron.
 *  \param[in,out] Neuron La neurona a trabajar.
 *  \param[in] id La posición a conectar, id tiene que ser menor que Nel.
 *  \param[in] NInput La neurona de entrada que se conectará a la entrada X[id] de Neuron.
 *  \return TRUE si todo fue bien o FALSE si no (ej: Neuron==NULL). 
 *  \ingroup PdsNeuronGroup
 */
int pds_neuron_connect_input_with_ninput(PdsNeuron *Neuron,PdsSnNatural id,PdsNInput *NInput);



/** \fn int pds_neuron_connect_input_with_nivector(PdsNeuron *Neuron,const PdsNIVector *NIVector)
 *  \brief Conecta el valor de salida Y de las neuronas NIVector con la 
 *  entrada X[id] de la neurona Neuron.
 *  \param[in,out] Neuron La neurona a trabajar.
 *  \param[in] NIVector Vector de neuronas de entrada que se conectará a la entrada X[id] de Neuron.
 *  \return TRUE si todo fue bien o FALSE si no (ej: Neuron==NULL, Neuron->Nel!=NInput->Nel). 
 *  \ingroup PdsNeuronGroup
 */
int pds_neuron_connect_input_with_nivector(PdsNeuron *Neuron,const PdsNIVector *NIVector);


//@}

/** @name pds_neuron_<get_set_input_methods>
 *  Trabajando con las entradas
 *
 * @{
 */

/** \fn const PdsSnReal *pds_neuron_get_dendrite(const PdsNeuron *Neuron,PdsSnNatural id)
 *  \brief Devuelve la dirección de la entrada X[id] de la Neurona.
 *  \param[in] Neuron La neurona en consulta.
 *  \param[in] id La posición a consultar, id tiene que ser menor que Nel.
 *  \return La dirección de la entrada X[id] o NULL en caso de error (ej: Neuron==NULL). 
 *  \ingroup PdsNeuronGroup
 */
const PdsSnReal *pds_neuron_get_dendrite(const PdsNeuron *Neuron,PdsSnNatural id);


/** \fn int pds_neuron_get_input(const PdsNeuron *Neuron,PdsSnNatural id,PdsSnReal *m)
 *  \brief Devuelve el valor de la entrada X[id] de la Neurona.
 *  \param[in] Neuron La neurona en consulta.
 *  \param[in] id La posición a consultar, id tiene que ser menor que Nel.
 *  \param[out] m Donde se cargará el valor de la entrada X[id].
 *  \return TRUE si todo fue bien o FALSE si no (ej: Neuron==NULL). 
 *  \ingroup PdsNeuronGroup
 */
int pds_neuron_get_input(const PdsNeuron *Neuron,PdsSnNatural id,PdsSnReal *m);


//@}

/** @name pds_neuron_<weight_methods>
 *  Trabajando con el synaptic_weight
 *
 * @{
 */

/** \fn int pds_neuron_get_weight(const PdsNeuron *Neuron, PdsSnNatural x, PdsSnReal *m)
 *  \brief Devuelve el valor en la posición (x) del vector de pesos de la Neurona.
 *  (x) inicia con (0).
 *  \param[in] Neuron La neurona en consulta.
 *  \param[in] x Posición x, el primer valor de x es cero.
 *  \param[out] m El valor en la posición (x), en caso de error por fuera de rango 
 *  (x) entonces carga 0 en m, en caso de error de vector nulo carga cero en m.
 *  \return TRUE si todo fue bien o FALSE si no (ej: Neuron==NULL). 
 *  \ingroup PdsNeuronGroup
 */
int pds_neuron_get_weight(const PdsNeuron *Neuron, PdsSnNatural x, PdsSnReal *m);


/** \fn int pds_neuron_set_weight(PdsNeuron *Neuron, PdsSnNatural x, PdsSnReal m)
 *  \brief Escribe el valor m en la posición (x) del vector de pesos de la Neurona.
 *  (x) inicia con (0).
 *  \param[in,out] Neuron La neurona a escribir.
 *  \param[in] x Posición x, el primer valor de x es cero.
 *  \param[in] m El valor en la posición (x), en caso de error por fuera de rango 
 *  (x) entonces no hace nada y no se considera como error.
 *  \return TRUE si todo fue bien o FALSE si no (ej: Neuron==NULL). 
 *  \ingroup PdsNeuronGroup
 */
int pds_neuron_set_weight(PdsNeuron *Neuron, PdsSnNatural x, PdsSnReal m);

/** \fn int pds_neuron_init_weight(PdsNeuron *Neuron, PdsSnReal m)
 *  \brief Inicia el vector de pesos de la Neurona.
 *  \param[in,out] Neuron La neurona a escribir.
 *  \param[in] m El valor de los pesos de la neurona.
 *  \return TRUE si todo fue bien o FALSE si no (ej: Neuron==NULL). 
 *  \ingroup PdsNeuronGroup
 */
int pds_neuron_init_weight(PdsNeuron *Neuron, PdsSnReal m);


/** \fn int pds_neuron_init_weight_rand(PdsNeuron *Neuron, PdsSnReal min, PdsSnReal max)
 *  \brief Inicia el vector de pesos de la Neurona con valores aleatorios
 *  uniformemente distribuido entre min y max.
 *  \param[in,out] Neuron La neurona a escribir.
 *  \param[in] min El minimo valor encontrado en los pesos de la neurona.
 *  \param[in] max El minimo valor encontrado en los pesos de la neurona.
 *  \return TRUE si todo fue bien o FALSE si no (ej: Neuron==NULL). 
 *  \ingroup PdsNeuronGroup
 */
int pds_neuron_init_weight_rand(PdsNeuron *Neuron, PdsSnReal min, PdsSnReal max);

/** \fn int pds_neuron_init_u_and_weight_rand(PdsNeuron *Neuron, PdsSnReal min, PdsSnReal max)
 *  \brief Inicia el vector de pesos y U de la Neurona con valores aleatorios
 *  uniformemente distribuido entre min y max.
 *  \param[in,out] Neuron La neurona a escribir.
 *  \param[in] min El minimo valor encontrado en los pesos de la neurona.
 *  \param[in] max El minimo valor encontrado en los pesos de la neurona.
 *  \return TRUE si todo fue bien o FALSE si no (ej: Neuron==NULL). 
 *  \ingroup PdsNeuronGroup
 */
int pds_neuron_init_u_and_weight_rand(PdsNeuron *Neuron, PdsSnReal min, PdsSnReal max);

/** \fn int pds_neuron_printf_weight(const PdsNeuron *Neuron)
 *  \brief Imprime los valores de los pesos en la Neurona.
 *  \param[in] Neuron La neurona en consulta.
 *  \return TRUE si todo fue bien o FALSE si no (ej: Neuron==NULL). 
 *  \ingroup PdsNeuronGroup
 */
int pds_neuron_printf_weight(const PdsNeuron *Neuron);


/** \fn int pds_neuron_printf_u_and_weight(const PdsNeuron *Neuron)
 *  \brief Imprime los valores de U y los pesos en la Neurona.
 *  \param[in] Neuron La neurona en consulta.
 *  \return TRUE si todo fue bien o FALSE si no (ej: Neuron==NULL). 
 *  \ingroup PdsNeuronGroup
 */
int pds_neuron_printf_u_and_weight(const PdsNeuron *Neuron);

/** \fn int pds_neuron_get_number_of_inputs(const PdsNeuron *Neuron,PdsSnNatural *Nelements)
 *  \brief Devuelve el número de entradas de la Neurona.
 *  \param[in] Neuron La neurona en consulta.
 *  \param[out] Nelements En donde se guardará el número de entradas de la neurona.
 *  \return TRUE si todo fue bien o FALSE si no (ej: Neuron==NULL). 
 *  \ingroup PdsNeuronGroup
 */
int pds_neuron_get_number_of_inputs(const PdsNeuron *Neuron,PdsSnNatural *Nelements);

//@}

/** @name pds_neuron_<u_methods>
 *  Trabajando con el valor U
 *
 * @{
 */


/** \fn int pds_neuron_get_u(const PdsNeuron *Neuron,PdsSnReal *U)
 *  \brief Devuelve el valor de  U de la Neurona.
 *  \param[in] Neuron La neurona en consulta.
 *  \param[out] U En donde se guardará el valor de U de la neurona.
 *  \return TRUE si todo fue bien o FALSE si no (ej: Neuron==NULL). 
 *  \ingroup PdsNeuronGroup
 */
int pds_neuron_get_u(const PdsNeuron *Neuron,PdsSnReal *U);


/** \fn int pds_neuron_set_u(PdsNeuron *Neuron,PdsSnReal U)
 *  \brief Coloca el valor de  U de la Neurona.
 *  \param[in,out] Neuron La neurona a escribir.
 *  \param[in] U El valor de U de la neurona.
 *  \return TRUE si todo fue bien o FALSE si no (ej: Neuron==NULL). 
 *  \ingroup PdsNeuronGroup
 */
int pds_neuron_set_u(PdsNeuron *Neuron,PdsSnReal U);



//@}

/** @name pds_neuron_<evaluate>
 *  Trabajando con la 
 *
 * @{
 */


/** \fn int pds_neuron_evaluate_theta(PdsNeuron *Neuron)
 *  \brief Evalua la variable theta de la Neurona Neuron.
 *  \f[ \theta=\sum_{i=0}^{N_{el}-1}{W[i]X[i]} - U \f].
 *
 *  Valores de entrada apuntadas a NULL no son consideradas en la suma.
 *  \param[in,out] Neuron La neurona a evaluar.
 *  \return TRUE si todo fue bien o FALSE si no (ej: Neuron==NULL). 
 *  \ingroup PdsNeuronGroup
 */
int pds_neuron_evaluate_theta(PdsNeuron *Neuron);


/** \fn int pds_neuron_evaluate_func(PdsNeuron *Neuron)
 *  \brief Actualiza el valor a la salida de la Neurona Neuron.
 *  \f[ Y=func\left(\theta\right) \f]
 *  Adicionalmente (Neuron->Y[1]=0)
 *  \f[ e=0 \f]
 *  \param[in,out] Neuron La neurona a evaluar.
 *  \return TRUE si todo fue bien o FALSE si no (ej: Neuron==NULL). 
 *  \ingroup PdsNeuronGroup
 */
int pds_neuron_evaluate_func(PdsNeuron *Neuron);


/** \fn int pds_neuron_iterate(PdsNeuron *Neuron)
 *  \brief Itera la Neurona Neuron.
 *  \f[ \theta=\sum_{i=0}^{N_{el}-1}{W[i]X[i]} - U \f]
 *  \f[ func\left(\theta\right) \f]
 *  \f[ Y=func\left(\theta\right) \f]
 *  Adicionalmente (Neuron->Y[1]=0)
 *  \f[ e=0 \f]
 *  \param[in,out] Neuron La neurona a iterar.
 *  \return TRUE si todo fue bien o FALSE si no (ej: Neuron==NULL). 
 *  \ingroup PdsNeuronGroup
 */
int pds_neuron_iterate(PdsNeuron *Neuron);


//@}

/** @name pds_neuron_<output_methods>
 *  Obteniendo datos.
 *
 * @{
 */

/** \fn const PdsSnReal *pds_neuron_get_axon(const PdsNeuron *Neuron)
 *  \brief Devuelve la dirección del axon (salida) de la Neurona.
 *  \param[in] Neuron La neurona en consulta.
 *  \return La dirección del axon (salida) o NULL en caso de error (ej: Neuron==NULL). 
 *  \ingroup PdsNeuronGroup
 */
const PdsSnReal *pds_neuron_get_axon(const PdsNeuron *Neuron);


/** \fn int pds_neuron_get_output(const PdsNeuron *Neuron,PdsSnReal *m)
 *  \brief Devuelve el valor de la salida Y de la Neurona.
 *  \param[in] Neuron La neurona en consulta.
 *  \param[out] m Donde se cargará el valor de la salida Y.
 *  \return TRUE si todo fue bien o FALSE si no (ej: Neuron==NULL). 
 *  \ingroup PdsNeuronGroup
 */
int pds_neuron_get_output(const PdsNeuron *Neuron,PdsSnReal *m);


//@}


/** @name pds_neuron_<others_methods>
 *  Trabajando con la 
 *
 * @{
 */



/** \fn int pds_neuron_evaluate_diff_error(PdsNeuron *Neuron,PdsSnReal y)
 *  \brief Compara el valor de salida \f$Y\f$ de la neurona, con "y", luego lo carga
 *  en la variable Y[1] de la neurona (Neuron->Y[1]=y-Neuron->Y[0]).
 *  \f[ e= y - Y\f]
 *  \param[in,out] Neuron La neurona a evaluar.
 *  \param[in] y Valor a comparar.
 *  \return TRUE si todo fue bien o FALSE si no (ej: Neuron==NULL). 
 *  \ingroup PdsNeuronGroup
 */
int pds_neuron_evaluate_diff_error(PdsNeuron *Neuron,PdsSnReal y);


/** \fn int pds_neuron_backpropagate_error(PdsNeuron *Neuron,PdsSnReal Alpha)
 *  \brief Propaga el error de la salida de la neurona hacia las neuronas conectadas
 *  a sus entradas. <br>
 *  \f[ e = e_{\{Y\}} \f]
 *  \f[ e_i = e_{\{X[i]\}} \f]
 *  \f[ e_i \leftarrow e_i + func'\left(\theta\right)W[i]e\f]
 *  \image html backpropagation.png "Backpropagation"
 *  <br>
 *  \image html backpropagation2.png "Backpropagation"
 *  <br>
 *  "e" es el error de la salida de la neurona Neuron :Neuron->Y[1].<br>
 *  "func" es la función de activación da neurona Neuron.<br>
 *  \param[in,out] Neuron La neurona a retropropagar el error.
 *  \param[in] Alpha Factor de aprendizaje de los errores.
 *  \return TRUE si todo fue bien o FALSE si no (ej: Neuron==NULL). 
 *  \ingroup PdsNeuronGroup
 */
int pds_neuron_backpropagate_error(PdsNeuron *Neuron,PdsSnReal Alpha);




//@}

/** @name pds_neuron_<update_weight>
 *  Atualizando peso de las neuronas.
 *
 * @{
 */

/** \fn int pds_neuron_update_weight(PdsNeuron *Neuron, PdsSnReal Alpha)
 *  \brief Actualiza los pesos W[i] de la neurona (using the Delta rule).
 *  \f[ e = r-Y\f]
 *  \f[ \triangle W[i]= \alpha func'\left(\theta\right) X[i] e\f]
 *  "e" es el error de la salida de la neurona Neuron :Neuron->Y[1].<br>
 *  "Y" salida de la neurona Neuron :Neuron->Y[0].<br>
 *  "func" es la función de activación da neurona Neuron.<br>
 *  "X" Vector con los valores de todas las entradas de la neurona.<br>
 *  \param[in,out] Neuron La neurona a actualizar los pesos.
 *  \param[in] Alpha Factor de aprendizaje de los pesos.
 *  \return TRUE si todo fue bien o FALSE si no (ej: Neuron==NULL). 
 *  \ingroup PdsNeuronGroup
 */
int pds_neuron_update_weight(PdsNeuron *Neuron, PdsSnReal Alpha);


/** \fn int pds_neuron_update_u_and_weight(PdsNeuron *Neuron, PdsSnReal Alpha)
 *  \brief Actualiza los pesos W[i] de la neurona (using the Delta rule).
 *  \f[ e = r-Y\f]
 *  \f[ \triangle W[i]= \alpha func'\left(\theta\right) X[i] e\f]
 *  \f[ \triangle U= -\alpha func'\left(\theta\right) e\f]
 *  "e" es el error de la salida de la neurona Neuron :Neuron->Y[1].<br>
 *  "Y" salida de la neurona Neuron :Neuron->Y[0].<br>
 *  "func" es la función de activación da neurona Neuron.<br>
 *  "X" Vector con los valores de todas las entradas de la neurona.<br>
 *  \param[in,out] Neuron La neurona a actualizar los pesos.
 *  \param[in] Alpha Factor de aprendizaje de los pesos.
 *  \return TRUE si todo fue bien o FALSE si no (ej: Neuron==NULL). 
 *  \ingroup PdsNeuronGroup
 */
int pds_neuron_update_u_and_weight(PdsNeuron *Neuron, PdsSnReal Alpha);


/** \fn int pds_neuron_get_xtx(PdsNeuron *Neuron,PdsSnReal *XtX)
 *  \brief Devuelve la suma cuadrática de los valores de entrada de la neurona.
 *  \f[ X^T X=\sum^{Nel-1}_{i=0} X[i]^2 \f].
 *  \param[in,out] Neuron La neurona en consulta.
 *  \param[in] XtX Donde se cargará la suma cuadrática.
 *  \return TRUE si todo fue bien o FALSE si no (ej: Neuron==NULL). 
 *  \ingroup PdsNeuronGroup
 */
int pds_neuron_get_xtx(PdsNeuron *Neuron,PdsSnReal *XtX);


/** \fn int pds_neuron_update_u_and_weight_normalized(PdsNeuron *Neuron, PdsSnReal Alpha)
 *  \brief Actualiza los pesos W[i] de la neurona (using the Delta rule).
 *  \f[ e = r-Y\f]
 *  \f[ \triangle W[i]= \alpha func'\left(\theta\right) \frac{X[i]}{X^TX+1} e \f]
 *  \f[ \triangle U= -\alpha func'\left(\theta\right) \frac{1}{X^TX+1} e \f]
 *  "e" es el error de la salida de la neurona Neuron :Neuron->Y[1].<br>
 *  "Y" salida de la neurona Neuron :Neuron->Y[0].<br>
 *  "func" es la función de activación da neurona Neuron.<br>
 *  "X" Vector con los valores de todas las entradas de la neurona.<br>
 *  \param[in,out] Neuron La neurona a actualizar los pesos.
 *  \param[in] Alpha Factor de aprendizaje de los pesos.
 *  \return TRUE si todo fue bien o FALSE si no (ej: Neuron==NULL). 
 *  \ingroup PdsNeuronGroup
 */
int pds_neuron_update_u_and_weight_normalized(PdsNeuron *Neuron, PdsSnReal Alpha);

//@}

/** @name pds_neuron_<printf>
 *  Imprimeindo y leyendo datos.
 *
 * @{
 */

/** \fn int pds_neuron_fprintf(const PdsNeuron *Neuron, FILE *fd)
 *  \brief Guarda en un archivo de texto los pesos {W[i]} y  {U}.
 *  Ocupando una linea cada uno, y separando los elementos por un TAB.
 *  \param[in] Neuron La neurona a leer.
 *  \param[in,out] fd Manejador del fichero a escribir.
 *  \return TRUE si todo fue bien o FALSE si no (ej: NVector==NULL o fd==NULL). 
 *  \ingroup PdsNeuronGroup
 */
int pds_neuron_fprintf(const PdsNeuron *Neuron, FILE *fd);


/** \fn int pds_neuron_fscanf(PdsNeuron *Neuron, FILE *fd)
 *  \brief Lee de un archivo de texto los pesos {W[i]} y  {U}.
 *  Ocupando una linea cada uno, y separando los elementos por un TAB.
 *  \param[out] Neuron La neurona a escribir.
 *  \param[in,out] fd Manejador del fichero a leer.
 *  \return TRUE si todo fue bien o FALSE si no (ej: NVector==NULL o fd==NULL). 
 *  \ingroup PdsNeuronGroup
 */
int pds_neuron_fscanf(PdsNeuron *Neuron, FILE *fd);


/** \fn int pds_neuron_fwrite(const PdsNeuron *Neuron, FILE *fd)
 *  \brief Guarda en un archivo binario los pesos W[i], el valor de U.
 *  \param[in] Neuron La neurona a leer.
 *  \param[in,out] fd Manejador del fichero binario a escribir.
 *  \return TRUE si todo fue bien o FALSE si no (ej: NVector==NULL o fd==NULL). 
 *  \ingroup PdsNeuronGroup
 */
int pds_neuron_fwrite(const PdsNeuron *Neuron, FILE *fd);


/** \fn int pds_neuron_fread(PdsNeuron *Neuron, FILE *fd)
 *  \brief Lee de un archivo binario los pesos W[i], el valor de U.
 *  \param[out] Neuron La neurona a escribir.
 *  \param[in,out] fd Manejador del fichero binario a leer.
 *  \return TRUE si todo fue bien o FALSE si no (ej: NVector==NULL o fd==NULL). 
 *  \ingroup PdsNeuronGroup
 */
int pds_neuron_fread(PdsNeuron *Neuron, FILE *fd);

//@}

/** @name pds_neuron_<free>
 *  Limpiando memoria.
 *
 * @{
 */

/** \fn void pds_neuron_free(PdsNeuron *Neuron)
 *  \brief Libera una neurona de tipo puntero PdsNeuron.
 *  \param[in,out] Neuron La neurona a liberar.
 *  \return No retorna valor.
 *  \ingroup PdsNeuronGroup
 */
void pds_neuron_free(PdsNeuron *Neuron);


/** \fn void pds_neuron_destroy(PdsNeuron **Neuron)
 *  \brief Libera una neurona de tipo puntero PdsNeuron, y limpia el puntero con NULL.
 *  \param[in,out] Neuron La neurona a liberar y limpiar.
 *  \return No retorna valor.
 *  \ingroup PdsNeuronGroup
 */
void pds_neuron_destroy(PdsNeuron **Neuron);

//@}

/**
 * @}
 */

#ifdef __cplusplus
}
#endif 

#endif


