/*
 * example_neuron.c
 * 
 * Copyright 2011 Fernando Pujaico Rivera <fernando.pujaico.rivera@gmail.com>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 * 
 */

/** \example example_neuron.c
 *  \brief Programa para el testeo de algunas funciones de la biblioteca.
    
Para compilar el código de forma estática.
\code{.c}
    #Statically linked.
    gcc -static -o example_neuron example_neuron.c -lpdssn -lm
\endcode
    
Para compilar el código de forma estática.
\code{.c}
    #Dynamically linked.
    gcc -o example_neuron example_neuron.c -lpdssn -lm
\endcode
\n\n

 */
    
#include <stdlib.h>
#include <pds/pdssn.h>
    
PdsSnReal load_input_and_iterate_neuron(PdsNIVector *Input,const PdsSnReal *X,PdsSnNatural N,PdsNeuron *Neuron)
{
    PdsSnReal m;
    // Load data in the input vector
    pds_nivector_set_input_from_array(Input,X,N);
    
    // Iterate
    pds_neuron_iterate(Neuron);
    
    // Get the output value
    pds_neuron_get_output(Neuron,&m);
    return m;
}
    
void evaluate_error_and_update_weight(PdsNeuron *Neuron,PdsSnReal Y,PdsSnReal Alpha)
{
    pds_neuron_evaluate_diff_error(Neuron,Y);
    pds_neuron_update_u_and_weight(Neuron,Alpha);
}
    
int main(int argc, char** argv)
{
    PdsNeuron *Neuron=NULL;
    PdsSnReal Out0,Out1;
    int i=0;
    PdsSnReal Alpha=0.1;

    PdsNIVector *Input=NULL;
    PdsSnReal X0[4]={1.0, 0.5, -0.3, -1.7};
    PdsSnReal X1[4]={1.0,-1.5, +0.3, -0.7};
    
    Input=pds_nivector_new(4);
    Neuron=pds_neuron_new(4);
    
    if( (Input==NULL)||(Neuron==NULL) ) return EXIT_SUCCESS;
    
    // Connect the input vector with the input of neuron
    pds_neuron_connect_input_with_nivector(Neuron,Input);
    
    // Initializing weights
    pds_neuron_init_u_and_weight_rand(Neuron,-1.0,+1.0);
    pds_neuron_printf_u_and_weight(Neuron);
    
    Out1=load_input_and_iterate_neuron(Input,X1,4,Neuron);
    Out0=load_input_and_iterate_neuron(Input,X0,4,Neuron);
    printf("Iterando %d:\tY0=%7.3f\tY1=%7.3f\n",i,Out0,Out1);
    
    // Updating weights
    for(i=0;i<200;i++)
    {
        load_input_and_iterate_neuron(Input,X0,4,Neuron);        
        evaluate_error_and_update_weight(Neuron,-1.0,Alpha);        
        
        load_input_and_iterate_neuron(Input,X1,4,Neuron);
        evaluate_error_and_update_weight(Neuron,+1.0,Alpha);
    }
    
    Out1=load_input_and_iterate_neuron(Input,X1,4,Neuron);
    Out0=load_input_and_iterate_neuron(Input,X0,4,Neuron);
    printf("Iterando %d:\tY0=%7.3f\tY1=%7.3f\n",i,Out0,Out1);
    
    pds_neuron_printf_u_and_weight(Neuron);
    
    // Release the memory
    pds_nivector_destroy(&Input);
    pds_neuron_destroy(&Neuron);
    
    return EXIT_SUCCESS;
}
