/***************************************************************************
 *   Copyright (C) 2006-2008 by Paul-Louis Ageneau                         *
 *   paullouisageneau@gmail.com                                            *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.           *
 ***************************************************************************/

#include "animatedmesh.h"
#include "bufferarray.h"
#include "bufferobject.h"
#include "vector3.h"

CAnimatedMesh::CAnimatedMesh(	index_t nvertices,
								const index_t *indices, 
								index_t nindices)

{
	mNbrVertices=nvertices;
	mNbrIndices=nindices;
	mNbrTexCoords=1;
	mTexCoordBuffers=new CSmartPtr<TexCoordBuffer_t>[1];
	mFaceNeighbours=NULL;
	
	// Cration des Buffers
	if(glewIsSupported("GL_vertex_buffer_object"))	// Teste la disponibilit des VBOs
	{
		mIndexBuffer	= new IndexBuffer_t(new CIndexBufferObject);
		mVertexBuffer 	= new VertexBuffer_t(new CDataBufferObject);
		mNormalBuffer	= new NormalBuffer_t(new CDataBufferObject);
		mTexCoordBuffers[0] = new TexCoordBuffer_t(new CDataBufferObject);
	} 
	else
	{
		mIndexBuffer	= new IndexBuffer_t(new CBufferArray);
		mVertexBuffer	= new VertexBuffer_t(new CBufferArray);
		mNormalBuffer	= new NormalBuffer_t(new CBufferArray);
		mTexCoordBuffers[0] = new TexCoordBuffer_t(new CBufferArray);
	}
	
	mIndexBuffer->Fill(indices,nindices);

	// Matriau par dfaut
	setMaterial(CMaterial::Default,0);

	mCurrentFrame = -1.;
}

CAnimatedMesh::~CAnimatedMesh(void)
{

}


void CAnimatedMesh::AddFrame(double frame,const float *vertices,const float *texcoords,const float *normals)
{
	KeyMap_t::iterator it;
		
	if(vertices)
	{
		// Si la frame existait dj
		it = mVertexKeyMap.find(frame);
		float *p;
		if(it != mVertexKeyMap.end()) p = it->second;
		else p = new float[mNbrVertices*3];

		// Cration du nouveau buffer
		std::copy(vertices,vertices+mNbrVertices*3,p);
		mVertexKeyMap.insert(it,std::make_pair(frame,p));	// insertion

		// Calcul du rayon
		float radius2=0.f;
		for(index_t i=0; i<mNbrVertices*3; i+=3)
		{
			float d2=CCoord3(vertices[i],vertices[i+1],vertices[i+2]).Norm2();
			radius2=std::max(radius2,d2);
		}
		mRadius=std::max(std::sqrt(radius2),mRadius);
	}

	// Buffer des normales
	if(normals || vertices)
	{
		// Si la frame existait dj
		it = mNormalKeyMap.find(frame);
		float *p;
		if(it != mNormalKeyMap.end()) p=it->second;
		else p = new float[mNbrVertices*3];
		
		if(normals) std::copy(normals,normals+mNbrVertices*3,p);
		else {
			// TODO: code prsent aussi dans CMesh: en faire une seule fonction
			std::fill(p,p+mNbrVertices*3,0.f);

			// Calcul des normales
			const index_t *indices=mIndexBuffer->Lock(0,mIndexBuffer->getCount(),GL_READ_ONLY);
			for(index_t i=0; i<mNbrIndices; i+=3)
			{
				// Les 3 vertices de la face
				CCoord3 v1(vertices + indices[i]*3);
				CCoord3 v2(vertices + indices[i+1]*3);
				CCoord3 v3(vertices + indices[i+2]*3);

				// Produit vectoriel pour trouver une normale
				CVector3 normal=(v2-v1).Crosspoint(v3-v1);
				normal.Normalize();

				// Ajoute la normale de la face  celle de chaque vertice
				for(index_t j=i;j<i+3;++j)
				{
					p[indices[j]*3]+=normal.x;
					p[indices[j]*3+1]+=normal.y;
					p[indices[j]*3+2]+=normal.z;
				}
			}
			mIndexBuffer->Unlock();
		}

		mNormalKeyMap.insert(it,std::make_pair(frame,p));	// insertion
	}

	// Buffer des coordonnes de texture
	if(texcoords)
	{
		it = mTexCoordKeyMap.find(frame);
		if(it != mTexCoordKeyMap.end()) delete[] it->second;

		float *p=new float[mNbrVertices*2];
		std::copy(texcoords,texcoords+mNbrVertices*2,p);
		mTexCoordKeyMap.insert(it,std::make_pair(frame,p));
	}
}

int CAnimatedMesh::Draw(int pass,double frame,pMaterial force)
{
	BuildFrame(frame);
	return CMesh::Draw(pass,frame,force);
}

void CAnimatedMesh::EnableShadowVolume(void)
{
	if(mCurrentFrame<0.) BuildFrame(0.);
	return CMesh::EnableShadowVolume();
}

int CAnimatedMesh::DrawShadowVolume(double frame,const CVector3 &light,bool directional)
{
	BuildFrame(frame);
	return CMesh::DrawShadowVolume(frame,light,directional);
}

float CAnimatedMesh::Intersect(double frame,const CCoord3 &pos,const CVector3 &move,float radius,CCoord3 *intersection)
{
	BuildFrame(frame);
	return CMesh::Intersect(frame,pos,move,radius,intersection);
}

void CAnimatedMesh::BuildFrame(double frame)
{			  
	if(std::abs(mCurrentFrame-frame)<=std::numeric_limits<double>::epsilon()) return;
	mCurrentFrame = frame;
	
	// Vertices
	mVertexBuffer->Fill(NULL,mNbrVertices*3,GL_STREAM_DRAW);					// Invalide les anciennes donnes
	float *vertices=mVertexBuffer->Lock(0,mNbrVertices*3,GL_WRITE_ONLY);		// Verrouille en criture
	Interpolate(mVertexKeyMap,frame,vertices,mNbrVertices*3);					// Ecrit le buffer interpol
	mVertexBuffer->Unlock();													// Dverrouille

	// Normales
	mNormalBuffer->Fill(NULL,mNbrVertices*3,GL_STREAM_DRAW);
	float *normals=mNormalBuffer->Lock(0,mNbrVertices*3,GL_WRITE_ONLY);
	Interpolate(mNormalKeyMap,frame,normals,mNbrVertices*3);
	mNormalBuffer->Unlock();
	
	// Coordonnes de texture
	mTexCoordBuffers[0]->Fill(NULL,mNbrVertices*2,GL_STREAM_DRAW);
	float *texcoords=mTexCoordBuffers[0]->Lock(0,mNbrVertices*2,GL_WRITE_ONLY);
	Interpolate(mTexCoordKeyMap,frame,texcoords,mNbrVertices*2);
	mTexCoordBuffers[0]->Unlock();
}

void CAnimatedMesh::Interpolate(const KeyMap_t &keymap,double frame,float *buffer,index_t size)
{
	if(keymap.empty()) throw CException("Modle anim sans cl d'animation");
	
	// Si une seule cl pas d'animation
	if(keymap.size()==1)
	{
		std::copy(keymap.begin()->second,keymap.begin()->second+size,buffer);
		return;
	}

	// Cherche la frame disponible qui suit
	KeyMap_t::const_iterator it2 = keymap.upper_bound(frame);
	
	while(it2==keymap.end())
	{
		frame-=((--keymap.end())->first-keymap.begin()->first);
		it2=keymap.upper_bound(frame);
	}
	
	while(it2==keymap.begin())
	{
		frame+=((--keymap.end())->first-keymap.begin()->first);
		it2=keymap.upper_bound(frame);
	}
	
	KeyMap_t::const_iterator it1=it2;
	--it1;

	float t = (frame - it1->first)/(it2->first-it1->first);
	const float *sbuffer1 = it1->second;
	const float *sbuffer2 = it2->second;
	
	// Interpolation des valeurs
	for(index_t i=0; i<size; ++i) buffer[i] = sbuffer1[i]+t*(sbuffer2[i]-sbuffer1[i]);
}
