/***************************************************************************
 *   Copyright (C) 2006-2008 by Paul-Louis Ageneau                         *
 *   paullouisageneau@gmail.com                                            *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.           *
 ***************************************************************************/

#include "engine.h"
#include "mediamanager.h"
#include "resourcemanager.h"
#include "sound.h"
#include "text.h"
#include "imageloader.h"
#include "3dsloader.h"
#include "md2loader.h"
#include "oggvorbisloader.h"
#include "scriptloader.h"
#include "exception.h"


CEngine::CEngine(void) :
	mOldMousex(0),
	mOldMousey(0),
	mFPS(0.f)
{

}

CEngine::~CEngine(void)
{
	Cleanup();
}

void CEngine::Init(void)
{
	srand(time(NULL));		// initialisation des nombres alatoires
	
	glfwInit();				// initialisation glfw
	CSound::Init();			// initialisation du son

	// Enregistrement des loaders
	CMediaManager::Instance()->RegisterLoader<CImage>(new CImageLoader,"bmp,jpg,jpeg,png,pcx,dds,pnm,sgi,tga,tif,gif");
	CMediaManager::Instance()->RegisterLoader<CTexture>(new CTextureLoader,"bmp,jpg,jpeg,png,pcx,dds,pnm,sgi,tga,tif,gif");
	CMediaManager::Instance()->RegisterLoader<CMesh>(new C3dsLoader,"3ds");
	CMediaManager::Instance()->RegisterLoader<CMesh>(new CMd2Loader,"md2");
	CMediaManager::Instance()->RegisterLoader<CSample>(new COggVorbisSampleLoader,"ogg");
	CMediaManager::Instance()->RegisterLoader<CMusic>(new COggVorbisMusicLoader,"ogg");
	CMediaManager::Instance()->RegisterLoader<CScript>(new CScriptLoader,"txt,p3d,xml,script");
}

void CEngine::Cleanup(void)
{
	mStates.clear();	// vidange de la liste d'tats
	CloseWindow();		// ferme la fentre
	
	CSound::Cleanup();	// fermeture du son
	glfwTerminate();	// fermeture glfw
}

void CEngine::OpenWindow(int width, int height, bool fullscreen)
{
	// TODO: rglage stencil

	int mode;
	if(fullscreen) mode=GLFW_FULLSCREEN;
	else mode=GLFW_WINDOW;
	if(!glfwOpenWindow(width, height,		// ouvre la fentre
			8, 8, 8, 0,	// RGBA bits
			16, 		// depth bits
			16, 		// stencil bits
			mode)) throw CException("Echec lors de l'ouverture de la fentre");

	glfwPollEvents();

	// callbacks
	glfwSetWindowCloseCallback(CloseCallback);
	glfwSetKeyCallback(KeyCallback);
	glfwSetCharCallback(CharCallback);
	glfwSetMouseButtonCallback(MouseCallback);
	glfwSwapInterval(0);

	glewInit();			// initialisation glew

	// Rglages globaux OpenGL
	glHint(GL_PERSPECTIVE_CORRECTION_HINT,GL_NICEST);
	
	// Ecran noir
	glClearColor(0.f,0.f,0.f,0.f);
	glClear(GL_COLOR_BUFFER_BIT);
	glfwSwapBuffers();
	
	// Mise  jour des anciennes positions de la souris
	glfwGetMousePos(&mOldMousex,&mOldMousey);

	CText::Init();			// initialisation du texte
}

void CEngine::CloseWindow(void)
{
	if(glfwGetWindowParam(GLFW_OPENED)) 
	{
		glfwCloseWindow();
		CText::Cleanup();	// fermeture du texte
	}
}

void CEngine::setWindowTitle(const char *title)
{
	glfwSetWindowTitle(title);
}

void CEngine::setWindowSize(int width, int height)
{
	glfwSetWindowSize(width,height);
}

void CEngine::setWindowPosition(int x, int y)
{
	glfwSetWindowPos(x,y);
}

void CEngine::setCursor(bool visible)
{
	if(visible) glfwEnable(GLFW_MOUSE_CURSOR);
	else glfwDisable(GLFW_MOUSE_CURSOR);
}

pEngineState CEngine::ChangeState(pEngineState state)
{
	pEngineState oldstate=PopState();
	PushState(state);
	return oldstate;
}

void CEngine::PushState(pEngineState state)
{
	CSound::setGlobalGain(0.f);
	
	mStates.push_back(state);
	state->Init();
	mMesureTime=mOldTime=glfwGetTime();
	mMesureFrames=0;
}

pEngineState CEngine::PopState(void)
{
	CSound::setGlobalGain(0.f);
	
	pEngineState oldstate=mStates.back();
	oldstate->Cleanup();
	mStates.pop_back();
	
	CResourceManager::Instance()->Flush();

	mMesureTime=mOldTime=glfwGetTime();
	mMesureFrames=0;
	return oldstate;
}

pEngineState CEngine::getTopState(void)
{
	if(mStates.empty()) return NULL;
	else return mStates.back();
}

bool CEngine::Update(void)
{
	glfwPollEvents();
	if(mStates.empty() || !glfwGetWindowParam(GLFW_OPENED)) return false;

	double CurrentTime=glfwGetTime();		// temps courant
	double time=CurrentTime-mOldTime;		// le temps coul depuis la dernire frame
	
	if(time < 1./MAX_FRAMERATE) 
	{
		glfwSleep(1./MAX_FRAMERATE-time);
		CurrentTime=glfwGetTime();
		time=CurrentTime-mOldTime;
	}

	mOldTime=CurrentTime;					// la frame en cours devient l'ancienne

	while(!mStates.back()->Update(time)) {		// mise  jour pour l'tat courant
		PopState();				// si tat termin on l'enlve de la pile
		if(mStates.empty()) return false;
	}
	
	CSound::setGlobalGain(1.f);
	CSound::Process();
	return true;
}

int CEngine::Display(void)
{
	if(mStates.empty() || !glfwGetWindowParam(GLFW_OPENED)) return 0;

	glMatrixMode(GL_MODELVIEW);
	glLoadIdentity(); 
	
	int count=mStates.back()->Draw();
	glfwSwapBuffers();

	// Cacul du FPS courant
	++mMesureFrames;
	if(mMesureFrames>10)
	{
		mFPS=mMesureFrames/(glfwGetTime()-mMesureTime);
		mMesureTime=glfwGetTime();
		mMesureFrames=0;
	}

	return count;
}

bool CEngine::isKeyDown(int key)
{
	return (glfwGetKey(key)==GLFW_PRESS);
}

bool CEngine::isMouseButtonDown(int button)
{
	return (glfwGetMouseButton(button)==GLFW_PRESS);
}

void CEngine::getMousePosition(int *x,int *y,int *z)
{
	glfwGetMousePos(x,y);
	if(z) *z = glfwGetMouseWheel();
}

void CEngine::getMouseMove(int *x,int *y,int *z)
{
	int newx,newy,newz;
	glfwGetMousePos(&newx,&newy);
	newz = glfwGetMouseWheel();

	if(x) *x=newx-mOldMousex;
	if(y) *y=newy-mOldMousey;
	if(z) *z=newz-mOldMousez;
	mOldMousex=newx;
	mOldMousey=newy;
	mOldMousez=newz;
}

int CEngine::getMouseWheel(void)
{
	return glfwGetMouseWheel();
}

double CEngine::getTime(void)
{
	return glfwGetTime();
}

double CEngine::getTimestamp(void)
{
	return mOldTime;
}

float CEngine::getFPS(void)
{
	return mFPS;
}

int GLFWCALL CEngine::CloseCallback(void)
{
	CEngine::Instance()->CloseWindow();
	return GL_TRUE;
}

void GLFWCALL CEngine::KeyCallback(int key,int action)
{
	if(!CEngine::Instance()->mStates.empty()) CEngine::Instance()->mStates.back()->KeyCallback(key,action);
}
		
void GLFWCALL CEngine::CharCallback(int character,int action)
{
	if(!CEngine::Instance()->mStates.empty()) CEngine::Instance()->mStates.back()->CharCallback(character,action);
}

void GLFWCALL CEngine::MouseCallback(int button,int action)
{
	if(!CEngine::Instance()->mStates.empty()) CEngine::Instance()->mStates.back()->MouseCallback(button,action);
}
