/***************************************************************************
 *   Copyright (C) 2006-2008 by Paul-Louis Ageneau                         *
 *   paullouisageneau@gmail.com                                            *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.           *
 ***************************************************************************/

#ifndef ENGINE_H
#define ENGINE_H

#include "singleton.h"
#include "smartptr.h"
#include "include.h"

#define MAX_FRAMERATE 80.	// 80 frames/sec max

class IEngineState;
typedef CSmartPtr<IEngineState> pEngineState;

// Classe singleton CEngine
// Systme central du Moteur 3D
class CEngine : public ISingleton<CEngine>
{
	
public:
	void Init(void);
	void Cleanup(void);
	void OpenWindow(int width, int height, bool fullscreen=false);
	void CloseWindow(void);
	void setWindowTitle(const char *title);
	void setWindowSize(int width, int height);
	void setWindowPosition(int x, int y);
	void setCursor(bool visible);
	bool isKeyDown(int key);
	bool isMouseButtonDown(int button);
	void getMousePosition(int *x,int *y,int *z=NULL);
	void getMouseMove(int *x,int *y,int *z=NULL);
	int getMouseWheel(void);
	double getTime(void);
	double getTimestamp(void);
	float  getFPS(void);

	pEngineState ChangeState(pEngineState state);
	void PushState(pEngineState state);
	pEngineState PopState(void);
	pEngineState getTopState(void);
	
	bool Update(void);
	int Display(void);

#ifdef GLFWCALL
	static int GLFWCALL CloseCallback(void);
	static void GLFWCALL KeyCallback(int key,int action);
	static void GLFWCALL CharCallback(int character,int action);
	static void GLFWCALL MouseCallback(int button,int action );
#else
	static int CloseCallback(void);
	static void KeyCallback(int key,int action);
	static void CharCallback(int character,int action);
	static void MouseCallback(int button,int action );
#endif

private:
	std::vector<pEngineState> mStates;
	double mOldTime,mMesureTime;
	unsigned long mMesureFrames;
	float mFPS;
	int mOldMousex,mOldMousey,mOldMousez;

	CEngine(void);
	~CEngine(void);

	friend class ISingleton<CEngine>;

};

#define Engine CEngine::Instance()

class IEngineState : public IRefCount
{
protected:
	//   Raccourcis
	inline bool isTopState(void) 			{return (Engine->getTopState()==this);}
	inline bool isKeyDown(int key) 			{return Engine->isKeyDown(key);}
	inline bool isMouseButtonDown(int button)	{return Engine->isMouseButtonDown(button);}
	inline void getMousePosition(int *x,int *y)	{return Engine->getMousePosition(x,y);}
	inline void getMouseMove(int *x,int *y)		{return Engine->getMouseMove(x,y);}
	inline int getMouseWheel(void)			{return Engine->getMouseWheel();}
	inline double getTime(void)			{return Engine->getTime();}
	
public:
	IEngineState(void) {}
	virtual ~IEngineState(void) {}
	
	virtual void Init(void)=0;
	virtual void Cleanup(void)=0;

	virtual void KeyCallback(int key,int action) {}
	virtual void CharCallback(int character,int action) {}
	virtual void MouseCallback(int button,int action) {}

	virtual bool Update(double time)=0;
	virtual int Draw(void)=0;
};


#define KEY_SPACE	GLFW_KEY_SPACE
#define KEY_ESCAPE	GLFW_KEY_ESC
#define KEY_F1		GLFW_KEY_F1
#define KEY_F2		GLFW_KEY_F2
#define KEY_F3		GLFW_KEY_F3
#define KEY_F4		GLFW_KEY_F4
#define KEY_F5		GLFW_KEY_F5
#define KEY_F6		GLFW_KEY_F6
#define KEY_F7		GLFW_KEY_F7
#define KEY_F8		GLFW_KEY_F8
#define KEY_F9		GLFW_KEY_F9
#define KEY_F10		GLFW_KEY_F10
#define KEY_F11		GLFW_KEY_F11
#define KEY_F12		GLFW_KEY_F12
#define KEY_UP		GLFW_KEY_UP
#define KEY_DOWN	GLFW_KEY_DOWN
#define KEY_LEFT	GLFW_KEY_LEFT
#define KEY_RIGHT	GLFW_KEY_RIGHT
#define KEY_LSHIFT	GLFW_KEY_LSHIFT
#define KEY_RSHIFT	GLFW_KEY_RSHIFT
#define KEY_LCTRL	GLFW_KEY_LCTRL
#define KEY_RCTRL	GLFW_KEY_RCTRL
#define KEY_LALT	GLFW_KEY_LALT
#define KEY_RALT	GLFW_KEY_RALT
#define KEY_TAB		GLFW_KEY_TAB
#define KEY_ENTER	GLFW_KEY_ENTER
#define KEY_BACKSPACE	GLFW_KEY_BACKSPACE
#define KEY_INSERT	GLFW_KEY_INSERT
#define KEY_DEL		GLFW_KEY_DEL
#define KEY_PAGEUP	GLFW_KEY_PAGEUP
#define KEY_PAGEDOWN	GLFW_KEY_PAGEDOWN
#define KEY_HOME	GLFW_KEY_HOME
#define KEY_END		GLFW_KEY_END

#define KEY_PRESS		GLFW_PRESS
#define KEY_RELEASE		GLFW_RELEASE

#define MOUSE_BUTTON_LEFT	GLFW_MOUSE_BUTTON_LEFT
#define MOUSE_BUTTON_RIGHT	GLFW_MOUSE_BUTTON_RIGHT	
#define MOUSE_BUTTON_MIDDLE	GLFW_MOUSE_BUTTON_MIDDLE

#endif
