/***************************************************************************
 *   Copyright (C) 2006-2008 by Paul-Louis Ageneau                         *
 *   paullouisageneau@gmail.com                                            *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.           *
 ***************************************************************************/

#ifndef ENTITY_H
#define ENTITY_H

#include "scriptable.h"
#include "animation.h"
#include "frustum.h"
#include "matrix4.h"
#include "quaternion.h"
#include "vector3.h"
#include "smartptr.h"
#include "include.h"

class CScene;
typedef CSmartPtr<CScene> pScene;

class CEntity;
typedef CSmartPtr<CEntity> pEntity;

typedef enum {
	SIGNAL_ATTACH,
	SIGNAL_DETACH
} signal_t;

// Classe reprsentant une entit
class CEntity : virtual public IRefCount, public IScriptable
{
public:	
	CEntity(const std::string &filename="");	// Constructeur depuis fichier de script
	virtual ~CEntity(void);						// Destructeur
	
	void Load(std::istream &in);
	
	// Gestion de la hirarchie
	void	Attach(CEntity *parent);		// Relie hirarchiquement  un parent
	void	Detach(void);					// Detache du parent
	void	AddChild(pEntity ptr);			// Ajoute un enfant
	bool	RemoveChild(pEntity ptr);		// Enlve en enfant (LENT!)
	bool	RemoveBrother(pEntity ptr);		// Enlve un frre	(LENT!)
	pEntity getFather(void);				// Retourne l'entit parente
	pEntity getBrother(int num=0);			// Retourne une entit frre
	int		getBrothersCount(void);			// Retourne le nombre de frres
	pEntity getChild(int num=0);			// Retourne un enfant
	int		getChildsCount(void);			// Retourne le nombre d'enfants
	
	inline bool isHierarchyChanged(void) { return mHierarchyChanged; }

	// Gestion des proprits
	const std::string &getName(void)		{return mName;}
	void setName(const std::string &name)	{mName = name;}
	pEntity Call(const std::string &name);	// Appelle une entit du sous-arbre selon son nom
	
	// Gestion des transformations
	void Translate(const CVector3 &v);
	void Rotate(float x,float y,float z);
	void Rotate(const CQuaternion &q);
	void Scale(const CVector3 &s);

	CMatrix4 &getLocalMatrix(void);					// Accs  la matrice locale
	CMatrix4 getGlobalMatrix(void) const;			// Accs  la matrice globale
	
	void setTransformation(const CMatrix4 &matrix);
	void Transform(const CMatrix4 &matrix);
	void AbsoluteTransform(const CMatrix4 &matrix);
	void Identity(void);
	
	const CMatrix4 *getTextureMatrix(int unit=0) const;			// Accs une matrice de texture locale
	void setTextureMatrix(const CMatrix4 *matrix,int unit=0);
	
	void LockDirection(pEntity entity)	{mLockDirection = entity;}

	// Gestion de l'environnement
	void setEnv(GLenum name,bool state);	// Change l'environnement
	void setLife(float life);
	float getLife(void);
	
	// Gestion de l'animation
	void setAnimation(pAnimation animation);
	pAnimation getAnimation(void) const;
	void PlayAnimation(void);
	void StopAnimation(void);
	bool isAnimationPlaying(void);
	void setAnimationFrame(double start,double end = -1.);
	void setAnimationSpeed(double speed);
	float getAbsoluteAnimationSpeed(void) const;

	// Gestion de l'affichage
	int Display(int pass);		// Passage d'affichage 
	
	// Gestion des collisions
	bool Collision(const CCoord3 &pos,const CVector3 &move,float radius,CVector3 *result,CCoord3 *intersect=NULL);
	float Intersection(CCoord3 pos,CVector3 move,float radius,CCoord3 *intersection);
	
	// Fonctions de scriptage
	virtual pScriptable Tag(const std::string &tag);
	virtual void Parameter(const std::string &param,const std::string &data);
	
	// Routines standards principales des entits
	//virtual bool Signal(signal_t signal);			// Passage d'un signal
	virtual bool Update(double time);				// Mise  jour
	virtual int Draw(int pass);		// Dessin
	virtual float Intersect(const CCoord3 &pos,const CVector3 &move,float radius,CCoord3 *intersection);

protected:
	void setRadius(float radius);

	// Contrle d'animation
	pAnimation 	mAnimation;
	bool		mAnimPlaying;
	double		mAnimStart,mAnimEnd;
	double		mCurrentFrame,mAnimSpeed;

	bool		mHierarchyChanged;

private:
	// Arbre des entits
	pEntity mBrother;		// Petit frre de l'entit (retient l'arbre)
	pEntity mChild;			// Fils ain de l'entit (retient l'arbre)
	CEntity	*mFather;		// Parent de l'entit
	CEntity	*mBigBrother;	// Grand frre de l'entit
	
	// Proprits de l'entit
	std::string	mName;		// Nom
	double		mLife;		// Temps restant avant destruction si positif
	
	// Environnement de l'entit
	CMatrix4	mLocalMatrix;							// Matrice de transformation locale
	typedef std::map<int,CMatrix4> TextureMatrixMap_t;
	TextureMatrixMap_t mTextureMatrixes;				// Map des matrices de texture
	typedef std::map<GLenum,bool> EnvMap_t;				
	EnvMap_t mEnv;										// Environnement de dessin OpenGL
	pEntity		mLockDirection;							// Verrouillage de la direction si non NULL
	float		mInternalRadius;						// Bounding sphere propre de l'entit
	
	// Variables temporaires de dessin
	float	mExternalRadius;					// Bounding sphere avec enfants compris
	bool	mInternalVisible,mExternalVisible;	// Flags de visibilit

	friend class CScene;
};

#include "scene.h"

#endif	// ENTITY_H
