/***************************************************************************
 *   Copyright (C) 2006-2008 by Paul-Louis Ageneau                         *
 *   paullouisageneau@gmail.com                                            *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.           *
 ***************************************************************************/

#ifndef EXCEPTION_H
#define EXCEPTION_H

#include "include.h"
#include <exception>

// Classe de base pour les exceptions
class CException : public std::exception
{
public:

    // Constructeur
    CException(const std::string& Message = "");

    // Destructeur
    virtual ~CException() throw();

    // Renvoie le message associ  l'exception
    virtual const char* what() const throw();

protected:

    // Donnes membres
    std::string mMessage; // Message dcrivant l'exception
};

// Exception lance si une condition n'est pas vrifie
struct CAssertException : public CException
{
    CAssertException(const std::string &File, int Line, const std::string& Message);
};
#ifdef _DEBUG
#   define Assert(condition) if (!(condition)) throw CAssertException(__FILE__, __LINE__, "Condition non satisfaite : " #condition)
#else
    inline void DoNothing(bool) {}
#   define Assert(condition) DoNothing(!(condition))
#endif

// Exception lance lors d'erreur de chargement de fichiers
struct CLoadingFailed : public CException
{
    CLoadingFailed(const std::string& File, const std::string& Message);
};

// Exception lance lors de l'utilisation d'une caractristique non supporte
struct CUnsupported : public CException
{
    CUnsupported(const std::string& Feature);
};

// Exception lance lors de saturations de mmoire
struct COutOfMemory : public CException
{
    COutOfMemory(void);
};

// Exception lance lors d'une division par zro
struct CDivideByZero : public CException
{
	CDivideByZero(void);
};

// Exception lance  la suite d'une erreur Opengl
struct CGlError : public CException
{
	CGlError(void);
};

inline void AssertZero(int nbr) { if(!nbr) throw CDivideByZero(); }
inline void AssertZero(float nbr) { if(std::fabs(nbr) <= std::numeric_limits<float>::epsilon()) throw CDivideByZero(); }
inline void AssertZero(double nbr) { if(std::fabs(nbr) <= std::numeric_limits<double>::epsilon()) throw CDivideByZero(); }

template<typename T> T operator/(T x,T y)
{
	AssertZero(y);
	return x/y;
}

#endif // EXCEPTION_H
