/***************************************************************************
 *   Copyright (C) 2006-2008 by Paul-Louis Ageneau                         *
 *   paullouisageneau@gmail.com                                            *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.           *
 ***************************************************************************/

#include "material.h"
#include "mediamanager.h"

const pMaterial CMaterial::Default=new CMaterial;

// Constructeur par dfaut
CMaterial::CMaterial(void) : 
	mAmbient(.5f,.5f,.5f),
	mDiffuse(1.f,1.f,1.f),
	mSpecular(.5f,.5f,.5f),
	mEmissive(0.f,0.f,0.f),
	mShininess(16),
	mRgbScale(1)
{

}

// Constructeur spcifiant la texture
CMaterial::CMaterial(pTexture texture,float alpha) : 
	mAmbient(.5f,.5f,.5f),
	mDiffuse(1.f,1.f,1.f),
	mSpecular(.5f,.5f,.5f),
	mEmissive(0.f,0.f,0.f),
	mShininess(16),
	mRgbScale(1)
{
	setTexture(texture,0);
	setAlpha(alpha);
}

// Destructeur
CMaterial::~CMaterial(void)
{

}

pScriptable CMaterial::Tag(const std::string &tag)
{
	return NULL;
}

void CMaterial::Parameter(const std::string &param,const std::string &data)
{
	std::stringstream sdata(data);
	
	if(param == "texture0" || param == "texture") setTexture(MediaManager->Get<CTexture>(data),0);
	else if(param == "texture1") setTexture(MediaManager->Get<CTexture>(data),1);
	else if(param == "texture2") setTexture(MediaManager->Get<CTexture>(data),2);
	else if(param == "texture3") setTexture(MediaManager->Get<CTexture>(data),3);

	else if(param == "rgbscale") sdata>>mRgbScale;
	else if(param == "ambient") sdata>>mAmbient;
	else if(param == "diffuse") sdata>>mDiffuse;
	else if(param == "specular") sdata>>mSpecular;
	else if(param == "emissive") sdata>>mEmissive;
	else if(param == "alpha")	sdata>>mDiffuse.a;
	else if(param == "shine")	sdata>>mShininess;
}

void CMaterial::setRgbScale(GLuint scale)
{
	mRgbScale=scale;
}

void CMaterial::setTexture(pTexture texture,int level)
{
	if(level<0) return;
	
	if(texture!=NULL)
	{
		TextureHolder_t &holder=mTextures[level];
		holder.Texture=texture;
		holder.Mode=GL_MODULATE;
		holder.Source[0]=GL_PREVIOUS;
		holder.Source[1]=GL_TEXTURE;
		holder.Source[2]=GL_TEXTURE;
	}
	else {
		TexturesMap_t::iterator it=mTextures.find(level);
		if(it!=mTextures.end()) mTextures.erase(it);
	}
}

void CMaterial::setTextureMode(int level,GLuint mode,GLuint source0,GLuint source1,GLuint source2)
{
	if(level<0) return;
	
	TexturesMap_t::iterator it=mTextures.find(level);
	if(it!=mTextures.end())
	{
		it->second.Mode=mode;
		it->second.Source[0]=source0;
		it->second.Source[1]=source1;
		it->second.Source[2]=source2;
	}
}

bool CMaterial::hasBlending(void) const
{ 
	return (mDiffuse.a+EPSILON < 1.f);
}	

bool CMaterial::Bind(int pass,float alpha,bool repeat)
{
	// Activation du blending si ncessaire
	if (hasBlending() ||alpha+EPSILON < 1.f)
	{
		if(pass>0 && pass!=2) return false;
		glEnable(GL_BLEND);
		//glBlendFunc(GL_SRC_ALPHA, GL_ONE);				// blend additif
		glBlendFunc(GL_SRC_ALPHA,GL_ONE_MINUS_SRC_ALPHA);	// blend modulation
		glDepthMask(GL_FALSE);
	}
	else {
		if(pass>0 && pass!=1) return false;
		glDisable(GL_BLEND);
		glDepthMask(GL_TRUE);
	}

	glEnable(GL_ALPHA_TEST);
	glAlphaFunc(GL_GREATER,EPSILON);

	CColor diffuse=mDiffuse;
	diffuse.a*=alpha;
	glMaterialfv(GL_FRONT_AND_BACK, GL_DIFFUSE, diffuse);

	glMaterialfv(GL_FRONT_AND_BACK, GL_AMBIENT, mAmbient);
	glMaterialfv(GL_FRONT_AND_BACK, GL_SPECULAR, mSpecular);
	glMaterialfv(GL_FRONT_AND_BACK, GL_EMISSION, mEmissive);
	glMateriali(GL_FRONT_AND_BACK ,GL_SHININESS, mShininess);

	if(GLEW_ARB_multitexture)	// avec multitexturing
	{
		int max_textures;
		glGetIntegerv(GL_MAX_TEXTURE_UNITS,&max_textures);

		for(int unit=0; unit<max_textures; ++unit)
		{
			glActiveTextureARB(GL_TEXTURE0+unit);
			TexturesMap_t::const_iterator it=mTextures.find(unit);
			if(it!=mTextures.end()) 
			{
				if(repeat) it->second.Texture->Bind(GL_REPEAT);
				else it->second.Texture->Bind(GL_CLAMP);
				
				if(GLEW_ARB_texture_env_combine)	// avec combinaison des textures
				{
					glTexEnvi(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_COMBINE);
					glTexEnvi(GL_TEXTURE_ENV, GL_COMBINE_RGB, it->second.Mode);
					for(int i=0;i<3;++i) glTexEnvi(GL_TEXTURE_ENV, GL_SOURCE0_RGB+i, it->second.Source[i]);
				}
				else //sans combinaison des textures
				{
					glTexEnvi(GL_TEXTURE_ENV,GL_TEXTURE_ENV_MODE,GL_MODULATE);
				}
			}
		}

		if(GLEW_ARB_texture_env_combine) glTexEnvi(GL_TEXTURE_ENV,GL_RGB_SCALE, mRgbScale);
	}
	else //sans multitexturing
	{	
		TexturesMap_t::const_iterator it=mTextures.find(0);
		if(it!=mTextures.end()) it->second.Texture->Bind();
	}
	return true;
}

void CMaterial::Unbind()
{
	if(hasBlending())
	{
		glDisable(GL_BLEND);
		glDepthMask(GL_TRUE);
	}

	if(GLEW_ARB_multitexture)	// avec multitexturing
	{
		int max_textures;
		glGetIntegerv(GL_MAX_TEXTURE_UNITS,&max_textures);

		for(int unit=0; unit<max_textures; ++unit)
		{
			glActiveTextureARB(GL_TEXTURE0+unit);
			TexturesMap_t::const_iterator it=mTextures.find(unit);
			if(it!=mTextures.end()) it->second.Texture->Unbind();
		}
	}
	else {	//sans multitexturing
		
		TexturesMap_t::const_iterator it=mTextures.find(0);
		if(it==mTextures.end()) it->second.Texture->Unbind();		
	}

}
