/***************************************************************************
 *   Copyright (C) 2006-2008 by Paul-Louis Ageneau                         *
 *   paullouisageneau@gmail.com                                            *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.           *
 ***************************************************************************/

#include "mesh.h"
#include "bufferarray.h"
#include "bufferobject.h"
#include "vector3.h"

CMesh::CMesh(void)
{
	mNbrVertices = 0;
	mNbrIndices = 0;
	mNbrTexCoords = 0;
	mTexCoordBuffers = NULL;
	mFaceNeighbours=NULL;
}

CMesh::CMesh(	const float *vertices, 
				index_t nvertices,
				index_t *indices, 
				index_t nindices,
				const float **texcoords,
				int ntexcoords)
{	
	mNbrVertices = 0;
	mNbrIndices = 0;
	mNbrTexCoords = 0;
	mTexCoordBuffers = NULL;
	mFaceNeighbours=NULL;

	Add(vertices,nvertices,indices,nindices,texcoords,ntexcoords);
}

CMesh::~CMesh(void)
{
	delete[] mTexCoordBuffers;
	delete[] mFaceNeighbours;
}

void CMesh::Copy(pMesh mesh)
{
	if(mesh != NULL)
	{
		mIndexBuffer 	= mesh->mIndexBuffer;
		mVertexBuffer	= mesh->mVertexBuffer;
		mNormalBuffer	= mesh->mNormalBuffer;
		
		mNbrIndices		= mesh->mNbrIndices;
		mNbrVertices	= mesh->mNbrVertices;
		mNbrTexCoords	= mesh->mNbrTexCoords;

		mTexCoordBuffers = new CSmartPtr<TexCoordBuffer_t>[mNbrTexCoords];
		for(int i=0; i<mNbrTexCoords; ++i) 
				mTexCoordBuffers[i] = mesh->mTexCoordBuffers[i];

		mMaterials 	= mesh->mMaterials;
		mRadius		= mesh->mRadius;

		if(mesh->mFaceNeighbours)
		{
			mFaceNeighbours=new index_t[mNbrIndices];
			for(int i=0; i<mNbrIndices; ++i) 
				mFaceNeighbours[i] = mesh->mFaceNeighbours[i];
		}
		else mFaceNeighbours = NULL;
	}
	else {
		mNbrVertices = 0;
		mNbrIndices = 0;
		mNbrTexCoords = 0;
		mTexCoordBuffers = NULL;
		mFaceNeighbours=NULL;
	}
}

void CMesh::Add(	const float *vertices, 
					index_t nvertices,
					index_t *indices, 
					index_t nindices,
					const float **texcoords,
					int ntexcoords)
{
	index_t offset = mNbrVertices;
	mNbrVertices+=nvertices;
	mNbrIndices+=nindices;
	
	if(mIndexBuffer == NULL)	// si rien n'a t allou pour l'instant
	{
		mNbrTexCoords = ntexcoords;	// le premier appel fixe le nombre de textures
		mTexCoordBuffers = new CSmartPtr<TexCoordBuffer_t>[mNbrTexCoords];

		// Cration des Buffers
		if(glewIsSupported("GL_vertex_buffer_object"))	// Teste la disponibilit des VBOs
		{
			mIndexBuffer	= new IndexBuffer_t(new CIndexBufferObject);
			mVertexBuffer 	= new VertexBuffer_t(new CDataBufferObject);
			mNormalBuffer	= new NormalBuffer_t(new CDataBufferObject);
			for(int i=0; i<mNbrTexCoords; ++i) 
				mTexCoordBuffers[i] = new TexCoordBuffer_t(new CDataBufferObject);
		} 
		else {
			mIndexBuffer	= new IndexBuffer_t(new CBufferArray);
			mVertexBuffer	= new VertexBuffer_t(new CBufferArray);
			mNormalBuffer	= new NormalBuffer_t(new CBufferArray);
			for(int i=0; i<mNbrTexCoords; ++i)
				mTexCoordBuffers[i] = new TexCoordBuffer_t(new CBufferArray);
		}
	}

	// Cration et mise  0 du tableau des normales
	float *normals=new float[nvertices*3];
	std::fill(normals,normals+nvertices*3,0.f);

	// Calcul des normales
	for(index_t i=0; i<nindices; i+=3)
	{
		// Les 3 vertices de la face
		CCoord3 v1(vertices + indices[i]*3);
		CCoord3 v2(vertices + indices[i+1]*3);
		CCoord3 v3(vertices + indices[i+2]*3);

		// Produit vectoriel pour trouver une normale
		CVector3 normal=(v2-v1).Crosspoint(v3-v1);
		normal.Normalize();

		// Ajoute la normale de la face  celle de chaque vertice
		for(index_t j=i;j<i+3;++j)
		{
			normals[indices[j]*3]+=normal.x;
			normals[indices[j]*3+1]+=normal.y;
			normals[indices[j]*3+2]+=normal.z;
		}
	}

	mNormalBuffer->Add(normals,nvertices*3);		// 3 coordonnes par normale
	delete[] normals;

	// Dcalage des indices si ncessaire
	if(offset != 0)
		for(index_t i=0; i<nindices; ++i)
			indices[i]+=offset;

	// Remplissage des buffers
	mIndexBuffer->Add(indices,nindices);
	mVertexBuffer->Add(vertices,nvertices*3);		// 3 coordonnes de position par vertex
	
	for(int i=0; i<mNbrTexCoords; ++i)
		if(ntexcoords>0) mTexCoordBuffers[i]->Add(texcoords[std::min(ntexcoords-1,i)],nvertices*2);	// 2 coordonnes de texture par vertex
		else mTexCoordBuffers[i]->Add(NULL,nvertices*2);
	
	// Calcul du rayon
	float radius2=0.f;
	for(index_t i=0; i<nvertices*3; i+=3)
	{
		float d2=CCoord3(vertices[i],vertices[i+1],vertices[i+2]).Norm2();
		radius2=std::max(radius2,d2);
	}
	mRadius=std::sqrt(radius2);

	// Materiau par dfaut
	setMaterial(CMaterial::Default,0);
}

// Spcifie un matriau dans la srie d'indices (NULL pour supprimer)
void CMesh::setMaterial(pMaterial material, index_t offset)
{
	if(material!=NULL) mMaterials[offset]=material;
	else {
		MaterialsMap_t::iterator it=mMaterials.find(offset);
		if(it!=mMaterials.end()) mMaterials.erase(it);
	}
}

bool CMesh::hasBlending(void)
{
	for(MaterialsMap_t::iterator it=mMaterials.begin(); it!=mMaterials.end(); ++it)
		if(it->second->hasBlending()) return true;

	return false;
}

float CMesh::getRadius(void) const
{
	return mRadius;
}

void CMesh::EnableBuffers(void) const
{
	// Mise en place des pointeurs sur les lments
	glEnableClientState(GL_VERTEX_ARRAY);
	glVertexPointer(3,GL_FLOAT,0,mVertexBuffer->Bind());
	mVertexBuffer->Unbind();
	
	glEnableClientState(GL_NORMAL_ARRAY);
	glNormalPointer(GL_FLOAT,0,mNormalBuffer->Bind());
	mNormalBuffer->Unbind();
	
	if(GLEW_ARB_multitexture)
	{
		int max_textures;
		glGetIntegerv(GL_MAX_TEXTURE_UNITS,&max_textures);
		max_textures=std::min(max_textures,mNbrTexCoords);
		
		for(int i=0; i<max_textures; ++i)
		{
			glClientActiveTextureARB(GL_TEXTURE0+i);
			glEnableClientState(GL_TEXTURE_COORD_ARRAY);
			glTexCoordPointer(2,GL_FLOAT,0,mTexCoordBuffers[i]->Bind());
			mTexCoordBuffers[i]->Unbind();
		}
	} 
	else {
		glEnableClientState(GL_TEXTURE_COORD_ARRAY);
		glTexCoordPointer(2,GL_FLOAT,0,mTexCoordBuffers[0]->Bind());
		mTexCoordBuffers[0]->Unbind();
	}
}

void CMesh::DisableBuffers(void) const
{
	// Dsactivation des tableaux d'lments
	glDisableClientState(GL_VERTEX_ARRAY);
	glDisableClientState(GL_NORMAL_ARRAY);

	if(GLEW_ARB_multitexture)
	{
		int max_textures;
		glGetIntegerv(GL_MAX_TEXTURE_UNITS,&max_textures);
		max_textures=std::min(max_textures,mNbrTexCoords);
		
		for(int i=0; i<max_textures; ++i)
		{
			glClientActiveTextureARB(GL_TEXTURE0+i);
			glDisableClientState(GL_TEXTURE_COORD_ARRAY);
		}
	} 
	else glDisableClientState(GL_TEXTURE_COORD_ARRAY);

}

int CMesh::DrawElements(int pass,index_t pos, index_t nbr) const
{
	int count = 0;
	mIndexBuffer->Bind();
	
	// Indices pour chaque matriau
	// OPTI (begin=end en fin de boucle)
	for(MaterialsMap_t::const_iterator it=mMaterials.lower_bound(pos); it!=mMaterials.end(); ++it)
	{
		index_t begin,end;
		begin=it->first;

		MaterialsMap_t::const_iterator next_it=it;
		++next_it;
		if(next_it==mMaterials.end()) end=mIndexBuffer->getCount();
		else end=next_it->first;
		if(end>pos+nbr) end=pos+nbr;

		if(it->second->Bind(pass))	// Attache du matriau
		{
			// Dessine les lments  partir des indices
			glDrawElements(GL_TRIANGLES,end-begin,GL_UNSIGNED_INT,mIndexBuffer->Offset(begin));
			it->second->Unbind();
			count+=end-begin;
		}

		if(end==pos+nbr) break;
	}
	
	mIndexBuffer->Unbind();
	return count/3;
}

// Dessine la structure
int CMesh::Draw(int pass,double frame,pMaterial force)
{
	int count = 0;
	EnableBuffers();
	
	if(force != NULL)
	{
		if(force->Bind(pass))	// Attache du matriau
		{
			// Dessine les lments  partir des indices
			glDrawElements(GL_TRIANGLES,mIndexBuffer->getCount(),GL_UNSIGNED_INT,mIndexBuffer->Bind());
			mIndexBuffer->Unbind();
			force->Unbind();
			count = mNbrIndices/3;
		}
	}
	else count = DrawElements(pass,0,mNbrIndices);

	DisableBuffers();
	return count;
}

float CMesh::Intersect(double frame,const CCoord3 &pos,const CVector3 &move,float radius,CCoord3 *intersection)
{
	// cherche la premire face entrant en collision
	float nearestDistance = std::numeric_limits<float>::infinity();
	CCoord3 nearestIntersection;
	
	const float *vertices=mVertexBuffer->Lock(0,mVertexBuffer->getCount(),GL_READ_ONLY);
	const index_t *indices=mIndexBuffer->Lock(0,mIndexBuffer->getCount(),GL_READ_ONLY);
	
	for(index_t i=0; i<mNbrIndices; ++i)
	{
		CCoord3 v1(vertices + indices[i]*3);
		CCoord3 v2(vertices + indices[++i]*3);
		CCoord3 v3(vertices + indices[++i]*3);
		
		CCoord3 intersect;
		float t=pos.Intersect(move,radius,v1,v2,v3,&intersect);
		if(t<nearestDistance)
		{
			nearestDistance=t;
			nearestIntersection=intersect;
		}
	}

	mVertexBuffer->Unlock();
	mIndexBuffer->Unlock();
	
	if(intersection) *intersection = nearestIntersection;
	return nearestDistance;
}

int CMesh::DrawShadowVolume(double frame,const CVector3 &light,bool directional)
{
	if(!mFaceNeighbours) return 0;
	
	const float *vertices=mVertexBuffer->Lock(0,mVertexBuffer->getCount(),GL_READ_ONLY);
	const index_t *indices=mIndexBuffer->Lock(0,mIndexBuffer->getCount(),GL_READ_ONLY);
	
	bool *lighted =new bool[mNbrIndices];

	// Teste si chaque face est claire
	for(index_t i=0; i<mNbrIndices; ++i)
	{
		CCoord3 v1(vertices + indices[i]*3);
		CCoord3 v2(vertices + indices[++i]*3);
		CCoord3 v3(vertices + indices[++i]*3);

		CVector3 lightvector(light);
		if(!directional) lightvector=(v1+v2+v3)/3-light;
	
		// Produit scalaire avec une normale
		lighted[i/3]=(lightvector.Dotpoint((v2-v1).Crosspoint(v3-v1))<0);
	}

	int count=0;
	for(int pass=0; pass<2; ++pass)
	{
		if(pass==0)
		{
			glFrontFace(GL_CW);
			glStencilOp(GL_KEEP, GL_INCR, GL_KEEP);
		}
		else {
			glFrontFace(GL_CCW);
			glStencilOp(GL_KEEP, GL_DECR, GL_KEEP);
		}

		// Dessine le volume d'ombre
		for(index_t i=0; i<mNbrIndices; ++i)	// pour chaque face (i incrment 2 fois dans la boucle)
		{
			if(lighted[i/3])					// si claire
			{
				for(int j=0; j<3; ++j)			// pour chaque edge dans cette face
				{
					// Si l'edge appartient  la silhouette
					bool pass=true;
					index_t neighbour=mFaceNeighbours[i+j];
					if(neighbour!=INVALID_INDEX)
						if(lighted[neighbour/3]) pass=false;

					if(pass)
					{	
						CCoord3 v1(vertices+indices[i+j]*3);
						CCoord3 v2(vertices+indices[i+(j+1)%3]*3);

						CVector3 lightvector(light);

						glBegin(GL_TRIANGLE_STRIP);
							glVertex3fv(v1);
							if(!directional) lightvector=v1-light;
							glVertex3fv(v1+lightvector*INFINI);
							glVertex3fv(v2);
							if(!directional) lightvector=v2-light;
							glVertex3fv(v2+lightvector*INFINI);
						glEnd();
						count+=2;			
					}
				}

				CCoord3 v1(vertices + indices[i]*3);
				CCoord3 v2(vertices + indices[++i]*3);
				CCoord3 v3(vertices + indices[++i]*3);
				
				glBegin(GL_TRIANGLES);	
					// cap near
					glVertex3fv(v1);
					glVertex3fv(v2);
					glVertex3fv(v3);

					// cap far
					/*if(!directional)
					{
						glVertex3fv(v3+(v3-light)*INFINI);
						glVertex3fv(v2+(v2-light)*INFINI);
						glVertex3fv(v1+(v1-light)*INFINI);
					}*/
				glEnd();
			}
		}
	}

	delete[] lighted;
	mVertexBuffer->Unlock();
	mIndexBuffer->Unlock();

	return count/2;
}

void CMesh::EnableShadowVolume(void)
{
	if(mFaceNeighbours) return;
	
	const float *vertices=mVertexBuffer->Lock(0,mVertexBuffer->getCount(),GL_READ_ONLY);
	const index_t *indices=mIndexBuffer->Lock(0,mIndexBuffer->getCount(),GL_READ_ONLY);

	mFaceNeighbours=new index_t[mNbrIndices];
	for(index_t i=0; i<mNbrIndices; ++i) mFaceNeighbours[i]=INVALID_INDEX;

	// Recherche les faces voisines de chaque face
	for(index_t i1=0; i1<mNbrIndices; i1+=3)	// pour chaque face
		for(int j1=0; j1<3; ++j1)				// pour chaque edge dans cette face
			if(mFaceNeighbours[i1+j1]==INVALID_INDEX)
			{
				// Les 2 vertices de l'edge en cours
				CCoord3 v1a(vertices + indices[i1+j1]*3);
				CCoord3 v1b(vertices + indices[i1+(j1+1)%3]*3);
				
				for(index_t i2=0; i2<mNbrIndices; i2+=3)	// pour chaque autre face
				{
					if(i2==i1) continue;
					
					bool found=false;
					for(int j2=0; j2<3; ++j2)			// pour chaque edge dans cette face
					{
						// Les 2 vertices de l'edge en cours
						CCoord3 v2a(vertices + indices[i2+j2]*3);
						CCoord3 v2b(vertices + indices[i2+(j2+1)%3]*3);
						
						// Teste si les edges sont les mmes
						// REM:	tester les indices dans le tableau ne suffit pas
						//		car certains vertices ne diffrent que par leurs coords de textures
						if((v1a==v2a && v1b == v2b) || (v1a==v2b && v1b==v2a))
						{
							mFaceNeighbours[i1+j1]=i2;
							mFaceNeighbours[i2+j2]=i1;
							found=true;
							break;
						}
					}

					if(found) break;
				}
			}

	mVertexBuffer->Unlock();
	mIndexBuffer->Unlock();
}

void CMesh::DisableShadowVolume(void)
{
	delete[] mFaceNeighbours;
	mFaceNeighbours=NULL;
}
