/***************************************************************************
 *   Copyright (C) 2006-2008 by Paul-Louis Ageneau                         *
 *   paullouisageneau@gmail.com                                            *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.           *
 ***************************************************************************/

#include "scene.h"
#include "light.h"

pScene CScene::Current = NULL;

CScene::CScene(void) :
	mRoot(new CEntity),
	mfogColor(0.f,0.f,0.f),
	mfogNear(900.f),
	mfogFar(1000.f)
{
	
}

CScene::~CScene(void)
{

}

void CScene::setFogColor(const CColor &c)
{ 
	mfogColor=c; 
}

void CScene::setFogDistance(float n,float f)
{ 
	mfogNear=std::fabs(n);
	mfogFar=std::fabs(f);
}

void CScene::setBackground(pTexture front,pTexture back,pTexture right,pTexture left,pTexture top,pTexture bottom)
{
	mFront=front;
	mBack=back;
	mRight=right;
	mLeft=left;
	mTop=top;
	mBottom=bottom;
}

pEntity CScene::getRootEntity(void)
{ 
	return mRoot;
}

void CScene::Update(double time)
{
	CScene::Current = this;
	mRoot->Update(time);

	for(size_t i=0;i<mParticles.size();)
	{
		pParticle p=mParticles[i];

		p->mLife-=time;
		if(p->mLife<=0. || p->mEmitter==NULL) mParticles.erase(mParticles.begin()+i);
		else {
			// si la particule est active
			p->mPos+=p->mSpeed*time;
			p->mSpeed+=p->mEmitter->mGlobalFriction*time;
			if(p->mEmitter->mIsRotRandom) p->mAnglez+=rand()*p->mEmitter->mRotSpeed*time/RAND_MAX;
			else p->mAnglez+=p->mEmitter->mRotSpeed*time;
			++i;
		}	
	}

	CScene::Current = NULL;
}

void CScene::setClipPlane(const CPlane *plane,int unit)
{
	if(plane!=NULL) mClipPlanes[unit]=*plane;
	else {
		ClipPlaneMap_t::iterator it=mClipPlanes.find(unit);
		if(it!=mClipPlanes.end()) mClipPlanes.erase(it);
	}
}

int CScene::Render(void)
{
	Current = this;
	
	int count=0;
	ILight::Reset();			// efface les lumires de la frame prcdente

	// Initialise la matrice des modles
	CMatrix4 modl;
	modl.Get(GL_MODELVIEW_MATRIX);
	CCoord3 campos = modl.Inverse().getTranslation();
	
	// Active les plans de clipping
	int max_planes;
	glGetIntegerv(GL_MAX_CLIP_PLANES,&max_planes);
	for(int i=0; i<max_planes; ++i)
	{
		ClipPlaneMap_t::const_iterator it=mClipPlanes.find(i);
		if(it!=mClipPlanes.end())
		{
			glEnable(GL_CLIP_PLANE0+i);
			double equation[4]={it->second.a,it->second.b,it->second.c,it->second.d};
			glClipPlane(GL_CLIP_PLANE0+i,equation);
		}
	}

	glFogi(GL_FOG_MODE, GL_LINEAR);			// Brouillard lineaire
	glFogfv(GL_FOG_COLOR, mfogColor);		// couleur du brouillard
	glHint(GL_FOG_HINT, GL_DONT_CARE);		// laisse opengl choisir
	glFogf(GL_FOG_START, mfogNear);			// dbut
	glFogf(GL_FOG_END, mfogFar);			// fin

	// Extrait le frustum courant
	pFrustum frustum=new CFrustum;
	frustum->Extract();

	// Active le test de profondeur
	glEnable(GL_DEPTH_TEST);
	glDepthFunc(GL_LEQUAL);
	glDepthMask(GL_TRUE);

	// Rglages de polygones
	glPolygonMode(GL_FRONT_AND_BACK, GL_FILL);
	glShadeModel(GL_SMOOTH);		
	glCullFace(GL_BACK);

	// Matriaux
	glDisable(GL_COLOR_MATERIAL);
	glColor4f(1.f,1.f,1.f,1.f);
	
	// Par dfaut
	glEnable(GL_LIGHTING);
	glEnable(GL_FOG);
	glEnable(GL_CULL_FACE);

	count+=mRoot->Display(0);	// Passage des lumires + frustum culling
	count+=mRoot->Display(1);	// Passage principal
	count+=mRoot->Display(2);	// Passage de la transparence
	
	// Les ombres portes
	count+=DrawShadows();
	// Les particles
	count+=DrawParticles(frustum);
	
	// Dsactive les plans de clipping
	for(int i=0; i<max_planes; ++i)
	{
		ClipPlaneMap_t::const_iterator it=mClipPlanes.find(i);
		if(it!=mClipPlanes.end()) glDisable(GL_CLIP_PLANE0+i);
	}

	Current = NULL;
	return count;
}

int CScene::DrawParticles(pFrustum frustum)
{
	glPushAttrib(GL_ENABLE_BIT | GL_POLYGON_BIT);
	glDisable(GL_NORMALIZE);

	// Par dfaut
	glEnable(GL_LIGHTING);
	glEnable(GL_FOG);
	glDisable(GL_CULL_FACE);
	
	glMatrixMode(GL_MODELVIEW);
	CMatrix4 modl;
	modl.Get(GL_MODELVIEW_MATRIX);
	CCoord3 campos=modl.Inverse().getTranslation();
	
	typedef std::multimap<float,pParticle> ParticlesMap_t;
	ParticlesMap_t sorted;

	for(size_t i=0;i<mParticles.size();++i)
		if(frustum->PointInFrustum(mParticles[i]->mPos))
			sorted.insert(std::pair<float,pParticle>(mParticles[i]->mPos.Distance2(campos),mParticles[i]));
	
	for(ParticlesMap_t::reverse_iterator it=sorted.rbegin();it!=sorted.rend();++it)
	{
		pParticle particle=it->second;

		// Applique du matriau
		if(particle->mMaterial==NULL) particle->mMaterial=CMaterial::Default;
		particle->mMaterial->Bind(-1,particle->mLife/particle->mStartLife);
		
		glPushMatrix();
		
		// OPTI
		// Construit le repre local
		CVector3 up(0.f,1.f,0.f);
		CVector3 front = campos-particle->mPos;
		front.Normalize();
		CVector3 right = up.Crosspoint(front);
		right.Normalize();
		up = right.Crosspoint(front);
		
		right*=particle->mSizex/2.f;
		up*=particle->mSizey/2.f;

		// Matrice locale
		CMatrix4 modl;
		
		modl(0,0)=right.x;
		modl(1,0)=right.y;
		modl(2,0)=right.z;

		modl(0,1)=up.x;
		modl(1,1)=up.y;
		modl(2,1)=up.z;

		modl(0,2)=front.x;
		modl(1,2)=front.y;
		modl(2,2)=front.z;

		modl(0,3)=particle->mPos.x;
		modl(1,3)=particle->mPos.y;
		modl(2,3)=particle->mPos.z;

		modl.Apply();
		glRotatef(particle->mAnglez/PIDIV180,0.f,0.f,1.f);

		// Dessine le quad
		glBegin(GL_QUADS);
			glNormal3f(0.0f,0.0f,1.0f);
			glTexCoord2f(0.0f,0.0f);
			glVertex2f(-1.0f,-1.0f);
			glTexCoord2f(1.0f,0.0f);
			glVertex2f(1.0f,-1.0f);
			glTexCoord2f(1.0f,1.0f);
			glVertex2f(1.0f,1.0f);
			glTexCoord2f(0.0f,1.0f);
			glVertex2f(-1.0f,1.0f);	
		glEnd();

		glPopMatrix();

		particle->mMaterial->Unbind();
	}

	glPopAttrib();
	return mParticles.size()*2;
}

int CScene::DrawShadows(void)
{
	if(mShadowLights.empty() || mShadowCasters.empty()) return 0;
	
	glPushAttrib( GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT | GL_ENABLE_BIT | GL_POLYGON_BIT | GL_STENCIL_BUFFER_BIT );
	
	int count=0;
	for(size_t i=0; i<mShadowLights.size(); ++i)
	{
		glDisable(GL_LIGHTING);
		glEnable(GL_CULL_FACE);
		
		glColorMask(GL_FALSE, GL_FALSE, GL_FALSE, GL_FALSE);	// Pas d'criture dans le framebuffer
		
		// Configuration du test de profondeur
		glEnable(GL_DEPTH_TEST);
		glDepthFunc(GL_LEQUAL);
		glDepthMask(GL_FALSE);	// pas d'criture
		
		// Configuration du test de stencil
		glEnable(GL_STENCIL_TEST);
		glStencilFunc(GL_ALWAYS, 0x0, 0xFF);
		
		// Dcalge des polygones pour viter perturbations avec le test de profondeur
		glEnable(GL_POLYGON_OFFSET_FILL);
		glPolygonOffset(0.0f, 10.0f);
		
		// Dessine les volumes d'ombre
		CVector3 light;
		if(mShadowLights[i]->isDirectional()) light=mShadowLights[i]->getGlobalMatrix().getAxisZ();
		else light=mShadowLights[i]->getGlobalMatrix().getTranslation();
		for(size_t j=0; j<mShadowCasters.size(); ++j)
			count+=mShadowCasters[j]->DrawShadowVolume(light,mShadowLights[i]->isDirectional());

		// Redessine un quad sur tout l'cran o le stencil a t modifi
		glColorMask( GL_TRUE, GL_TRUE, GL_TRUE, GL_TRUE );

		glDisable(GL_CULL_FACE);
		glDisable(GL_DEPTH_TEST);
		glDisable(GL_POLYGON_OFFSET_FILL);
		
		glEnable(GL_BLEND);
		glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
		glColor4fv(mShadowLights[i]->mShadowColor);
		
		glStencilFunc(GL_NOTEQUAL, 0, 0xFF);
		glStencilOp(GL_KEEP, GL_KEEP, GL_KEEP);

		glMatrixMode(GL_PROJECTION);
		glPushMatrix();
		glLoadIdentity();
		glOrtho(-1.f,1.f,-1.f,1.f,-1.f,1.f);
		
		glMatrixMode(GL_MODELVIEW);	
		glPushMatrix();
		glLoadIdentity();

		glBegin(GL_QUADS);
			glVertex3f(-1.0f, 1.0f, 0.0f);	
			glVertex3f( 1.0f, 1.0f, 0.0f);	
			glVertex3f( 1.0f,-1.0f, 0.0f);	
			glVertex3f(-1.0f,-1.0f, 0.0f);
		glEnd();							
		
		glMatrixMode(GL_PROJECTION);
		glPopMatrix();
		glMatrixMode(GL_MODELVIEW);	
		glPopMatrix();
	
	}
	
	glPopAttrib();
	mShadowLights.clear();
	mShadowCasters.clear();
	return count;
}

int CScene::RenderBackground(void)
{
	int count=0;

	CMatrix4 modl;
	modl.Get(GL_MODELVIEW_MATRIX);
	CCoord3 campos=modl.Inverse().getTranslation();
	float near = 100.f;
	
	glDisable(GL_LIGHTING);
	glDisable(GL_FOG);
	glDisable(GL_NORMALIZE);
	glDisable(GL_DEPTH_TEST);
	glDepthMask(GL_FALSE);
	glDisable(GL_BLEND);
	glDisable(GL_CULL_FACE);
	glColor3f(1.f,1.f,1.f);

	pMaterial material=new CMaterial;
	material->Bind();
	if(GLEW_ARB_multitexture) glActiveTextureARB(GL_TEXTURE0);

	float d=near/SQRT3;
	float x = campos.x-d;
	float y = campos.y-d;
	float z = campos.z-d;
	d*=2;

	if(mFront!=NULL)
	{
		mFront->Bind(GL_CLAMP_TO_EDGE);
		glBegin(GL_TRIANGLE_STRIP);					
		glTexCoord2f(0.f, 0.f); glVertex3f(x,y,z);
		glTexCoord2f(1.f, 0.f); glVertex3f(x+d, y,z);
		glTexCoord2f(0.f, 1.f); glVertex3f(x,y+d, z);
		glTexCoord2f(1.f, 1.f); glVertex3f(x+d, y+d, z); 
		glEnd();
		mBack->Unbind();
		++count;
	}

	if(mBack!=NULL)
	{
		mBack->Bind(GL_CLAMP_TO_EDGE);
		glBegin(GL_TRIANGLE_STRIP);	
		glTexCoord2f(1.f, 0.f); glVertex3f(x,y,z+d);
		glTexCoord2f(1.f, 1.f); glVertex3f(x,y+d, z+d);
		glTexCoord2f(0.f, 0.f); glVertex3f(x+d, y, z+d);
		glTexCoord2f(0.f, 1.f); glVertex3f(x+d, y+d, z+d); 	
		glEnd();
		mFront->Unbind();
		++count;
	}

	if(mLeft!=NULL)
	{
		mLeft->Bind(GL_CLAMP_TO_EDGE);	
		glBegin(GL_TRIANGLE_STRIP);				
		glTexCoord2f(1.f, 0.f); glVertex3f(x, y, z);
		glTexCoord2f(1.f, 1.f); glVertex3f(x, y+d, z);
		glTexCoord2f(0.f, 0.f); glVertex3f(x, y, z+d);
		glTexCoord2f(0.f, 1.f); glVertex3f(x, y+d, z+d); 
		glEnd();
		mRight->Unbind();
		++count;
	}

	if(mRight!=NULL)
	{
		mRight->Bind(GL_CLAMP_TO_EDGE);
		glBegin(GL_TRIANGLE_STRIP);				
		glTexCoord2f(0.f, 0.f); glVertex3f(x+d, y, z);
		glTexCoord2f(1.f, 0.f); glVertex3f(x+d, y, z+d);
		glTexCoord2f(0.f, 1.f); glVertex3f(x+d, y+d, z);
		glTexCoord2f(1.f, 1.f); glVertex3f(x+d, y+d, z+d); 
		glEnd();
		mLeft->Unbind();
		++count;
	}

	if(mTop!=NULL)
	{
		mTop->Bind(GL_CLAMP_TO_EDGE);
		glBegin(GL_TRIANGLE_STRIP);					
		glTexCoord2f(1.f, 1.f); glVertex3f(x,y+d,z);
		glTexCoord2f(0.f, 1.f); glVertex3f(x+d, y+d, z);
		glTexCoord2f(1.f, 0.f); glVertex3f(x, y+d, z+d);
		glTexCoord2f(0.f, 0.f); glVertex3f(x+d, y+d, z+d); 	
		glEnd();
		mTop->Unbind();
		++count;
	}
	
	if(mBottom!=NULL)
	{
		mBottom->Bind(GL_CLAMP_TO_EDGE);
		glBegin(GL_TRIANGLE_STRIP);				
		glTexCoord2f(1.f, 0.f); glVertex3f(x,y,z);
		glTexCoord2f(1.f, 1.f); glVertex3f(x,y,	z+d);
		glTexCoord2f(0.f, 0.f); glVertex3f(x+d, y,z);
		glTexCoord2f(0.f, 1.f); glVertex3f(x+d, y,z+d); 	
		glEnd();
		mBottom->Unbind();
		++count;
	}

	material->Unbind();
	return count*2;
}

void CScene::AddParticle(pParticle particle)
{
	mParticles.push_back(particle);
}

void CScene::RegisterShadowLight(pLight light)
{
	mShadowLights.push_back(light);
}

void CScene::RegisterShadowCaster(pObject object)
{
	mShadowCasters.push_back(object);
}
