/***************************************************************************
 *   Copyright (C) 2006-2008 by Paul-Louis Ageneau                         *
 *   paullouisageneau@gmail.com                                            *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.           *
 ***************************************************************************/

#ifndef SMARTPTR_H
#define SMARTPTR_H

#include "include.h"

// Interface IRefCount: objet pouvant tre point par un SmartPtr
class IRefCount
{
private:
	int ref_count;

public:
	
	IRefCount () : ref_count(0)
	{ }
	
	virtual ~IRefCount ()
	{
		if(ref_count!=0)
			std::cout<<"Attention: Destruction de "<<this<<" avec "<<ref_count<<" rfrences"<<std::endl;
	}
 
	inline void addRef(void)
	{
		++ref_count;
	}
 

	inline void releaseRef(void)
	{
		--ref_count;
	}

	inline int getRefCount() const
	{
		return ref_count;
	}
};

// Classe SmartPtr: Pointeur intelligent
template<class T> class CSmartPtr 
{

public:

	inline CSmartPtr(void) : ptr(NULL) {}

	inline CSmartPtr(T *n) : ptr(NULL)
	{
		(*this) = n;
	}
	
	inline CSmartPtr(const CSmartPtr &sp) : ptr(NULL)
	{
		(*this) = sp;
	}

	template<class T2> inline CSmartPtr(const CSmartPtr<T2> &sp) : ptr(NULL)
	{
		(*this) = static_cast<T*>(sp.getPtr());
	}

	inline ~CSmartPtr(void)
	{
		release();
	}
	
	inline bool is_null(void) const
	{
		return ptr==NULL;
	}

	inline T *getPtr(void) const
	{
		return ptr;
	}
	
	inline CSmartPtr &operator = (T *n)
	{
		if(n!=ptr) acquire(n);
		return (*this);
	}

	inline CSmartPtr &operator = (const CSmartPtr &sp)
	{
		(*this) = sp.ptr;			// on lance l'operateur = sur l'objet
		return (*this);
	}

	inline operator T*() const
	{
		return ptr;
	}
	
	inline T * operator -> () const
	{
		//if(ptr==NULL) throw "Tentative d'utilisation d'un pointeur nul";
		return ptr;
	}

	inline T & operator * () const
	{
		//if(ptr==NULL) throw "Tentative d'utilisation d'un pointeur nul";
		return *ptr;
	}

protected:
	T *ptr; // l'objet point
	
private:	
	// relache l'objet point
	void release(void)
	{
		if(ptr==NULL) return;
		ptr->releaseRef();			// on enleve une reference au compteur de l'objet
		if(ptr->getRefCount()==0)	// si plus aucun pointeur ne referencie l'objet
			delete ptr;				// on le delete
		ptr = NULL;
	}

	void acquire(T *n)
	{
		if(ptr!=NULL) release();	// on relache le pointeur courant
		ptr = n;
		if(n!=NULL) ptr->addRef();	// on ajoute une reference sur le compteur de l'objet
	}

};

#endif	// SMARTPTR_H
