/***************************************************************************
 *   Copyright (C) 2006-2008 by Paul-Louis Ageneau                         *
 *   paullouisageneau@gmail.com                                            *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.           *
 ***************************************************************************/

#include "terrain.h"
#include "mediamanager.h"
#include "bufferarray.h"
#include "bufferobject.h"
#include "exception.h"
#include <IL/il.h>

CTerrain::CTerrain(	float sizex,
					float sizey,
					int nbrx,
					int nbry,
					int detailx,
					int detaily,
					pMaterial material,
					const float *heightmap)
{
	mNbrx=nbrx+1;
	mNbry=nbry+1;
	mScalex=sizex/nbrx;
	mScaley=sizey/nbry;
	
	// Coordonnes de vertices
	float *vertices=new float[mNbrx*mNbry*3];
	for(int y=0; y<mNbry; ++y)
		for(int x=0; x<mNbrx; ++x)
		{
			int offset=(x+y*mNbrx)*3;
			vertices[offset]=x*mScalex;									// Coordonne x
			if(heightmap) vertices[offset+1]=heightmap[offset];			// Coordonne y (=altitude)
			else vertices[offset+1]=0.f;
			vertices[offset+2]=y*mScaley;								// Coordonne z
		}

	float *texcoords[2];

	// Coordonnes de texture niveau 0
	texcoords[0]=new float[mNbrx*mNbry*2];
	for(int y=0; y<mNbry; ++y)
		for(int x=0; x<mNbrx; ++x)
		{
			texcoords[0][(x+y*mNbrx)*2]=float(x)/mNbrx;		// Coordonne u
			texcoords[0][(x+y*mNbrx)*2+1]=float(y)/mNbry;	// Coordonne v
		}

	// Coordonnes de texture niveau 1 (Texture de dtail)
	texcoords[1]=new float[mNbrx*mNbry*2];
	for(int y=0; y<mNbry; ++y)
		for(int x=0; x<mNbrx; ++x)
		{
			texcoords[1][(x+y*mNbrx)*2]=float(x)/detailx;		// Coordonne u
			texcoords[1][(x+y*mNbrx)*2+1]=float(y)/detaily;		// Coordonne v
		}
	
	// Indices
	index_t *indices=new index_t[nbrx*nbry*6];
	index_t *ptr=indices;
	for(int y=0; y<nbrx; ++y)
		for(int x=0; x<nbry; ++x)
		{
			//for(int i=0;i<6;++i) *(ptr++)=0;
			*(ptr++) = x+y*mNbrx;
			*(ptr++) = x+(y+1)*mNbrx;
			*(ptr++) = x+1+y*mNbrx;
			*(ptr++) = x+1+y*mNbrx;
			*(ptr++) = x+(y+1)*mNbrx;
			*(ptr++) = x+1+(y+1)*mNbrx;
		}

	Add(vertices,mNbrx*mNbry,indices,nbrx*nbry*6,(const float**)texcoords,2);

	delete[] vertices;
	delete[] texcoords[0];
	delete[] texcoords[1];
	delete[] indices;

	if(material!=NULL) setMaterial(material,0);
	Build();
}

CTerrain::~CTerrain()
{
	
}

void CTerrain::setHeight(int x,int y,float height)
{
	*mVertexBuffer->Lock((x+y*mNbrx)*3+1,1,GL_WRITE_ONLY)=height;
	mVertexBuffer->Unlock();
}

void CTerrain::setHeight(const float *heightmap,float scale,int cols,int rows)
{
	float *vertices=mVertexBuffer->Lock(0,mVertexBuffer->getCount(),GL_WRITE_ONLY);
	
	if(cols<0) cols = mNbrx;
	if(rows<0) rows = mNbry;

	for(int y=0;y<mNbry;++y)
		for(int x=0;x<mNbrx;++x)
			vertices[(x+y*mNbrx)*3+1]=heightmap[(y*cols/mNbrx)*rows+x*rows/mNbry]*scale;

	mVertexBuffer->Unlock();

	Update();	// met l'octree  jour
}

void CTerrain::setHeight(const std::string &heightmap,float scale)
{
	ILuint imageNbr=0;
	ilGenImages(1,&imageNbr);
	ilBindImage(imageNbr);

	if(!ilLoadImage(const_cast<ILstring>(CMediaManager::Instance()->FindMedia(heightmap).c_str()))) 
		throw CLoadingFailed(heightmap,"Impossible de charger la heightmap");
	if(!ilConvertImage(IL_LUMINANCE,IL_FLOAT)) 
		throw CLoadingFailed(heightmap,"Impossible de convertir la heightmap");

	int cols=ilGetInteger(IL_IMAGE_WIDTH);
	int rows=ilGetInteger(IL_IMAGE_HEIGHT);
	const float *data=reinterpret_cast<const float *>(ilGetData());
	
	setHeight(data,scale,cols,rows);

	ilDeleteImages(1,&imageNbr);
}

float CTerrain::getHeight(int x,int y)
{
	float height=*mVertexBuffer->Lock((x+y*mNbrx)*3+1,1,GL_READ_ONLY);
	mVertexBuffer->Unlock();
	return height;
}

CVector3 CTerrain::getNormal(int x,int y)
{
	if(x<=0 || x>=mNbrx-1 || y<=0 || y>=mNbry-1) 
		return CVector3(0.f,1.f,0.f);
	
	float h=getHeight(x,y);
	CVector3 a(mScalex,getHeight(x+1,y)-h,0.0f);
	CVector3 b(0.0f,getHeight(x,y-1)-h,-mScaley);
	CVector3 c(-mScalex,getHeight(x-1,y)-h,0.0f);
	CVector3 d(0.0f,getHeight(x,y+1)-h,mScaley);

	CVector3 normal=(a.Crosspoint(b)+c.Crosspoint(d));
	normal.Normalize();
	return normal;
}

CVector3 CTerrain::getGroundNormal(float x,float y)
{
	if(x<0.f || x>mNbrx*mScalex || y<0.f || y>mNbry*mScaley) 
		return CVector3(0.0f,1.0f,0.0f);
	
	int x0=int(x/mScalex);
	int y0=int(y/mScaley);
	float h=getHeight(x0,y0);
	CVector3 a(mScalex, getHeight(x0+1,y0+1)-h, mScaley);
	CVector3 n;
	if(x-x0<=y-y0) {
		CVector3 b(mScalex, getHeight(x0+1,y0)-h, 0.f);
		n=a.Crosspoint(b);
	} else {
		CVector3 b(0.f, getHeight(x0,y0+1)-h, mScaley);
		n=b.Crosspoint(a);
	}
	return n.Normalize();
}

float CTerrain::getGroundHeight(float x,float y)
{
	if(x<0.f || x>mNbrx*mScalex || y<0.f || y>mNbry*mScaley) return 0.f;
	
	int x0=int(x/mScalex);
	int y0=int(y/mScaley);
	float h=getHeight(x0,y0);

	CVector3 n=getGroundNormal(x,y);

	// Rsout l'quation du plan pour trouver la hauteur
	return (h-(n.x*(x-x0*mScalex)+n.z*(y-y0*mScaley))/n.y);
}

void CTerrain::updateNormals(void)
{
	CVector3 *normals=reinterpret_cast<CVector3*>(mNormalBuffer->Lock(0,mNormalBuffer->getCount(),GL_WRITE_ONLY));

	for(int y=0; y<mNbry; ++y)
		for(int x=0; x<mNbrx; ++x)
			normals[x+y*mNbrx]=getNormal(x,y);

	mNormalBuffer->Unlock();
}
