/***************************************************************************
 *   Copyright (C) 2006-2008 by Paul-Louis Ageneau                         *
 *   paullouisageneau@gmail.com                                            *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.           *
 ***************************************************************************/

#include "vector3.h"

// ** Sous-routines de collision **

// TODO: Passer en CPlane

// intersection avec un plan d'un rayon projet  partir du point
float CVector3::IntersectPlane(const CVector3 &direction,const CCoord3 &planeOrigin,const CVector3 &planeNormal) const
{
	/*
	n.p = D
	n.p(t) = D
	n.(s+d*t) = D	// s=source d=direction
	(n.s) + (n.d)*t = D
	*/

	float d=-(planeNormal.Dotpoint(planeOrigin));
	return -(planeNormal.Dotpoint(*this)+d)/planeNormal.Dotpoint(direction);
}

float CVector3::IntersectSphere(CVector3 direction,const CCoord3 &centre,float radius) const
{
	float len=direction.Norm();
	direction.Normalize();

	CVector3 dst = (*this) - centre;
	float b = dst.Dotpoint(direction);
	float c = dst.Norm2() - radius*radius;
	float d = b*b - c;

	if(d<0.f) return std::numeric_limits<float>::infinity();
	float t = -b-std::sqrt(d);
	if(t<0.f) return std::numeric_limits<float>::infinity();
	else return t/len;
}

bool CVector3::isInTriangle(const CCoord3 &p1,const CCoord3 &p2, const CCoord3 &p3) const
{
	CVector3 v1=(*this)-p1;
	CVector3 v2=(*this)-p2;
	CVector3 v3=(*this)-p3;
  
	v1.Normalize();
	v2.Normalize();
	v3.Normalize();

	float total=acos(v1.Dotpoint(v2));
	total+=acos(v2.Dotpoint(v3));
	total+=acos(v3.Dotpoint(v1)); 

	if(fabs(total-2*PI)<=EPSILON) return true;
	else return false;
}

CCoord3 CVector3::ClosestPointOnTriangle(const CCoord3 &p1,const CCoord3 &p2, const CCoord3 &p3) const
{
	CCoord3 nearest[3];
	nearest[0]=ClosestPointOnSegment(p1,p2);
	nearest[1]=ClosestPointOnSegment(p2,p3);
	nearest[2]=ClosestPointOnSegment(p3,p1);

	float dist[3];
	for(int i=0;i<3;++i) dist[i]=Distance(nearest[i]);

	if(dist[0]<dist[1])
	{
		if(dist[2]<dist[0]) return nearest[2];
		else return nearest[0];
	}
	else {
		if(dist[2]<dist[1]) return nearest[2];
		else return nearest[1];
	}
}

CCoord3 CVector3::ClosestPointOnSegment(const CCoord3 &p1,const CCoord3 &p2) const
{
	CVector3 c=*this-p1;
	CVector3 v=(p2-p1).Normalize();
	float d=p1.Distance(p2);
	float t=v.Dotpoint(c);

	if(t<0.f) return p1;
	if(t>d) return p2;

	v.Normalize();
	v*=t;
	return p1+v;
}

// avec une boite
float CVector3::IntersectBox(const CVector3 &move,float radius,const CCoord3 &min,const CCoord3 &max,CCoord3 *intersection) const
{
	float t[6];
	CCoord3 inter[6];
	
	// OPTI: calcul de l'intersection mme si NULL spcifi !
	// OPTI: sur un cube, il ne peut y avoir que 2 intersection, donc une seule  l'endroit
	t[1]=Intersect(move,radius,CCoord3(min.x,min.y,min.z),CCoord3(max.x,min.y,min.z),CCoord3(max.x,max.y,min.z),CCoord3(min.x,max.y,min.z),&inter[0]);
	t[2]=Intersect(move,radius,CCoord3(min.x,min.y,max.z),CCoord3(max.x,min.y,max.z),CCoord3(max.x,max.y,max.z),CCoord3(min.x,max.y,max.z),&inter[1]);
	t[3]=Intersect(move,radius,CCoord3(min.x,min.y,min.z),CCoord3(min.x,max.y,max.z),CCoord3(min.x,max.y,max.z),CCoord3(min.x,min.y,max.z),&inter[2]);
	t[4]=Intersect(move,radius,CCoord3(max.x,min.y,min.z),CCoord3(max.x,max.y,min.z),CCoord3(max.x,max.y,max.z),CCoord3(max.x,min.y,max.z),&inter[3]);
	t[5]=Intersect(move,radius,CCoord3(min.x,min.y,min.z),CCoord3(max.x,min.y,min.z),CCoord3(max.x,min.y,max.z),CCoord3(min.x,min.y,max.z),&inter[4]);
	t[6]=Intersect(move,radius,CCoord3(min.x,max.y,min.z),CCoord3(max.x,max.y,min.z),CCoord3(max.x,max.y,max.z),CCoord3(min.x,max.y,max.z),&inter[5]);
	
	float tmin = std::numeric_limits<float>::infinity();
	for(int i=0; i<6; ++i)
			if(t[i]<=tmin) {
				tmin=t[i];
				if(intersection) *intersection=inter[i];
			}

	return tmin;
}

// avec un rectangle
float CVector3::Intersect(const CVector3 &move,float radius,const CCoord3 &p1,const CCoord3 &p2,const CCoord3 &p3,const CCoord3 &p4,CCoord3 *intersection) const
{
	CCoord3 inter1,inter2;
	float t1=Intersect(move,radius,p1,p2,p3,&inter1);
	float t2=Intersect(move,radius,p1,p3,p4,&inter2);

	if(t1<t2)
	{
		if(intersection) *intersection=inter1;
		return t1;
	}
	
	if(intersection) *intersection=inter2;
	return t2;
}

// avec un triangle
float CVector3::Intersect(const CVector3 &move,float radius,const CCoord3 &p1,const CCoord3 &p2,const CCoord3 &p3,CCoord3 *intersection) const
{
	CVector3 normal((p1-p2).Crosspoint(p1-p3));		// normale au plan de la face
	normal.Normalize();

	// NB: p1 sert d'origine
	float planeDist = IntersectPlane(-normal,p1,normal);	// distance de la source au plan de la face
	if(planeDist < 0.f)  return std::numeric_limits<float>::infinity();

	CCoord3 planeIntersection;

	// si la sphre est entre dans le plan
	if(fabs(planeDist) <= radius) planeIntersection = (*this)-normal*planeDist;
	else
	{
		CCoord3 ellipsoidIntersection=(*this)-normal*radius;
		float t = ellipsoidIntersection.IntersectPlane(move,p1,normal);
		planeIntersection = ellipsoidIntersection+move*t;			// calcule l'intersection avec le plan
	}

	CCoord3 polygonIntersection;
	if(planeIntersection.isInTriangle(p1,p2,p3)) polygonIntersection = planeIntersection;
	else polygonIntersection = planeIntersection.ClosestPointOnTriangle(p1,p2,p3);
	
	if(intersection) *intersection=polygonIntersection;

	// contre-intersection avec la sphre
	float t = polygonIntersection.IntersectSphere(-move,(*this),radius);
	return t;
}
