/***************************************************************************
 *   Copyright (C) 2006-2008 by Paul-Louis Ageneau                         *
 *   paullouisageneau@gmail.com                                            *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.           *
 ***************************************************************************/


#ifndef VECTOR3_H
#define VECTOR3_H

#include "include.h"
#include "exception.h"

class CVector3;
typedef CVector3 CCoord3;
class CMatrix4;

template<typename T> T sqr(T x) { return x*x; }

// Classe Vector3
class CVector3
{

public:
	// Donnes membres
	float x,y,z;

	// Constrcuteur par dfaut
	inline CVector3(float _x=0.f,float _y=0.f,float _z=0.f) : x(_x),y(_y),z(_z)	{}
	// Constructeur par floats
	inline CVector3(const float *f) : x(f[0]),y(f[1]),z(f[2]) {}

	CVector3(const std::string &s);
	
	// Assignation
	inline void set(float _x,float _y,float _z)	{ x=_x; y=_y; z=_z; }
	inline void setNull(void)					{ x=0.f; y=0.f; z=0.f; }

	// Surcharge d'operateurs
	inline CVector3 &operator=(const float *f)				{ x=f[0]; y=f[1]; z=f[2]; return (*this); }
	inline CVector3 &operator+=(float f)					{ x+=f; y+=f; z+=f; return (*this); }
	inline CVector3 &operator-=(float f)					{ x-=f; y-=f; z-=f; return (*this); }
	inline CVector3 &operator*=(float f)					{ x*=f; y*=f; z*=f; return (*this); }
	inline CVector3 &operator/=(float f)					{ AssertZero(f); x/=f; y/=f; z/=f; return (*this); }
	inline bool operator!=(float f)							{ return x != f || y != f || z != f; }

	inline CVector3 operator-(void) const					{ CVector3 dest; dest.x=-x; dest.y=-y; dest.z=-z; return dest; };
	inline bool operator!=(const CVector3 &p) const		{ return x != p.x || y != p.y || z != p.z; }
	inline bool operator==(const CVector3 &p) const		{ return x == p.x && y == p.y && z == p.z; }
	//inline bool operator!=(const CVector3 &p) const		{ return std::fabs(p.x-x)>EPSILON || std::fabs(p.y-y)>EPSILON || std::fabs(p.z-z)>EPSILON; }
	//inline bool operator==(const CVector3 &p) const		{ return std::fabs(p.x-x)<=EPSILON && std::fabs(p.y-y)<=EPSILON && std::fabs(p.z-z)<=EPSILON; }*/
	
	inline CVector3 &operator+=(const CVector3 &p)			{ x+=p.x; y+=p.y; z+=p.z; return (*this); }
	inline CVector3 &operator-=(const CVector3 &p)			{ x-=p.x; y-=p.y; z-=p.z; return (*this); }
	inline CVector3 &operator*=(const CVector3 &p)			{ x*=p.x; y*=p.y; z*=p.z; return (*this); }
	inline CVector3 &operator/=(const CVector3 &p)			{	 AssertZero(p.x); AssertZero(p.y); AssertZero(p.z);
																x/=p.x; y/=p.y; z/=p.z; return (*this); }

	inline CVector3 operator*(float f) const		{ return CVector3(x*f,y*f,z*f); }
	inline CVector3 operator+(float f) const		{ return CVector3(x+f,y+f,z+f); }
	inline CVector3 operator-(float f) const		{ return CVector3(x-f,y-f,z-f); }
	inline CVector3 operator/(float f) const		{ return CVector3(x/f,y/f,z/f); }

	inline CVector3 operator+(const CVector3 &p) const		{ CVector3 dest = *this; dest+=p; return dest; }
	inline CVector3 operator-(const CVector3 &p) const		{ CVector3 dest = *this; dest-=p; return dest; }
	inline CVector3 operator*(const CVector3 &p) const		{ CVector3 dest = *this; dest*=p; return dest; }
	inline CVector3 operator/(const CVector3 &p) const		{ CVector3 dest = *this; dest/=p; return dest; }
	
	inline CVector3 abs(void) const	{ return CVector3(std::fabs(x),std::fabs(y),std::fabs(z)); }
	inline float min(void) const	{ return std::min(std::min(x,y),z); }
	inline float max(void) const	{ return std::max(std::max(x,y),z); }

	// Transformation matricielle
	inline CVector3	operator*(const CMatrix4 &m) const;
	inline CVector3 &operator*=(const CMatrix4 &m);
	
	// 3d maths
	CVector3 Crosspoint(const CVector3 &v) const;
	float Dotpoint(const CVector3 &v) const;
	float Norm(void) const;
	float Norm2(void) const;
	CVector3 &Normalize(void);
	float Distance(const CVector3 &p) const;
	float Distance2(const CVector3 &p) const;
	CVector3 Lerp(const CVector3 &v,float t) const;
	
	bool isInBox(const CCoord3 &p1,const CCoord3 &p2) const;

	// ** Sous-routines de collision **

	// sous-routines de collision ellipse/face
	float Intersect(const CVector3 &move,float radius,const CCoord3 &p1,const CCoord3 &p2,const CCoord3 &p3,CCoord3 *intersection) const;
	float Intersect(const CVector3 &move,float radius,const CCoord3 &p1,const CCoord3 &p2,const CCoord3 &p3,const CCoord3 &p4,CCoord3 *intersection) const;
	float IntersectBox(const CVector3 &move,float radius,const CCoord3 &min,const CCoord3 &max,CCoord3 *intersection) const;

	// intersection avec un plan d'un rayon projet  partir du point
	float IntersectPlane(const CVector3 &direction,const CCoord3 &planeOrigin,const CVector3 &planeNormal) const;

	// intersection avec une sphere d'un rayon projet  partir du point
	float IntersectSphere(CVector3 direction,const CCoord3 &centre,float rayon) const;

	bool isInTriangle(const CCoord3 &p1,const CCoord3 &p2, const CCoord3 &p3) const;
	CCoord3 ClosestPointOnTriangle(const CCoord3 &p1,const CCoord3 &p2, const CCoord3 &p3) const;
	CCoord3 ClosestPointOnSegment(const CCoord3 &p1,const CCoord3 &p2) const;

	// cast en const float*
	operator const float*() const;
	// cast en float*
	operator float*();
};

inline std::istream &operator >>(std::istream &in,CVector3 &v)
{
	in>>v.x;
	in>>v.y;
	in>>v.z;
	return in;
}

inline CVector3::CVector3(const std::string &s)
{
	std::stringstream ss(s);
	ss>>*this;
}

#include "matrix4.h"
#include "vector3.inl"

#endif	// VECTOR3_H
