/***************************************************************************
 *   Copyright (C) 2006-2008 by Paul-Louis Ageneau                         *
 *   paullouisageneau@gmail.com                                            *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.           *
 ***************************************************************************/

#include "game.h"
#include "select.h"

CGame::CGame(void)
{
	
}

CGame::~CGame(void)
{

}

// Initialisation de l'tat
void CGame::Init(void)
{
	mScene=new CScene;
	mScene->setFogColor(CColor(242,236,217));
	mScene->setFogDistance(8000.f,15000.f);

	mScene->setBackground( MediaManager->Get<CTexture>("sky/front.jpg"),
			MediaManager->Get<CTexture>("sky/back.jpg"),
			MediaManager->Get<CTexture>("sky/right.jpg"),
			MediaManager->Get<CTexture>("sky/left.jpg"),
			MediaManager->Get<CTexture>("sky/up.jpg"),
			NULL);
	
	mCamera=new CCamera;
	mCamera->Attach(mScene->getRootEntity());
	mCamera->setRange(2.f,15000.f);
	mCamMode = 0;
	
	pLight light=new CPointLight;
	light->Translate(CVector3(100000.f,100000.f,100000.f));
	light->Attach(mScene->getRootEntity());
	light->CastShadow(true);

	pMaterial mat_clouds=new CMaterial;
	mat_clouds->setAmbient(CColor(0.7f,0.7f,0.7f));
	mat_clouds->setDiffuse(CColor(1.0f,1.0f,1.0f));
	mat_clouds->setSpecular(CColor(0.3f,0.3f,0.3f));
	mat_clouds->setAlpha(0.4f);
	mat_clouds->setTexture(MediaManager->Get<CTexture>("map/clouds.png"),0);
	pTerrain clouds=new CTerrain(50000.f,50000.f,32,32,10,10,mat_clouds);
	clouds->setHeight("map/cloudsheight.png",1000.f);
	clouds->updateNormals();

	pObject cloudsobject=new CObject(clouds);
	cloudsobject->setEnv(GL_CULL_FACE,false);
	cloudsobject->Attach(mScene->getRootEntity());
	cloudsobject->Translate(CVector3(0.f,4000.f,0.f));
	
	cloudsobject=new CObject(clouds);
	cloudsobject->setEnv(GL_CULL_FACE,false);
	cloudsobject->Attach(mScene->getRootEntity());
	cloudsobject->Translate(CVector3(500.f,3000.f,500.f));
	
	// Le terrain
	pMaterial mat_terrain=new CMaterial;
	mat_terrain->setAmbient(CColor(0.7f,0.7f,0.7f));
	mat_terrain->setDiffuse(CColor(1.0f,1.0f,1.0f));
	mat_terrain->setSpecular(CColor(0.3f,0.3f,0.3f));
	mat_terrain->setTexture(MediaManager->Get<CTexture>("map/color.png"),0);
	mat_terrain->setTexture(MediaManager->Get<CTexture>("map/detail.png"),1);
	mTerrain=new CTerrain(50000.f,50000.f,128,128,3,3,mat_terrain);
	mTerrain->setHeight("map/height.png",4000.f);
	mTerrain->updateNormals();

	mTerrainObject=new CObject(mTerrain/*new COctree(MediaManager->Get<CMesh>("map/map.3ds"))*/);
	mTerrainObject->Script("map/map.p3d");
	mTerrainObject->PlayAnimation();
	mTerrainObject->Attach(mScene->getRootEntity());

	mScene->getRootEntity()->setAnimationSpeed(1.);
	mScene->getRootEntity()->PlayAnimation();
	
	// Scene de l'interface
	mInterface = new CScene;
	mInterface->getRootEntity()->setEnv(GL_FOG,false);
	light = new CDirectionalLight;
	light->Rotate(CVector3(0.f,0.f,-1.f));
	light->Attach(mInterface->getRootEntity());

	// Console
	mTextRoot = new CEntity;
	mTextRoot->Translate(CVector3(30.f,30.f,0.f));
	mTextRoot->Attach(mInterface->getRootEntity());
	
	pMaterial font = new CMaterial(MediaManager->Get<CTexture>("font.png"),0.99f);
	mTyping = new CText;
	mTyping->setFont(font,15.f,30.f);
	mTyping->Translate(CVector3(0.f,-20.f,0.f));
	mTyping->Attach(mTextRoot);
	
	// Support de la minicarte
	mMap = new CEntity;
	mMap->Attach(mInterface->getRootEntity());
	
	mMapCursors = new CEntity;
	mMapCursors->Attach(mMap);

	// Minicarte
	pPlain mapimage = new CPlain(50000.f,50000.f,new CMaterial(MediaManager->Get<CTexture>("panel/map.png"),0.99f));
	mapimage->Attach(mMap);
	
	pPlain support = new CPlain(100000.f,100000.f,new CMaterial(MediaManager->Get<CTexture>("panel/support.png"),0.99f));
	support->Attach(mMap);
	
	for(int i=0;i<2;++i)
	{
		mPointers[i] = new CPlain(128,128,new CMaterial(MediaManager->Get<CTexture>("panel/pointer.png"),0.99f));
		mPointers[i]->Attach(mInterface->getRootEntity());
	}
	
	pPlain plain = new CPlain(512,512,new CMaterial(MediaManager->Get<CTexture>("panel/left.png"),0.99f));
	plain->Translate(CVector3(650,256,0));
	plain->Attach(mInterface->getRootEntity());
	
	for(int i=0;i<3;++i)
	{
		mMapGears[i] = new CPlain(128,128,new CMaterial(MediaManager->Get<CTexture>("panel/gear.png"),0.99f));
		mMapGears[i]->Attach(mInterface->getRootEntity());
	}
	
	plain = new CPlain(512,512,new CMaterial(MediaManager->Get<CTexture>("panel/right.png"),0.99f));
	plain->Translate(CVector3(1280-256,256,0));
	plain->Attach(mInterface->getRootEntity());
	
	mCamAnglex = 0.f;
	mCamAngley = PI;
	mCamLock = 0;

	mMusicVolume = 0.7f;
	mMusicTransition = 1.f;
	ChangeMusic("normal",1);

	Engine->PushState(new CSelect);
}

// Fermeture de l'tat
void CGame::Cleanup(void)
{

}

// Callback de mise  jour de l'tat
bool CGame::Update(double time)
{
	if(isKeyDown(KEY_ESCAPE)) return false;
	if(isTopState() && isKeyDown(KEY_TAB))
	{
		Process("QUIT");
		Engine->PushState(new CSelect);
	}
	
	mScene->Update(time);
	mInterface->Update(time);

	// Mise  jour de l'interfaces
	mMap->Identity();
	mMap->Translate(CVector3(1280-200,200,0.f));
	mMap->Scale(CVector3(256.f/50000.f,256.f/50000.f,1.f));
	CVector3 dir = mCamera->getGlobalMatrix().getAxisZ();
	float gangle = std::atan2(dir.x,-dir.z)-PI;
	mMap->Rotate(CQuaternion(CVector3(0.f,0.f,1.f),gangle));
	
	// Engrenages
	for(int i=0;i<3;++i)
	{
			mMapGears[i]->Identity();
			mMapGears[i]->Translate(CVector3(1280-120-80*i,80+20*i,0));
			mMapGears[i]->Scale(CVector3(1.f+0.5f*i,1.f+0.5f*i,1.f));
			mMapGears[i]->Rotate(CQuaternion(CVector3(0.f,0.f,1.f),gangle*3*(1.5f-i)+i*PI/3));
	}

	// Aiguilles des cadrans
	for(int i=0;i<2;++i)
	{
			mPointers[i]->Identity();
			mPointers[i]->Translate(CVector3(630+175*i,96,0));
			float angle = 0.f;
			if(mTargetCamera != NULL)
			{
				if(i==0) angle = mTargetCamera->getLocalMatrix().getTranslation().y*2*PI/10000.f-PI;
				else angle = mTargetCamera->getPowerFactor()*PI-2*PI/3;
			}
			mPointers[i]->Rotate(CQuaternion(CVector3(0.f,0.f,-1.f),angle));
	}

	// Choix des musiques appropries
	if(mTargetCamera != NULL)
	{
		if(mTargetCamera->getHeatOfBattle() > 0.6f)			ChangeMusic("battle",rand()%3+1);
		else if(mTargetCamera->getHeatOfBattle() < 0.2f)	ChangeMusic("normal",rand()%3+1);
	}
	
	// Mixage des musiques
	if(mNextMusic==NULL && mQueuedMusic!=NULL)
	{
		mNextMusic = mQueuedMusic;
		mQueuedMusic = NULL;
		mNextMusic->setGain(0.f);
		mNextMusic->Play(true);
	}
	
	if(mNextMusic != NULL)
	{
		mMusicTransition-=time*0.25f;
		if(mMusicTransition <= 0.f)
		{
			if(mCurrentMusic != NULL) 
			{
				mCurrentMusic->Stop();
				mCurrentMusic->Detach();
			}
			mCurrentMusic = mNextMusic;
			mNextMusic = NULL;
			mMusicTransition = 1.f;
		}
	}

	if(mCurrentMusic != NULL)	mCurrentMusic->setGain(mMusicTransition*mMusicVolume);
	if(mNextMusic != NULL)		mNextMusic->setGain((1.f-mMusicTransition)*mMusicVolume);
	
	return true;
}

// Callback de dessin de l'tat
int CGame::Draw(void)
{
	//mCamera->Clear(CColor(0.5f,0.5f,75.f));	// inutile
	mCamera->Apply();

	int polys=0;
	polys+=mScene->RenderBackground();
	polys+=mScene->Render();
	
	CCamera::Default->Apply();
	polys+=mInterface->Render();
	
	return polys;
}

// Sous-routine de mise  jour camera
void CGame::UpdateCamera(const usercmd_t &cmd,double time)
{
	if(mCamera == NULL || mTargetCamera == NULL) return;
	
	if(mTargetCamera->isDead())
	{
		CCoord3 campos = mCamera->getLocalMatrix().getTranslation();
		mCamera->Identity();
		mCamera->Translate(campos);
		mCamera->Rotate(campos-mTargetCamera->getGlobalMatrix().getTranslation());
		return;
	}

	float camdist = mTargetCamera->getRadius()*10.f;
	float camheight = camdist/2.f;

	mCamAnglex = -cmd.stickAux.y*PI/3;
	mCamAngley = -cmd.stickAux.x*PI/2;
	if(mCamAnglex>PI/3) mCamAnglex=PI/3;
	if(mCamAnglex<-PI/3) mCamAnglex=-PI/3;
	
	CMatrix4 matrix = mTargetCamera->getGlobalMatrix();
	CVector3 diry = matrix.getAxisY().Normalize();
	CVector3 dirz = matrix.getAxisY().Normalize();
	CCoord3 campos(matrix.getTranslation());
	CQuaternion camrot;

	pEntity called = mTargetCamera->Call("camera");
	if(mCamMode==1 && called==NULL) mCamMode = 0;
	if(mCamMode==1) matrix = called->getGlobalMatrix() ;

	CVector3 camdir;
	
	if(mCamLock==2) mCamLock=0;	// TODO: trop compliqu  manier

	switch(mCamLock)
	{
	case 0:
		{
			camdir.set(0.f,0.f,1.f);
			break;
		}

	case 1:
		{
			CCoord3 nearest;
			float dist = std::numeric_limits<float>::infinity();
			for(NetEntitiesMap_t::iterator it=mNetEntities.begin(); it!=mNetEntities.end(); ++it)
			{
				if(it->first >= INDEX_NONPLAYER) break;
				
				pPlayer player = it->second;
				if(mTargetCamera->getGroup() == player->getGroup()) continue;
				
				float d = it->second->getLocalMatrix().getTranslation().Distance(campos);
				if(d < 1.f) continue;
				if(d < dist) {
					nearest = it->second->getLocalMatrix().getTranslation();
					dist = d;
				}
			}

			nearest.y+=100.f;
			camdir = nearest-campos;
			camdir.Normalize();
			break;
		}

	case 2:
		{
			camdir.x = std::sin(mCamAngley);
			camdir.z = std::cos(mCamAngley);
			camdir.y = -std::tan(mCamAnglex);
			break;
		}
	}
		
	
	switch(mCamMode)
	{
	case 0:
		{			
			if(mCamLock != 1) camdir*=matrix.NoTranslation();
			campos-=camdir*camdist;
			if(mCamLock == 0)
			{
				campos+=diry*camheight;
				camrot.FromDirection(campos-matrix.getTranslation()-mTargetCamera->getVelocity()*0.3f);
			}
			else {
				camrot.FromDirection(campos-matrix.getTranslation());
			}
			break;
		}
	
	case 1:
		{
			campos = called->getGlobalMatrix().getTranslation();
			if(mCamLock!=1) camrot = matrix.getRotation();
			camrot*=CQuaternion(-camdir);
			break;
		}
	}
	
	float t = 1.f-std::exp(-float(time)*5.f);
	
	CVector3 newpos;
	if(mCamMode == 1) newpos = campos;
	else {
		CVector3 pos = mCamera->getLocalMatrix().getTranslation();
		newpos = pos.Lerp(campos,t);
		CVector3 move = newpos-pos;
		CVector3 newmove;
		if(mTerrainObject->Collision(pos,move,4.f,&newmove))
			move = newmove;	
		newpos = pos + move;
	}
	
	CQuaternion newrot = mCamera->getLocalMatrix().getRotation().Lerp(camrot,t);

	mCamera->Identity();
	mCamera->Translate(newpos);
	mCamera->Rotate(newrot);

	CSound::Listener(mCamera,time);
}

pNetEntity CGame::getNetEntity(int object)
{
	NetEntitiesMap_t::iterator it = mNetEntities.find(object);
	if(it == mNetEntities.end()) return NULL;
	return it->second;
}

bool CGame::Process(const std::string &cmd,const std::string &data)
{
	if(cmd == "WRITE") 	Write(data,COLOR_NORMAL);
	else if(cmd == "NOTIF")	Write(data,COLOR_NOTIF);
	else if(cmd == "EVENT")	Write(data,COLOR_EVENT);
	else return false;
	return true;
}

// Entres joueur
void CGame::SampleInput(usercmd_t &cmd,double time)
{
	int mousemx,mousemy,mousemz;
	Engine->getMouseMove(&mousemx,&mousemy,&mousemz);
		
	cmd.stick.x += float(mousemx)/200.f;
	cmd.stick.y += float(mousemy)/200.f;
	cmd.stick.z += float(mousemz)/10.f;

	if(isKeyDown(KEY_PAGEUP)) cmd.stick.z += 0.5f*time;
	if(isKeyDown(KEY_PAGEDOWN)) cmd.stick.z -= 0.5f*time;
	
	unsigned buttons = 0x0;
	if(isKeyDown(KEY_RCTRL) || isMouseButtonDown(MOUSE_BUTTON_LEFT)) buttons |= 0x1;
	if(isMouseButtonDown(MOUSE_BUTTON_RIGHT)) buttons |= 0x2;
	if(isKeyDown(KEY_ENTER)) buttons |= 0x4;
	if(isKeyDown(KEY_DEL)) buttons |= 0x8;
	if(isKeyDown(KEY_END)) buttons |= 0x10;
	if(isKeyDown(KEY_SPACE)) buttons |= 0x20;

	cmd.buttons = buttons;
}

void CGame::ChangeMusic(const std::string &name,int nbr)
{
	if(name == mLastMusicName) return;
	mLastMusicName = name;
	std::stringstream file;
	file<<"music/"<<name<<nbr<<".ogg";
	
	try {
		mQueuedMusic = new CSound(MediaManager->Get<CMusic>(file.str()));
		mQueuedMusic->Attach(mCamera);
	}
	catch(...)
	{
	
	}
}

void CGame::Write(const std::string &message,const CColor &color)
{
	pMaterial font = new CMaterial(MediaManager->Get<CTexture>("font.png"));
	font->setAmbient(color);
	font->setDiffuse(color);
	font->setAlpha(0.99f);

	pText text = new CText(message);
	text->setFont(font,15.f,30.f);

	if(mLastText!=NULL && mLastText->getFather()!=NULL) mLastText->Attach(text);
	mLastText = text;
	text->Attach(mTextRoot);
	text->Translate(CVector3(0.f,28.f,0.f));
	text->setLife(10.f);
}

void CGame::KeyCallback(int key,int action)
{
	switch(key)
	{
	case KEY_ENTER:
		if(action == KEY_PRESS)
		{
			const std::string &text = mTyping->getText();
			if(text.empty()) mTyping->setText("_");
			else {
				if(text.size() >= 2) Process("SAY",text.substr(0,text.size()-1));
				mTyping->setText("");
			}
		}
		break;

	case KEY_BACKSPACE:
		if(action == KEY_PRESS)
		{
			const std::string &text = mTyping->getText();
			if(text.size() >= 2) 
				mTyping->setText(text.substr(0,text.size()-2)+'_');
		}
		break;

	}
}

void CGame::CharCallback(int character,int action)
{
	if(action == KEY_PRESS)
	{
		std::string text = mTyping->getText();
		if(!text.empty()) 
		{
			text[text.size()-1]=char(character);
			mTyping->setText(text+'_');
		}
	}
}


void CGame::MouseCallback(int button,int action)
{
	switch(button)
	{
	case MOUSE_BUTTON_MIDDLE:
		if(action == KEY_PRESS) ++mCamMode;
		if(mCamMode >= 2) mCamMode = 0;
		break;

	case MOUSE_BUTTON_RIGHT:
		if(action == KEY_PRESS) ++mCamLock;
		if(mCamLock >= 3) mCamLock = 0;
		break;

	}
}
