/***************************************************************************
 *   Copyright (C) 2006-2008 by Paul-Louis Ageneau                         *
 *   paullouisageneau@gmail.com                                            *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.           *
 ***************************************************************************/

#include "server.h"
#include "player.h"
#include "bot.h"

CServer::CServer(void)
{
	mSockStream = INVALID_SOCKET;
	mSockDgram = INVALID_SOCKET;
	
	addrinfo aiHints;
	memset(&aiHints, 0, sizeof(aiHints));
	addrinfo *aiList = NULL;
	
	// Mise en place des sockets d'coute
	try {
		aiHints.ai_family = AF_UNSPEC;
		aiHints.ai_socktype = 0;
		aiHints.ai_protocol = 0;
		aiHints.ai_flags = AI_PASSIVE;
		if(getaddrinfo(NULL, NET_PORT, &aiHints, &aiList) != 0)
			throw CException("Impossible d'obtenir l'adresse locale");
			
		// Socket de flux
		mSockStream = socket(aiList->ai_family,SOCK_STREAM,0);
		if(bind(mSockStream,(sockaddr*)aiList->ai_addr,aiList->ai_addrlen) != 0) 
			throw CException("Impossible d'attacher au port");
		if(listen(mSockStream,SOMAXCONN) != 0)
			throw CException("Impossible d'couter les connexions");
			
		// Socket de datagrammes
		mSockDgram = socket(aiList->ai_family,SOCK_DGRAM,0);
		if(bind(mSockDgram,(sockaddr*)aiList->ai_addr,aiList->ai_addrlen) != 0)
			throw CException("Impossible d'attacher au port");

		freeaddrinfo(aiList);
		aiList = NULL;
		
		if(mSockStream == INVALID_SOCKET || mSockDgram == INVALID_SOCKET) 
			throw CException("Protocole manquant");

		ctl_t b = 1;
		if(ioctl(mSockStream,FIONBIO,&b)<0 || ioctl(mSockDgram,FIONBIO,&b)<0)
			throw CException("Impossible d'utiliser des sockets non bloquantes");

	}
	catch(...)
	{
		if(aiList != NULL) freeaddrinfo(aiList);
		if(mSockDgram != INVALID_SOCKET) close(mSockDgram);
		if(mSockStream != INVALID_SOCKET) close(mSockStream);
		throw;
	}

	mCurrentStamp = 0.;
	mSendCooldown = 0.;

	// Initialisation du pseudo-client local
	mLocalClient.sock					= INVALID_SOCKET;
	mLocalClient.nPlayer				= 0;
	mLocalClient.lastRecvStamp			= 0.;
	mLocalClient.lastAckStamp			= 0.;
	mLocalClient.lastRecvLocalStamp		= 0.;
	mLocalClient.nEntity				= -1;
	mLocalClient.latency				= 0.;
	mLocalClient.flags					= 0x0;
	mLastPlayer = 0;

	mNextBot = INDEX_BOTS;
}

CServer::~CServer(void)
{

}

void CServer::Init(void)
{
	CGame::Init();
}

void CServer::AddBot(const std::string &team,const std::string &plane)
{
	pBot bot = new CBot;
	bot->Script(team+"/"+plane+".p3d");
	bot->setGroup(team);
	bot->setName("** CPU **");
	bot->setIdentifier(mNextBot);
	bot->Attach(mScene->getRootEntity());
	bot->setChanged(DATA_CREATE);
	bot->setCursor(team+".png");

	mNetEntities[mNextBot] = pNetEntity(bot);
	++mNextBot;
}

bool CServer::Update(double time)
{
	mCurrentStamp = Engine->getTime();
	
	// Boucle de rception des nouvelles connexions
	SOCKET sock;
	addr_t addr;
	addr.len = sizeof(addr.addr);
	while((sock = accept(mSockStream,(sockaddr*)&addr.addr,&addr.len)) != INVALID_SOCKET)
	{
		ctl_t b = 1;
		if(ioctl(sock,FIONBIO,&b)<0) 
		{
			close(sock);
			continue;
		}
		
		client_t &client = mClients[addr];	// insertion du nouveau client
		client.sock					= sock;
		client.nPlayer				= ++mLastPlayer;
		client.lastRecvStamp		= 0.;
		client.lastAckStamp			= 0.;
		client.lastRecvLocalStamp	= 0.;
		client.nEntity				= -1;
		client.latency				= 0.;
		client.flags				= 0x0;
		
		client.name = "Anonyme";
		client.team = "empire";
	}
	if(sockerrno != EWOULDBLOCK) 
		Log<<"Erreur: retour accept = "<<sockerrno<<std::endl;
	
	// Par chaque client
	for(ClientsMap_t::iterator it=mClients.begin(); it!=mClients.end();)
	{
		// Boucle de rception du socket de flux
		client_t &client = it->second;
		char chr;
		while(recv(client.sock,&chr,1,0) == 1)
		{
			if(chr!='\r' && chr!='\n')
			{
				if(chr==8) {
					if(!client.recvLine.empty())
						client.recvLine.erase(--client.recvLine.end());
				}
				else client.recvLine+= chr;
			}
			if(chr=='\n') {
				size_t p = client.recvLine.find_first_of(' ');
				client.recvLine.erase(p,1);
				
				std::string command = client.recvLine.substr(0,p);
				std::string data = client.recvLine.substr(p,std::string::npos);
				
				Process(client,command,data);
				client.recvLine.clear();
			}

		}
		
		if(sockerrno != EWOULDBLOCK)
		{
			Process(client,"QUIT");
			close(client.sock);
			mClients.erase(it++);
		}
		else ++it;
	}

	// Boucle de rception du socket de datagrammes
	char buffer[NET_BUFFERSIZE];
	int size;
	addr.len = sizeof(addr.addr);
	while((size = recvfrom(mSockDgram,buffer,NET_BUFFERSIZE,0,(sockaddr*)&addr.addr,&addr.len)) > 0)
	{
		// Cherche le client dans la map
		ClientsMap_t::iterator it = mClients.find(addr);
		if(it == mClients.end()) continue;
		client_t &client = it->second;

		try {
			buffer_t data(buffer,size);

			// Lit l'en-tte
			double stamp	= data.readTime();		// stamp courant
			double ack		= data.readTime();		// dernier stamp reu par le serveur
			double decal	= data.readTime();		// temps d'envoi du dernier, permet de calculer la latence
			
			if(stamp <= client.lastRecvStamp) continue;	// droppe si trop vieux ou dj reu
			client.lastRecvStamp = stamp;
			client.lastAckStamp = ack;
			client.lastRecvLocalStamp = mCurrentStamp;
			
			double latency = (mCurrentStamp - (ack+decal))/2;	// calcul de la latence
			client.latency = (client.latency*9 + latency)/10;	// moyennage
			
			client.flags |=	CLIENT_DGRAM;
			if(data.left() == 0) continue;				// paquet vide
			
			int object	= data.readInt8();				// object
			int flags	= data.readInt8();				// flags
				
			if(object != client.nEntity) continue;		// droppe si illgal
			pNetEntity player = getNetEntity(object);
			if(player != NULL) player->Input(client.latency,flags & DATA_CLIENT,data);
		}
		catch(const CException &e)
		{
			Log<<"Erreur: deconnexion de "<<client.name<<" -> "<<e.what();
			Process(client,"QUIT");
			close(client.sock);
			mClients.erase(it);
		}
	}
	if(sockerrno != EWOULDBLOCK) 
		std::cout<<"Erreur: retour recvfrom sur dgram = "<<sockerrno<<std::endl;
	
	// Rcupration et excution de l'entre joueur
	pPlayer player = getNetEntity(0);
	if(player != NULL)
	{
		usercmd_t cmd = player->getCommand();
		SampleInput(cmd,time);
		player->setCommand(cmd);
	}
	
	// Mise  jour du jeu
	if(!CGame::Update(time)) return false;
	if(player != NULL)
		UpdateCamera(player->getCommand(),time);

	// Mise  jour serveur des entites
	for(NetEntitiesMap_t::iterator it=mNetEntities.begin(); it!=mNetEntities.end();)
	{
		it->second->ServerUpdate(time);
		if(it->second->getFather() == NULL) mNetEntities.erase(it++);
		else ++it;
	}
	
	// Envoi des mises  jour
	mSendCooldown-=time;
	if(mSendCooldown <= 0.)
	{
		mSendCooldown+= 1./NET_UPDATERATE;
		
		for(ClientsMap_t::iterator iclient=mClients.begin(); iclient!=mClients.end(); ++iclient)
		{
			if(iclient->second.nEntity == -1 || !(iclient->second.flags & CLIENT_DGRAM)) continue;
			
			buffer_t data;
			data.writeTime(mCurrentStamp);											// stamp
			data.writeTime(iclient->second.lastRecvStamp);							// ack
			data.writeTime(mCurrentStamp - iclient->second.lastRecvLocalStamp);		// decal
			data.writeInt8(iclient->second.nEntity);								// entite controlle
			
			for(NetEntitiesMap_t::iterator it=mNetEntities.begin(); it!=mNetEntities.end(); ++it)
			{
				int flags = it->second->getChanged(iclient->second.lastAckStamp);
					
				if(it->first != iclient->second.nEntity) flags |= DATA_LATENCY;
				else flags &= ~(DATA_CLIENT|DATA_LATENCY);
					
				data.writeInt8(it->first);		// object
				data.writeInt8(flags);			// flags
					
				/*if(it->first == iclient->second.nEntity) data.writeInt8(0);	// events
				else it->second->OutputEvents(flags,data);*/
				it->second->OutputData(flags,data);		// data
			}

			if(sendto(mSockDgram,data.ptr(),data.size(),0,(sockaddr*)&iclient->first.addr,iclient->first.len) != data.size())
				std::cout<<"Erreur: retour sendto sur dgram = "<<sockerrno<<std::endl;
		}
	}

	return true;
}

bool CServer::Process(client_t &client,const std::string &command,const std::string &data)
{				  
	if(client.nEntity<0)
	{
		if(command=="NAME")		client.name = data;
		else if(command=="TEAM")	client.team = data;
		else if(command=="PLANE")	client.plane = data;
		else if(command=="JOIN")
		{
			pPlayer entity = new CPlayer;
			entity->setGroup(client.team);
			entity->setName(client.name);
			entity->Script(client.team+'/'+client.plane+".p3d");
			entity->Attach(mScene->getRootEntity());
			entity->setChanged(DATA_CREATE);
			entity->setCursor(client.team+".png");
			
			entity->setIdentifier(client.nPlayer);
			client.nEntity = client.nPlayer;

			if(client.nPlayer == 0)
			{
				entity->setCursor("player.png");
				entity->setNameVisibility(false);
				mTargetCamera = pNetEntity(entity);
			}

			Process("EVENT",client.name+" a rejoint la partie.");
		}
		else return false;
	}
	else if(command == "SAY")
	{
		Process("WRITE",client.name+": "+data);
	}
	else if(command=="QUIT")
	{
		NetEntitiesMap_t::iterator it = mNetEntities.find(client.nEntity);
		if(it != mNetEntities.end())
		{
			it->second->Attach(NULL);
			mNetEntities.erase(it);
		}
		client.nEntity = -1;

		Process("EVENT",client.name+" a quitt la partie.");
	}
	else return false;

	return true;
}

bool CServer::Process(const std::string &command,const std::string &data)
{
	if(Process(mLocalClient,command,data)) return true;
	if(!CGame::Process(command,data)) return false;
	
	std::string message = command+' '+data+'\n';
	for(ClientsMap_t::iterator it=mClients.begin(); it!=mClients.end(); ++it)
	{
		client_t &client = it->second;
		send(client.sock,message.data(),message.size(),0);
	}
	return true;
}
