#!/usr/bin/perl -w

# Copyright (C) 2004 Mark Seaborn
#
# This file is part of Plash, the Principle of Least Authority Shell.
#
# Plash is free software; you can redistribute it and/or modify it
# under the terms of the GNU Lesser General Public License as
# published by the Free Software Foundation; either version 2.1 of
# the License, or (at your option) any later version.
#
# Plash is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public
# License along with Plash; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
# USA.


# A variant such as
#   al_cat: arglist, arglist
# generates
#   struct arg_list { int variant; };
#   struct arg_list_al_cat {
#     int variant;
#     struct arg_list *a1;
#     struct arg_list *a2;
#   };
#   struct arg_list *mk_al_cat(region_t r, struct arg_list *a1, struct arg_list *a2)
#   {
#     struct arg_list_al_cat *n = amalloc(sizeof(struct arg_list_al_cat));
#     n->variant = 1;
#     n->a1 = a1;
#     n->a2 = a2;
#     return (struct arg_list *) n;
#   }
#   int m_al_cat(struct arg_list *in, struct arg_list **out1, struct arg_list **out2)
#   {
#     if(in->variant == 1) {
#       struct arg_list_al_cat *x = (void *) in;
#       *out1 = n->a1;
#       *out2 = n->a2;
#       return 1;
#     }
#     else return 0;
#   }


use IO::File;

local $count = 1;


my $defs =
[
  { Name => 'arg_list',
    Variants =>
      [['arg_empty', []],
       ['arg_cat', ['a1 arg_list', 'a2 arg_list']],
       ['arg_read', ['a arg_list']],
       ['arg_write', ['a arg_list']],
       ['arg_ambient', ['f arg_list']],
       ['arg_string', ['s char_cons']],
       ['arg_filename', ['f char_cons']],
       ['arg_glob_filename', ['f glob_path']],
       ['arg_redirection', ['fd char_cons', 'type T int', 'd redir_dest']],
       ['arg_fs_binding', ['filename char_cons', 'e shell_expr']],
      ]
  },
  { Name => 'command',
    Variants =>
      [['command', ['pl pipeline', 'bg_flag T int']],
       ['chdir', ['f char_cons']],
       ['command_fg', ['job char_cons']],
       ['command_bg', ['job char_cons']],
       ['def_binding', ['c char_cons', 'e shell_expr']],
       ['command_source', ['filename char_cons']],
      ]
  },
  { Name => 'command_list',
    Variants =>
      [['commands_cons', ['c command', 'rest command_list']]]
  },
  { Name => 'pipeline',
    Variants =>
      [['pipeline_cons', ['inv invocation', 'pl pipeline']],
       ['pipeline_inv', ['inv invocation']]
      ]
  },
  { Name => 'invocation',
    Variants =>
      [['invocation', ['no_sec T int', 'c char_cons', 'a arg_list']]]
  },
  { Name => 'glob_path',
    Variants =>
      [['glob_path', ['start path_start', 'p glob_path_aux']]]
  },
  { Name => 'glob_path_aux',
    Variants =>
      [['glob_path_cons', ['c char_cons', 'rest glob_path_aux']],
       ['glob_path_end', ['slash T int']],
      ]
  },
  { Name => 'path_start',
    Variants =>
      [['start_root', []],
       ['start_cwd', []],
       ['start_home', ['user char_cons']],
      ]
  },
  { Name => 'redir_dest',
    Variants =>
      [['dest_fd', ['number char_cons']],
       ['dest_file', ['path char_cons']],
      ]
  },
  { Name => 'shell_expr',
    Variants =>
      [['expr_var', ['c char_cons']],
       ['cap_cmd', ['c char_cons', 'a arg_list']],
       ['expr_filename', ['filename char_cons']],
       ['expr_mkfs', ['args arg_list']],
      ]
  },
];

my $banner =
  "/* This file was automatically generated by gen-variants.pl. */\n\n";

my $out = 'src/shell-variants';

print "Writing $out.c\n";
my $out_main = IO::File->new("$out.c", O_CREAT | O_WRONLY | O_TRUNC);
if(!defined $out_main) { die }
print $out_main $banner;
print $out_main "#include \"shell-variants.h\"\n\n";
foreach my $def (@$defs) { gen($out_main, $def); }

print "Writing $out.h\n";
my $out_hdr = IO::File->new("$out.h", O_CREAT | O_WRONLY | O_TRUNC);
if(!defined $out_hdr) { die }
print $out_hdr $banner;
print $out_hdr "#include \"region.h\"\n";
print $out_hdr "#include \"shell.h\"\n";
foreach my $desc (@$defs) {
  print $out_hdr "struct $desc->{Name};\n";
}
foreach my $def (@$defs) { gen_header($out_hdr, $def); }


sub process_args {
  my ($v) = @_;
  my @args;
  foreach my $arg (@{$v->[1]}) {
    my $name;
    my $type;
    if($arg =~ /^(\S+)\s+T\s+/) {
      $name = $1;
      $type = "$' ";
    }
    elsif($arg =~ /^(\S+)\s+(\S+)$/) {
      $name = $1;
      $type = "struct $2 *";
    }
    else { die "Bad: $v" }
    push(@args, { Name => $name, Type => $type });
  }
  \@args
}

sub gen_header {
  my ($out, $desc) = @_;

  foreach my $v (@{$desc->{Variants}}) {
    my $v_name = $v->[0];
    my $args = process_args($v);
    
    print $out "struct $desc->{Name} *mk_$v_name(region_t r".
      join('', map { ", $_->{Type}$_->{Name}" } @$args).");\n";
    
    print $out "int m_$v_name(struct $desc->{Name} *in_obj".
      join('', map { ", $_->{Type}*out_$_->{Name}" } @$args).");\n";
  }
}

sub gen {
  my ($out, $desc) = @_;

  # Generate struct
  print $out "struct $desc->{Name} { int variant; };\n";
  my $type_code = $count++;
  my $min_code = $type_code * 1000;
  my $max_code = $type_code * 1000 + scalar(@{$desc->{Variants}});
  my $v_number = $min_code;

  foreach my $v (@{$desc->{Variants}}) {
    my $v_name = $v->[0];
    my $args = process_args($v);

    print $out "\n";

    # Generate a struct for each variant
    print $out "struct $desc->{Name}_$v_name {\n";
    print $out "  int variant;\n";
    foreach my $n (@$args) { print $out "  $n->{Type}$n->{Name};\n"; }
    print $out "};\n";

    # Generate constructor function
    print $out "struct $desc->{Name} *mk_$v_name(region_t r".
      join('', map { ", $_->{Type}$_->{Name}" } @$args).")\n";
    print $out "{\n";
    print $out "  struct $desc->{Name}_$v_name *new_obj = ".
      "region_alloc(r, sizeof(struct $desc->{Name}_$v_name));\n";
    print $out "  new_obj->variant = $v_number;\n";
    foreach my $n (@$args) { print $out "  new_obj->$n->{Name} = $n->{Name};\n"; }
    print $out "  return (struct $desc->{Name} *) new_obj;\n";
    print $out "}\n";

    # Generate matching function
    print $out "int m_$v_name(struct $desc->{Name} *in_obj".
      join('', map { ", $_->{Type}*out_$_->{Name}" } @$args).")\n";
    print $out "{\n";
    print $out "  assert(in_obj);\n";
    print $out "  assert($min_code <= in_obj->variant && in_obj->variant < $max_code);\n";
    print $out "  if(in_obj->variant == $v_number) {\n";
    if(scalar(@$args) > 0) {
      print $out "    struct $desc->{Name}_$v_name *x = (void *) in_obj;\n";
      foreach my $n (@$args) {
	print $out "    if(out_$n->{Name}) *out_$n->{Name} = x->$n->{Name};\n";
      }
    }
    print $out "    return 1;\n";
    print $out "  }\n";
    print $out "  else return 0;\n";
    print $out "}\n";
    
    $v_number++;
  }
}
