(* Copyright (C) 2004 Mark Seaborn

   This file is part of Parser Generator.

   Parser Generator is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as
   published by the Free Software Foundation; either version 2, or (at
   your option) any later version.

   Parser Generator is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with Parser Generator; see the file COPYING.  If not, write
   to the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
   Boston, MA 02111-1307 USA.  *)


open StdLabels
open My_utils
open Parser_runtime

module G = Gram_out1
module Gen = C_generator

let compile opts file =
  let str = with_input file ~f:input_all_string in
    match do_parse G.f_defs str with
      | OK(i, defs1) when i = String.length str ->
          let defs = ref [] in
            Expand_out.add_def defs "null" Pexprs.P_null;
            Expand_out.add_def defs "fail" Pexprs.P_fail;
            (* C-specific version of `simplify' used: *)
            List.iter defs1
              ~f:(fun (v, exp) ->
                    Expand_out.add_def defs v
                      (Expand_out.C.simplify ~defs exp));
            Printf.printf "/* This file is automatically generated */\n\n";
            Gen.generate_defs ~opts stdout !defs;
            flush stdout
      | _ -> output_errors stderr

let _ =
  let prog_name::args = Array.to_list Sys.argv in
  let rec f opts = function
    | "--extra-arg"::name::decl::rest ->
        f { opts with Gen.extra_args = opts.Gen.extra_args @
                        [{ Gen.name=name; Gen.decl=decl }] } rest
    | [file] -> compile opts file
    | _ -> Printf.printf "Usage: %s input_file\n" prog_name
  in f { Gen.extra_args =
           [{ Gen.name="err_pos"; Gen.decl="const char **err_pos" }];
         Gen.inhibit_actions = false;
         Gen.track_fail_pos = true;
       } args
